package com.elitescloud.cloudt.system.service;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.req.SysOrgPageQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgDetailRespDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgPageRespDTO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 组织相关通用接口.
 *
 * @author Kaiser（wang shao）
 * 2022/9/30
 */
public interface OrgQueryService {

    /**
     * 获取组织树
     *
     * @param tree     是否返回树形结构
     * @param employee 是否返回员工
     * @param all      是否包所有组织数据，默认仅返回所属租户组织的
     * @return 组织树
     */
    ApiResult<List<OrgTreeNodeRespVO>> getOrgTree(Boolean tree, Boolean employee, Boolean all);

    /**
     * 异步获取组织树
     *
     * @param pId      父节点ID
     * @param employee 是否返回员工
     * @param all      是否包所有组织数据，默认仅返回所属租户组织的
     * @return 组织列表
     */
    ApiResult<List<OrgTreeNodeRespVO>> getOrgTreeAsync(Long pId, Boolean employee, Boolean all);

    /**
     * 获取直属组织ID
     * <p>
     * 所有直属下级组织的ID
     *
     * @param id 指定组织的ID
     * @return 直属组织ID列表
     */
    ApiResult<List<Long>> querySubOrgIds(@NotNull Long id);

    /**
     * 获取下属组织ID
     * <p>
     * 所有下属下级组织的ID
     *
     * @param id 指定组织的ID
     * @return 下属组织ID列表
     */
    ApiResult<List<Long>> queryBelongOrgIds(@NotNull Long id);

    /**
     * 根据组织名称获取组织ID
     * <p>
     * 可能存在多个id
     *
     * @param orgName 组织名称
     * @return 组织ID
     */
    ApiResult<List<Long>> getIdsByOrgName(@NotBlank String orgName);

    /**
     * 根据组织编码获取组织ID
     *
     * @param orgCode 组织编码
     * @return 组织ID
     */
    ApiResult<Long> getIdByOrgCode(@NotBlank String orgCode);

    /**
     * 根据ID查询组织编码和名称
     *
     * @param ids 组织ID
     * @return 组织信息
     */
    ApiResult<List<IdCodeNameParam>> queryCodeAndName(@NotEmpty Set<Long> ids);

    /**
     * 根据ID查询组织编码和名称
     *
     * @param codes 组织编码
     * @return 组织信息
     */
    ApiResult<List<IdCodeNameParam>> queryIdAndName(@NotEmpty Set<String> codes);

    /**
     * 根据ID获取组织信息
     *
     * @param id 组织ID
     * @return 组织信息
     */
    ApiResult<SysOrgBasicDTO> getById(Long id);

    /**
     * 根据ID获取组织详细信息
     *
     * @param id             组织ID
     * @param withParentType 指定类型的上级组织，{@link com.elitescloud.cloudt.system.constant.OrgType}
     * @return 组织信息
     */
    ApiResult<SysOrgDetailRespDTO> getDetailById(@NotNull Long id, String withParentType);

    /**
     * 根据编码获取组织信息
     *
     * @param code 组织编码
     * @return 组织信息
     */
    ApiResult<SysOrgBasicDTO> getByCode(String code);

    /**
     * 根据编码获取组织详细信息
     *
     * @param code           组织编码
     * @param withParentType 指定类型的上级组织，{@link com.elitescloud.cloudt.system.constant.OrgType}
     * @return 组织信息
     */
    ApiResult<SysOrgDetailRespDTO> getDetailByCode(@NotBlank String code, String withParentType);


    /**
     * 根据编码获取上级组织信息
     *
     * @param code       组织编码
     * @param parentType 指定上级类型
     * @return 组织信息
     */
    ApiResult<SysOrgBasicDTO> getParentByCode(@NotBlank String code, String parentType);

    /**
     * 根据编码获取上级组织信息
     *
     * @param codes      组织编码
     * @param parentType 指定上级类型
     * @return 组织信息
     */
    ApiResult<Map<String, SysOrgBasicDTO>> getParentByCode(@NotEmpty Set<String> codes, String parentType);

    /**
     * 查询组织列表
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    ApiResult<List<SysOrgBasicDTO>> queryList(SysOrgQueryDTO queryDTO);

    /**
     * 分页查询组织
     *
     * @param queryVO 查询参数
     * @return 组织列表
     */
    ApiResult<PagingVO<OrgPagedRespVO>> page(CommonOrgPageQueryVO queryVO);

    /**
     * 分页查询组织
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    ApiResult<PagingVO<SysOrgPageRespDTO>> queryPage(SysOrgPageQueryDTO queryDTO);

    /**
     * 获取特定类型的上级组织
     *
     * @param id         组织ID
     * @param code       组织编码
     * @param parentType 特定上级类型
     * @return 公司列表
     */
    ApiResult<IdCodeNameParam> getParentOrgByType(Long id, String code, String parentType);
}
