package com.elitescloud.cloudt.system.service;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.provider.dto.SysApiPermissionMetadataDTO;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 权限查询服务接口.
 *
 * @author Kaiser（wang shao）
 * 2022/11/7
 */
public interface PermissionQueryService {

    /**
     * 获取角色的权限菜单
     *
     * @param roleId  角色ID
     * @param appCode 应用编码
     * @param tree    是否返回树形列表
     * @return 权限菜单
     */
    ApiResult<List<RolePermissionRespVO>> getPermissionMenuByRole(@NotNull Long roleId, String appCode, Boolean tree);

    /**
     * 根据权限编码获取权限信息
     *
     * @param code 权限编码
     * @return 权限信息
     */
    ApiResult<PermissionDetailRespVO> getPermissionDetailByCode(@NotBlank String code);

    /**
     * 获取应用下的api信息
     *
     * @param appCode 应用编码
     * @return api信息
     */
    ApiResult<List<SysApiPermissionMetadataDTO>> queryApiMetadata(@NotBlank String appCode);

    /**
     * 数据权限的数据范围
     *
     * @return 范围列表
     */
    ApiResult<List<CodeNameParam>> dataPermissionRanges();

    /**
     * 数据权限的条件规则
     *
     * @return 条件规则
     */
    ApiResult<List<CodeNameParam>> dataPermissionRuleCondition();

    /**
     * 规则取值方式
     *
     * @return 取值方式
     */
    ApiResult<List<CodeNameParam>> dataPermissionRuleValueMethod();

    /**
     * 获取菜单下的API
     *
     * @param menuCode 菜单编码
     * @return API列表
     */
    ApiResult<List<SysMenuApiRespVO>> getApi(@NotBlank String menuCode);

    /**
     * 获取接口绑定的菜单
     *
     * @param apiId   接口ID
     * @param appCode 应用编码
     * @return 菜单ID
     */
    ApiResult<List<Long>> getMenuIdOfApi(@NotNull Long apiId, String appCode);

    /**
     * 获取接口绑定的菜单
     *
     * @param apiId   接口ID
     * @param appCode 应用编码
     * @return 菜单列表
     */
    ApiResult<List<PermissionMenuRespVO>> getMenuDetailOfApi(@NotNull Long apiId, String appCode);

    /**
     * 获取角色分配的数据权限
     *
     * @param roleId   角色ID
     * @param menuCode 菜单编码
     * @param apiCode  API接口编码
     * @return 数据权限信息
     */
    ApiResult<SysDprRespVO> getApiDataPermission(@NotNull Long roleId, @NotBlank String menuCode, @NotBlank String apiCode);

    /**
     * 根据角色编码查询用户ID
     *
     * @param roleCode 角色编码
     * @return 用户ID
     */
    ApiResult<Set<Long>> getUserIdByRoleCode(@NotBlank String roleCode);

    /**
     * 查询所有角色信息
     *
     * @param tenantCode 租户编码，为空则获取所有当前租户
     * @return 角色列表
     */
    ApiResult<List<IdCodeNameParam>> listAllRoles(String tenantCode);

    /**
     * 查询指定租户组织的所有角色信息
     *
     * @param tenantOrgCode 租户组织编码，为空则获取当前租户组织
     * @return 角色列表
     */
    ApiResult<List<IdCodeNameParam>> listAllRolesByTenantOrg(String tenantOrgCode);

    /**
     * 根据角色查询用户信息
     *
     * @param roleCode 角色编码
     * @return 用户列表
     */
    ApiResult<List<SysUserBasicDTO>> listUserByRole(@NotBlank String roleCode);

    /**
     * 获取数据权限树
     * <p>
     * 返回挂载有API接口的菜单及其应用
     *
     * @param roleId  角色ID
     * @param appCode 应用编码，为空则自动获取所有
     * @param tree    是否返回树状，默认true
     * @return 权限数
     */
    ApiResult<List<DataPermissionTreeNodeRespVO>> dataPermissionTree(@NotNull Long roleId, String appCode, Boolean tree);

    /**
     * 获取当前用户的数据权限
     *
     * @return 数据权限规则
     */
    ApiResult<SysDprRoleApiRowColumnRuleDTO> getDataPermissionOfCurrentUser();
}
