package com.elitescloud.cloudt.system.service;

import com.elitescloud.boot.common.param.SysSendVerifyCodeVO;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.req.UserCreateDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.user.UserSaveVO;
import com.elitescloud.cloudt.system.param.SysPasswordUpdate;
import com.elitescloud.cloudt.system.service.param.SysAccountUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysPasswordUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysUserAvatarUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysVerifyCodeParam;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 用户管理接口.
 *
 * @author Kaiser（wang shao）
 * 2022/9/22
 */
public interface UserMngService {

    /**
     * 保存(修改)用户信息
     *
     * @param saveVO 用户信息
     * @return 保存结果
     */
    ApiResult<Long> save(UserSaveVO saveVO);

    /**
     * 保存用户
     *
     * @param basicDTO 用户信息
     * @return 用户ID
     */
    ApiResult<Long> save(UserCreateDTO basicDTO);

    /**
     * 切换启用禁用
     *
     * @param id 用户ID
     * @return 用户ID
     */
    ApiResult<Long> updateEnabled(Long id);

    /**
     * 修改启用状态
     *
     * @param id      用户ID
     * @param enabled 启用状态
     * @return
     */
    ApiResult<Long> updateEnabled(Long id, Boolean enabled);

    /**
     * 重置密码
     *
     * @param id 用户ID
     * @return 用户ID
     */
    ApiResult<Long> updatePassword(Long id);

    /**
     * 修改密码
     *
     * @param id  用户ID
     * @param pwd 密码
     * @return 用户ID
     */
    ApiResult<Long> updatePassword(Long id, String pwd);

    /**
     * 修改密码
     *
     * @param passwordUpdate 密码修改信息
     * @return 修改结果
     */
    ApiResult<Long> updatePassword(SysPasswordUpdate passwordUpdate);

    /**
     * 修改密码时发送验证码
     *
     * @param verifyCodeParam 验证码参数
     * @return 是否发送成功
     */
    ApiResult<String> sendVerifyCodeForUpdatePassword(SysSendVerifyCodeVO verifyCodeParam);

    /**
     * 根据验证码更新用户密码
     *
     * @param updateParam 密码信息
     * @return 用户ID
     */
    ApiResult<Long> updatePasswordByVerifyCode(SysPasswordUpdateParam updateParam);

    /**
     * 更新手机号
     *
     * @param id     用户ID
     * @param mobile 手机号
     * @return 用户ID
     */
    ApiResult<Long> updateMobile(@NotNull Long id, String mobile);

    /**
     * 更新邮箱
     *
     * @param id    用户ID
     * @param email 邮箱
     * @return 用户ID
     */
    ApiResult<Long> updateEmail(@NotNull Long id, String email);

    /**
     * 修改账号时发送验证码
     *
     * @param verifyCodeParam 验证码参数
     * @return 是否发送成功
     */
    ApiResult<String> sendVerifyCodeForUpdateAccount(SysSendVerifyCodeVO verifyCodeParam);

    /**
     * 修改账号时校验验证码是否正确
     *
     * @param codeParam 验证码参数
     * @return 校验标识
     */
    ApiResult<String> verifyCodeForUpdateAccount(SysVerifyCodeParam codeParam);

    /**
     * 根据验证码更新用户账号
     *
     * @param updateParam 账号信息
     * @return 用户ID
     */
    ApiResult<Long> updateAccountByVerifyCode(SysAccountUpdateParam updateParam);

    /**
     * 修改用户头像
     *
     * @param updateParam 头像信息
     * @return 用户ID
     */
    ApiResult<Long> updateAvatar(SysUserAvatarUpdateParam updateParam);

    /**
     * 更新统一身份认证ID
     *
     * @param ids ID
     * @return ID
     */
    ApiResult<Boolean> clearCasUserId(@NotEmpty List<Long> ids);

    /**
     * 更新账号的用户类型
     *
     * @param id         用户ID
     * @param userType   用户类型
     * @param identityId 身份ID
     * @return 更新结果
     */
    ApiResult<Boolean> saveUserType(@NotNull Long id, String userType, String identityId);

    /**
     * 删除账号的用户类型
     *
     * @param id         用户ID
     * @param userType   用户类型
     * @param identityId 身份ID
     * @return 更新结果
     */
    ApiResult<Boolean> deleteUserType(@NotNull Long id, String userType, String identityId);

    /**
     * 删除用户
     *
     * @param id 用户ID
     * @return 用户ID
     */
    ApiResult<Long> delete(Long id);

    /**
     * 获取用户信息
     *
     * @param id 用户ID
     * @return 用户信息
     */
    ApiResult<UserDetailRespVO> get(Long id);

    /**
     * 分页查询用户信息
     *
     * @param queryVO 查询参数
     * @return 用户列表
     */
    ApiResult<PagingVO<UserPageRespVO>> page(UserPageQueryVO queryVO);
}
