package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.service.impl.SystemDataServiceImpl;

import java.io.File;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public abstract class AbstractBasicDataInitProvider {
    private List<AbstractBasicDataInitProvider> children;

    /**
     * 类型名称
     *
     * @return 类型名称
     */
    public abstract String typeName();

    /**
     * 表名
     *
     * @return 表名称
     */
    public abstract String tableName();

    /**
     * 表字段
     *
     * @return 字段
     */
    public abstract List<String> fields();

    /**
     * 标题
     *
     * @return 字段对应的标题
     */
    public abstract Map<String, String> fieldTitles();

    /**
     * 是否是租户数据
     *
     * @return boolean
     */
    public boolean isTenantData() {
        return false;
    }

    /**
     * 排序字段
     *
     * @return 字段
     */
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("create_time"));
    }

    /**
     * 导出前准备
     */
    public void prepareExport(File workDir) {
    }

    /**
     * 导入前准备
     *
     * @param importData 导入数据
     * @param tenantId   租户ID
     */
    public void prepareForImport(SystemDataServiceImpl.ImportedData importData, Long tenantId) {
    }

    /**
     * 导入、导出后处理
     */
    public void afterExecute(Throwable e) {
    }

    /**
     * 导出数据转换
     *
     * @param data 查询的数据
     * @return 导出的数据
     */
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        return data;
    }

    /**
     * 导入数据转换
     *
     * @param data         导入的数据
     * @param currentData  目前库中所有数据
     * @param lastStepData 上一步的数据
     * @return 更新类型
     */
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData,
                                       List<Map<String, Object>> lastStepData) {
        return UpdateType.IGNORE;
    }

    /**
     * 导入后的处理
     * <p>
     * 当有多个provider相互依赖时可通过该方式控制顺序
     *
     * @return 导入后的导入
     */
    public final List<AbstractBasicDataInitProvider> thenImport() {
        return ObjectUtil.defaultIfNull(children, Collections.emptyList());
    }

    public final AbstractBasicDataInitProvider thenImport(AbstractBasicDataInitProvider... children) {
        if (ArrayUtil.isNotEmpty(children)) {
            this.children = Arrays.asList(children);
        }
        return this;
    }

    final void normalizeBooleanValue(Map<String, Object> data, String key) {
        var value = this.getBooleanValue(data.get(key), null);
        data.put(key, value);
    }

    /**
     * 转换boolean值
     *
     * @param value 待转换的值
     * @param def 默认值
     * @return boolean
     */
    final Boolean getBooleanValue(Object value, Boolean def) {
        if (value == null) {
            return def;
        }
        if (value instanceof Boolean) {
            return (Boolean) value;
        }
        if (value instanceof Integer) {
            return (Integer) value == 1;
        }

        if (value.toString().equalsIgnoreCase("true")) {
            return true;
        }
        if (value.toString().equalsIgnoreCase("false")) {
            return false;
        }
        return def;
    }

    final String getStringValue(Map<String, Object> map, String key, String def) {
        var value = map.get(key);
        if (value == null || CharSequenceUtil.isBlank(value.toString())) {
            return def;
        }

        return value.toString();
    }

    final void normalizeLongValue(Map<String, Object> map, String key) {
        map.put(key, getLongValue(map, key, null));
    }

    final Long getLongValue(Map<String, Object> map, String key, Long def) {
        var value = map.get(key);
        if (value == null) {
            return def;
        }
        if (CharSequenceUtil.isBlank(value.toString())) {
            return def;
        }
        return Long.parseLong(value.toString());
    }

    final void normalizeIntegerValue(Map<String, Object> map, String key) {
        map.put(key, getIntegerValue(map, key, null));
    }

    final Integer getIntegerValue(Map<String, Object> map, String key, Integer def) {
        var value = map.get(key);
        if (value == null) {
            return def;
        }
        if (CharSequenceUtil.isBlank(value.toString())) {
            return def;
        }
        return Integer.parseInt(value.toString());
    }

    public static enum UpdateType {
        /**
         * 新增
         */
        ADD,
        /**
         * 修改
         */
        UPDATE,
        /**
         * 忽略
         */
        IGNORE
    }
}
