package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.platform.model.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.platform.model.constant.PlatformMenusOuterLinkTypeEnum;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 应用菜单.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformMenusInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "菜单";
    }

    @Override
    public String tableName() {
        return "sys_platform_menus";
    }

    @Override
    public List<String> fields() {
        return List.of("menus_app_code", "menus_name", "menus_type", "node_type", "menus_code",
                "menus_order", "menus_state", "menus_parent_code", "menus_route", "menus_describe",
                "menus_icon", "display", "create_time", "modify_time", "outer_link", "outer_link_type");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("menus_app_code", "应用编码");
        titles.put("menus_name", "菜单名称");
        titles.put("menus_type", "菜单类型");
        titles.put("menus_type_name", "菜单类型名称");
        titles.put("node_type", "节点类型");
        titles.put("node_type_name", "节点类型名称");
        titles.put("menus_code", "菜单编码");
        titles.put("menus_order", "顺序");
        titles.put("menus_parent_code", "上级菜单编码");
        titles.put("menus_route", "菜单路由");
        titles.put("menus_describe", "描述");
        titles.put("menus_icon", "图标");
        titles.put("display", "是否显示");
        titles.put("outer_link", "是否外部链接");
        titles.put("outer_link_type", "外部链接类型");
        titles.put("outer_link_type_name", "外部链接类型名称");
        titles.put("menus_state", "状态");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("menus_parent_code"), OrderItem.asc("menus_order"));
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var state = getBooleanValue(data.get("menus_state"), null);
        if (state == null || !state) {
            return null;
        }

        // 转换类型
        var outerLinkType = PlatformMenusOuterLinkTypeEnum.parse(ObjectUtil.defaultIfNull(data.get("outer_link_type"), "").toString());
        if (outerLinkType != null) {
            data.put("outer_link_type_name", outerLinkType.getDescription());
        }
        var menuType = PlatformAppMenusTypeEnum.parse(ObjectUtil.defaultIfNull(data.get("menus_type"), "").toString());
        if (menuType != null) {
            data.put("menus_type_name", menuType.getDescription());
        }
        var nodeType = PlatformMenusNodeEnum.parse(ObjectUtil.defaultIfNull(data.get("node_type"), "").toString());
        if (nodeType != null) {
            data.put("node_type_name", nodeType.getDescription());
        }
        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var menus_code = this.getStringValue(data, "menus_code", null);
        Assert.notBlank(menus_code, "菜单编码为空");
        this.normalizeBooleanValue(data, "menus_state");
        this.normalizeBooleanValue(data, "outer_link");
        this.normalizeBooleanValue(data, "display");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var existsData = currentData.stream()
                .filter(t -> menus_code.equals(t.get("menus_code")))
                .findFirst()
                .orElse(null);
        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("menus_state", this.getBooleanValue(data.get("menus_state"), true));
        data.put("menus_order", this.getIntegerValue(data, "menus_order", 0));
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
        data.put("menus_state", ObjectUtil.defaultIfNull(dataExists.get("menus_state"), this.getBooleanValue(data.get("menus_state"), true)));
        data.put("menus_order", ObjectUtil.defaultIfNull(dataExists.get("menus_order"), this.getIntegerValue(data, "menus_order", 0)));
    }
}
