package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.constant.NextValuePeriod;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 下一编号.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformNextNumberInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "下一编号";
    }

    @Override
    public String tableName() {
        return "sys_platform_next_number";
    }

    @Override
    public List<String> fields() {
        return List.of("create_time", "app_code", "code", "name", "step",
                "next_number", "nn_period", "nn_time", "version", "enabled", "internal");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("code", "编码");
        titles.put("name", "名称");
        titles.put("step", "步长");
        titles.put("nn_period", "编号周期");
        titles.put("nn_period_name", "编号周期名称");
        titles.put("internal", "是否内置");
        titles.put("next_number", "下一编号");
        titles.put("enabled", "是否启用");
        titles.put("create_time", "创建时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("app_code"), OrderItem.desc("create_time"));
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        // 过滤启用的数据
        var enabled = getBooleanValue(data.get("enabled"), null);
        if (enabled == null || !enabled) {
            return null;
        }

        // 转换类型
        var type = NextValuePeriod.parse(ObjectUtil.defaultIfNull(data.get("nn_period"), "").toString());
        if (type != null) {
            data.put("nn_period_name", type.getDescription());
        }
        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var code = this.getStringValue(data, "code", null);
        Assert.notBlank(code, "编码为空");

        this.normalizeBooleanValue(data, "internal");
        this.normalizeBooleanValue(data, "enabled");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        var existsData = currentData.stream()
                .filter(t -> code.equals(t.get("code")))
                .findFirst()
                .orElse(null);
        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        return UpdateType.IGNORE;
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("enabled", this.getBooleanValue(data.get("enabled"), true));
    }
}
