package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * UDC值.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
public class PlatformUdcValueInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "UDC值";
    }

    @Override
    public String tableName() {
        return "sys_platform_udc_value";
    }

    @Override
    public List<String> fields() {
        return List.of("create_time", "modify_time", "app_code", "udc_code", "udc_value_code", "udc_value_name",
                "value_alias_name", "udc_order", "allow_start", "udc_value_describe", "allow_default", "parent_udc_value_code");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("app_code", "应用编码");
        titles.put("udc_code", "UDC编码");
        titles.put("udc_value_code", "值编码");
        titles.put("udc_value_name", "值名称");
        titles.put("value_alias_name", "别名");
        titles.put("udc_order", "顺序");
        titles.put("allow_start", "是否启用");
        titles.put("udc_value_describe", "描述");
        titles.put("allow_default", "是否内置");
        titles.put("parent_udc_value_code", "上级UDC值");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.asc("app_code"),
                OrderItem.asc("udc_code"), OrderItem.asc("udc_order"));
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var app_code = this.getStringValue(data, "app_code", null);
        Assert.notBlank(app_code, "应用编码为空");
        var udc_code = this.getStringValue(data, "udc_code", null);
        Assert.notBlank(udc_code, "UDC编码为空");
        var value_code = this.getStringValue(data, "udc_value_code", null);
        Assert.notBlank(value_code, "值编码为空");

        this.normalizeBooleanValue(data, "allow_start");
        this.normalizeBooleanValue(data, "allow_default");

        if (currentData.isEmpty()) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }

        var existsData = currentData.parallelStream()
                .filter(t -> app_code.equals(t.get("app_code")) && udc_code.equals(t.get("udc_code")) && value_code.equals(t.get("udc_value_code")))
                .findFirst()
                .orElse(null);
        if (existsData == null) {
            this.normalizeForAdd(data);
            return UpdateType.ADD;
        }
        this.normalizeForUpdate(data, existsData);
        return UpdateType.UPDATE;
    }

    private void normalizeForAdd(Map<String, Object> data) {
        data.put("udc_order", this.getIntegerValue(data, "udc_order", 0));
    }

    private void normalizeForUpdate(Map<String, Object> data, Map<String, Object> dataExists) {
        data.put("id", dataExists.get("id"));
        data.put("udc_order", ObjectUtil.defaultIfNull(dataExists.get("udc_order"), this.getIntegerValue(data, "udc_order", 0)));
        data.put("value_alias_name", CharSequenceUtil.blankToDefault(getStringValue(dataExists, "value_alias_name", ""),
                getStringValue(data, "value_alias_name", "")));
    }
}
