package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAreaRepoProc;
import com.elitescloud.cloudt.system.convert.AreaConvert;
import com.elitescloud.cloudt.system.dto.req.SysAreaNamesQueryDTO;
import com.elitescloud.cloudt.system.dto.req.SysAreaQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysAreaNamesRespDTO;
import com.elitescloud.cloudt.system.dto.resp.SysAreaRespDTO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonAreaTreeRespVO;
import com.elitescloud.cloudt.system.service.AreaQueryService;
import com.elitescloud.cloudt.system.service.repo.AreaRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantAreaRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class AreaQueryServiceImpl extends BaseServiceImpl implements AreaQueryService {
    private static final AreaConvert CONVERT = AreaConvert.INSTANCE;

    @Autowired
    private TenantAreaRepoProc tenantAreaRepoProc;
    @Autowired
    private SysPlatformAreaRepoProc platformAreaRepoProc;
    @Autowired
    private AreaRepoProc areaRepoProc;

    @Override
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaQueryVO queryVO) {
        List<CommonAreaTreeRespVO> dataList = this.areaTemplate(() -> areaRepoProc.queryList(queryVO));

        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<Map<String, String>> queryNamesByAreaCode(Collection<String> areaCodes) {
        var codes = areaCodes.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (codes.isEmpty()) {
            return ApiResult.ok(Collections.emptyMap());
        }

        // 查询名称
        Map<String, String> nameMap = this.areaTemplate(() -> areaRepoProc.getCodeAndName(codes));
        return ApiResult.ok(nameMap);
    }

    @Override
    public ApiResult<List<SysAreaRespDTO>> listByAreaCodes(Set<String> areaCodes) {
        var codes = areaCodes.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (codes.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }
        var areaList = this.areaTemplate(() -> areaRepoProc.listByCode(codes))
                .stream()
                .map(CONVERT::do2DTO)
                .collect(Collectors.toList());

        return ApiResult.ok(areaList);
    }

    @Override
    public ApiResult<List<SysAreaRespDTO>> listByParentAreaCode(String areaCode) {
        if (CharSequenceUtil.isBlank(areaCode)) {
            return ApiResult.ok(Collections.emptyList());
        }
        var areaList = this.areaTemplate(() -> areaRepoProc.listByParentAreaCode(areaCode))
                .stream()
                .map(CONVERT::do2DTO)
                .collect(Collectors.toList());

        return ApiResult.ok(areaList);
    }

    @Override
    public ApiResult<List<SysAreaRespDTO>> queryList(SysAreaQueryDTO queryDTO) {
        var areaList = this.areaTemplate(() -> areaRepoProc.queryList(queryDTO))
                .stream()
                .map(CONVERT::do2DTO)
                .collect(Collectors.toList());

        return ApiResult.ok(areaList);
    }

    @Override
    public ApiResult<Boolean> existsCode(String code) {
        Assert.notBlank(code, "编码为空");

        boolean exists = areaRepoProc.existsCode(code);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<List<SysAreaNamesRespDTO>> queryListByNames(SysAreaNamesQueryDTO queryDTO) {
        var codeOrNames = queryDTO.getCodeOrNames();
        var fuzzy = Boolean.TRUE.equals(queryDTO.getFuzzy());
        Assert.notEmpty(codeOrNames, "行政区域编码或名称为空");

        var dtoList = codeOrNames.stream().map(t -> {
            SysAreaNamesRespDTO respDTO = new SysAreaNamesRespDTO();
            respDTO.setMatched(false);
            respDTO.setCodes(new ArrayList<>(8));
            respDTO.setNames(new ArrayList<>(8));
            respDTO.setParam(t);
            return respDTO;
        }).collect(Collectors.toList());

        // 查询树形数据
        var treeDataList = this.queryTreeData(codeOrNames, fuzzy);
        if (treeDataList.isEmpty()) {
            return ApiResult.ok(dtoList);
        }

        // 查询行政区域编码
        dtoList.parallelStream().forEach(d -> {
            var treeData = treeDataList.parallelStream().filter(t -> {
                if (CollUtil.isNotEmpty(d.getCodes())) {
                    return false;
                }
                return isMatchTreeData(t, d.getParam(), 0, fuzzy);
            }).findAny().orElse(null);
            if (treeData != null) {
                this.fillTreeData(d, treeData, d.getParam(), 0, fuzzy);
            }
        });

        return ApiResult.ok(dtoList);
    }

    private boolean fillTreeData(SysAreaNamesRespDTO respDTO, TreeData treeData, List<String> params, int paramIndex, boolean fuzzy) {
        var area = treeData.getAreaDO();
        if (paramIndex > params.size()) {
            return false;
        }

        var param = params.get(paramIndex);
        if (CharSequenceUtil.isBlank(param)) {
            return false;
        }

        if (isMatchTreeData(area.getAreaName(), param, fuzzy) || isMatchTreeData(area.getAreaCode(), param, fuzzy)) {
            // 匹配上
            respDTO.setMatched(true);
            respDTO.getCodes().add(area.getAreaCode());
            respDTO.getNames().add(area.getAreaName());

            var nextIndex = paramIndex + 1;
            if (params.size() == nextIndex) {
                // 已匹配完
                return true;
            }
            if (CollUtil.isNotEmpty(treeData.getChildren())) {
                for (TreeData child : treeData.getChildren()) {
                    if (this.fillTreeData(respDTO, child, params, nextIndex, fuzzy)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private boolean isMatchTreeData(TreeData treeData, List<String> params, int paramIndex, boolean fuzzy) {
        var area = treeData.getAreaDO();
        if (paramIndex > params.size()) {
            return false;
        }

        var param = params.get(paramIndex);
        if (CharSequenceUtil.isBlank(param)) {
            return false;
        }

        if (isMatchTreeData(area.getAreaName(), param, fuzzy) || isMatchTreeData(area.getAreaCode(), param, fuzzy)) {
            // 匹配上
            var nextIndex = paramIndex + 1;
            if (params.size() == nextIndex) {
                // 已匹配完
                return true;
            }
            if (CollUtil.isNotEmpty(treeData.getChildren())) {
                for (TreeData child : treeData.getChildren()) {
                    if (this.isMatchTreeData(child, params, nextIndex, fuzzy)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    private boolean isMatchTreeData(String matcher, String target, boolean fuzzy) {
        return fuzzy ? matcher.startsWith(target) : matcher.equals(target);
    }

    @SuppressWarnings("unchecked")
    private List<TreeData> queryTreeData(List<List<String>> codeOrNames, boolean fuzzy) {
        var paramList = codeOrNames.stream().flatMap(Collection::stream).filter(StringUtils::hasText).collect(Collectors.toSet());
        if (paramList.isEmpty()) {
            return Collections.emptyList();
        }

        List<SysPlatformAreaDO> areaDoList = this.areaTemplate(() -> {
            if (paramList.size() <= 50) {
                return fuzzy ? areaRepoProc.listByCodeOrNameFuzzily(paramList) : areaRepoProc.listByCodeOrName(paramList);
            }
            return areaRepoProc.all();
        });
        if (areaDoList.isEmpty()) {
            return Collections.emptyList();
        }

        // 转为树形
        var areaList = areaDoList.stream().map(TreeData::new).collect(Collectors.toList());
        return (List<TreeData>) new TreeDataUtil<>(areaList, TreeData::getCode, TreeData::getParentCode, TreeData::setChildren)
                .getRoots();
    }

    private <T> T areaTemplate(Supplier<T> supplier) {
        var enabled = this.enabledPlatform();
        return enabled ? super.tenantDataIsolateProvider.byDefaultDirectly(supplier) : supplier.get();
    }

    private boolean enabledPlatform() {
        var tenantId = super.currentTenantId();
        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        return enabled == null || enabled;
    }

    static class TreeData {
        private final SysPlatformAreaDO areaDO;
        private List<TreeData> children;

        public TreeData(SysPlatformAreaDO areaDO) {
            this.areaDO = areaDO;
        }

        public SysPlatformAreaDO getAreaDO() {
            return areaDO;
        }

        public List<TreeData> getChildren() {
            return children;
        }

        public void setChildren(List<TreeData> children) {
            this.children = children;
        }

        public String getCode() {
            return areaDO.getAreaCode();
        }

        public String getParentCode() {
            return areaDO.getParentAreaCode();
        }
    }
}
