package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.convert.SysPlatformCurrencyConvert;
import com.elitescloud.cloudt.platform.model.vo.extend.query.CurrencyPageQueryVO;
import com.elitescloud.cloudt.platform.model.vo.extend.resp.CurrencyPageRespVO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformCurrencyRepoProc;
import com.elitescloud.cloudt.system.service.CurrencyMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantCurrencyDO;
import com.elitescloud.cloudt.system.service.repo.CurrencyRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantCurrencyRepo;
import com.elitescloud.cloudt.system.service.repo.TenantCurrencyRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class CurrencyMngServiceImpl extends BaseServiceImpl implements CurrencyMngService {
    private static final SysPlatformCurrencyConvert CONVERT = SysPlatformCurrencyConvert.INSTANCE;

    @Autowired
    private TenantCurrencyRepo tenantCurrencyRepo;
    @Autowired
    private TenantCurrencyRepoProc tenantCurrencyRepoProc;
    @Autowired
    private CurrencyRepoProc currencyRepoProc;
    @Autowired
    private SysPlatformCurrencyRepoProc platformCurrencyRepoProc;

    @Override
    public ApiResult<PagingVO<CurrencyPageRespVO>> page(CurrencyPageQueryVO queryVO) {
        // 查询禁用的，则获取租户禁用的
        if (Boolean.FALSE.equals(queryVO.getEnabled())) {
            var tenantId = super.currentTenantId();
            var currCodes = tenantCurrencyRepoProc.queryCurrCodesOfDisabled(tenantId);
            var result = tenantDataIsolateProvider.byDefaultDirectly(() -> currencyRepoProc.pageMng(queryVO, currCodes))
                    .map(CONVERT::do2PageRespVO);
            return ApiResult.ok(result);
        }

        var result = tenantDataIsolateProvider.byDefaultDirectly(() -> currencyRepoProc.pageMng(queryVO, null))
                .map(CONVERT::do2PageRespVO);
        // 查询禁用的
        if (result.isNotEmpty()) {
            var tenantId = super.currentTenantId();
            var currCodes = tenantCurrencyRepoProc.queryCurrCodesOfDisabled(tenantId);
            result.each(t -> {
                if (currCodes.contains(t.getCurrCode())) {
                    t.setEnabled(false);
                }
            });
        }
        return ApiResult.ok(result);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> execSwitchEnabled(Long id) {
        // 查询货币编码
        String currCode = tenantDataIsolateProvider.byDefaultDirectly(() -> platformCurrencyRepoProc.getCurrCode(id));
        if (CharSequenceUtil.isBlank(currCode)) {
            return ApiResult.fail("货币不存在");
        }

        // 判断当前状态
        var tenantId = super.currentTenantId();
        var recordId = tenantCurrencyRepoProc.getId(tenantId, currCode);
        if (recordId != null) {
            // 已存在，则删掉
            tenantCurrencyRepoProc.delete(recordId);
            return ApiResult.ok(id);
        }

        // 新增禁用
        SysTenantCurrencyDO tenantCurrencyDO = new SysTenantCurrencyDO();
        tenantCurrencyDO.setSysTenantId(tenantId);
        tenantCurrencyDO.setCurrCode(currCode);
        tenantCurrencyDO.setEnabled(false);
        tenantCurrencyRepo.save(tenantCurrencyDO);
        return ApiResult.ok(id);
    }
}
