package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformCurrencyDO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformCurrencyRepoProc;
import com.elitescloud.cloudt.system.convert.CurrencyConvert;
import com.elitescloud.cloudt.system.dto.req.SysCurrencyQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysCurrencyRespDTO;
import com.elitescloud.cloudt.system.service.CurrencyQueryService;
import com.elitescloud.cloudt.system.service.repo.CurrencyRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantCurrencyRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class CurrencyQueryServiceImpl extends BaseServiceImpl implements CurrencyQueryService {
    private static final CurrencyConvert CONVERT = CurrencyConvert.INSTANCE;

    @Autowired
    private SysPlatformCurrencyRepoProc platformCurrencyRepoProc;
    @Autowired
    private CurrencyRepoProc repoProc;
    @Autowired
    private TenantCurrencyRepoProc tenantCurrencyRepoProc;

    @Override
    public ApiResult<List<CodeNameParam>> listCurrency() {
        // 先查出所有启用的
        var dataList = this.currencyTemplate(() -> platformCurrencyRepoProc.listAll());
        if (dataList.isEmpty()) {
            return ApiResult.ok(dataList);
        }

        // 过滤掉出租户禁用的
        var codesDisabled = this.disabledOfTenant();
        if (!codesDisabled.isEmpty()) {
            dataList = dataList.stream().filter(t -> !codesDisabled.contains(t.getCode())).collect(Collectors.toList());
        }
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<SysCurrencyRespDTO> getByCode(String currCode) {
        if (CharSequenceUtil.isBlank(currCode)) {
            return ApiResult.fail("编码为空");
        }
        // 查询货币
        var currencyDo = this.currencyTemplate(() -> repoProc.getByCode(currCode));
        if (currencyDo == null) {
            return ApiResult.noData();
        }
        var dto = CONVERT.vo2Dto(currencyDo);
        if (Boolean.TRUE.equals(dto.getEnabled())) {
            // 判断租户是否禁用
            var disabledCodes = this.disabledOfTenant();
            dto.setEnabled(!disabledCodes.contains(currencyDo.getCurrCode()));
        }

        return ApiResult.ok(dto);
    }

    @Override
    public ApiResult<List<SysCurrencyRespDTO>> listByCodes(Set<String> currCodes) {
        var codes = currCodes == null ? null : currCodes.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (CollUtil.isEmpty(codes)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var currencyDoList = this.currencyTemplate(() -> repoProc.getByCodes(codes));
        // 转dto
        var dtoList = this.do2Dto(currencyDoList);

        return ApiResult.ok(dtoList);
    }

    @Override
    public ApiResult<List<SysCurrencyRespDTO>> queryList(SysCurrencyQueryDTO queryDTO) {
        var currencyDoList = this.currencyTemplate(() -> repoProc.queryList(queryDTO));
        // 转dto
        var dtoList = this.do2Dto(currencyDoList);

        return ApiResult.ok(dtoList);
    }

    private <T> T currencyTemplate(Supplier<T> supplier) {
        return tenantDataIsolateProvider.byDefaultDirectly(supplier);
    }

    private List<String> disabledOfTenant() {
        var tenantId = super.currentTenantId();
        return tenantCurrencyRepoProc.queryCurrCodesOfDisabled(tenantId);
    }

    private List<SysCurrencyRespDTO> do2Dto(List<SysPlatformCurrencyDO> currencyDoList) {
        if (CollUtil.isEmpty(currencyDoList)) {
            return Collections.emptyList();
        }

        // 查询禁用的
        var disabledCodes = this.disabledOfTenant();
        // 转dto
        return currencyDoList.stream().map(t -> {
            var dto = CONVERT.vo2Dto(t);
            dto.setEnabled(Boolean.TRUE.equals(t.getEnabled()) && !disabledCodes.contains(t.getCurrCode()));
            return dto;
        }).collect(Collectors.toList());
    }
}
