package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.CurrencyRateCalcMethod;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.convert.CurrencyRateConvert;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRatePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.CurrencyRateDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.CurrencyRatePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.CurrencyRateSaveVO;
import com.elitescloud.cloudt.system.service.CurrencyRateMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysCurrencyRateDO;
import com.elitescloud.cloudt.system.service.repo.CurrencyRateRepoProc;
import com.elitescloud.cloudt.system.service.repo.CurrencyRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class CurrencyRateMngServiceImpl extends BaseServiceImpl implements CurrencyRateMngService {
    private static final CurrencyRateConvert CONVERT = CurrencyRateConvert.INSTANCE;

    @Autowired
    private CurrencyRateRepoProc currencyRateRepoProc;
    @Autowired
    private CurrencyRepoProc currencyRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(CurrencyRateSaveVO saveVO) {
        // 保存参数转换与校验
        SysCurrencyRateDO rateDO;
        try {
            rateDO = saveVO.getId() == null ? this.convertForInsert(saveVO) : this.convertForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存数据
        currencyRateRepoProc.save(rateDO);

        return ApiResult.ok(rateDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        currencyRateRepoProc.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        // 获取当前启用状态
        Boolean enabled = currencyRateRepoProc.getEnabled(id);

        // 修改启用状态
        enabled = enabled == null || !enabled;
        currencyRateRepoProc.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<PagingVO<CurrencyRatePageRespVO>> page(CurrencyRatePageQueryVO queryVO) {
        var pageData = currencyRateRepoProc.pageMng(queryVO)
                .map(CONVERT::do2PageRespVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 名称设置
        var currCodes = pageData.stream()
                .flatMap(t -> Stream.of(t.getFromCurr(), t.getToCurr()))
                .collect(Collectors.toSet());
        if (!currCodes.isEmpty()) {
            var nameMap = this.queryCurrName(currCodes);
            pageData.each(t -> {
                t.setFromCurrName(nameMap.get(t.getFromCurr()));
                t.setToCurrName(nameMap.get(t.getToCurr()));
                t.setCalMethodName(this.convertCalcMethodName(t.getCalMethod()));
            });
        }

        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<CurrencyRateDetailRespVO> get(Long id) {
        return currencyRateRepoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);

                    // 获取货币名称
                    var nameMap = this.queryCurrName(List.of(t.getFromCurr(), t.getToCurr()));
                    respVO.setFromCurrName(nameMap.get(t.getFromCurr()));
                    respVO.setToCurrName(nameMap.get(t.getToCurr()));
                    respVO.setCalMethodName(this.convertCalcMethodName(t.getCalMethod()));

                    return respVO;
                }).map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    private SysCurrencyRateDO convertForInsert(CurrencyRateSaveVO saveVO) {
        // 判断是否已存在
        var existsId = currencyRateRepoProc.getIdByFromAndTo(saveVO.getFromCurr(), saveVO.getToCurr());
        Assert.isNull(existsId, "已存在" + saveVO.getFromCurrName() + "至" + saveVO.getToCurrName() + "的转换配置");

        Assert.isTrue(saveVO.getRatio().compareTo(BigDecimal.ZERO) > 0, "汇率不可小于0");

        var calcMethod = CurrencyRateCalcMethod.parse(saveVO.getCalMethod());
        Assert.notNull(calcMethod, "未知算法");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        return CONVERT.saveVo2Do(saveVO);
    }

    private SysCurrencyRateDO convertForUpdate(CurrencyRateSaveVO saveVO) {
        var rateDO = currencyRateRepoProc.get(saveVO.getId());
        Assert.notNull(rateDO, "修改的数据不存在");

        // 如果转换有变化，则判断是否已存在
        if (!rateDO.getFromCurr().equals(saveVO.getFromCurr()) ||
                !rateDO.getToCurr().equals(saveVO.getToCurr())) {
            var existsId = currencyRateRepoProc.getIdByFromAndTo(saveVO.getFromCurr(), saveVO.getToCurr());
            Assert.isNull(existsId, "已存在" + saveVO.getFromCurrName() + "至" + saveVO.getToCurrName() + "的转换配置");
        }

        Assert.isTrue(saveVO.getRatio().compareTo(BigDecimal.ZERO) > 0, "汇率不可小于0");

        var calcMethod = CurrencyRateCalcMethod.parse(saveVO.getCalMethod());
        Assert.notNull(calcMethod, "未知算法");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        CONVERT.copySaveVo2Do(saveVO, rateDO);
        return rateDO;
    }

    private Map<String, String> queryCurrName(Collection<String> currCodes) {
        if (CollUtil.isEmpty(currCodes)) {
            return Collections.emptyMap();
        }
        return super.tenantDataIsolateProvider.byDefaultDirectly(() -> currencyRepoProc.getNameByCurrCode(currCodes));
    }

    private String convertCalcMethodName(String calcMethod) {
        var method = CurrencyRateCalcMethod.parse(calcMethod);
        return method == null ? null : method.getDescription();
    }
}
