package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysEmpGroupType;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.EmpGroupConvert;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupEmpPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupEmpPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmpGroupSaveVO;
import com.elitescloud.cloudt.system.service.EmpGroupMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysEmpGroupDO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmpGroupEmpDO;
import com.elitescloud.cloudt.system.service.repo.EmpGroupEmpRepoProc;
import com.elitescloud.cloudt.system.service.repo.EmpGroupRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/2
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class EmpGroupMngServiceImpl extends BaseServiceImpl implements EmpGroupMngService {
    private static final EmpGroupConvert CONVERT = EmpGroupConvert.INSTANCE;

    @Autowired
    private EmpGroupRepoProc repoProc;
    @Autowired
    private EmpGroupEmpRepoProc empGroupEmpRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(EmpGroupSaveVO saveVO) {
        // 保存参数转换与校验
        SysEmpGroupDO groupDO;
        try {
            groupDO = saveVO.getId() == null ? this.convertForInsert(saveVO) : this.convertForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存数据
        repoProc.save(groupDO);

        return ApiResult.ok(groupDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        // 删除员工组
        repoProc.delete(id);
        // 删除关联
        empGroupEmpRepoProc.deleteEmployeeByGroupId(id);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        // 获取当前启用状态
        Boolean enabled = repoProc.getEnabled(id);

        // 修改启用状态
        enabled = enabled == null || !enabled;
        repoProc.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> addEmployee(Long id, Long employeeId) {
        var empIdExists = empGroupEmpRepoProc.getEmployeeIdOfGroup(id);
        if (empIdExists.contains(employeeId)) {
            // 已存在，则不再添加
            return ApiResult.ok(id);
        }

        // 新增
        SysEmpGroupEmpDO empGroupEmpDO = new SysEmpGroupEmpDO();
        empGroupEmpDO.setGroupId(id);
        empGroupEmpDO.setEmpId(employeeId);
        empGroupEmpRepoProc.save(empGroupEmpDO);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteEmployee(Long id, Long employeeId) {
        empGroupEmpRepoProc.deleteEmployeeByGroupId(id, employeeId);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<PagingVO<EmpGroupPageRespVO>> page(EmpGroupPageQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(CONVERT::do2PageRespVO));
        }

        // 员工组类型
        var typeNames = super.udcMap(new SysEmpGroupType());
        var result = pageData.map(t -> {
            var vo = CONVERT.do2PageRespVO(t);
            vo.setTypeName(typeNames.get(t.getType()));
            return vo;
        });
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<EmpGroupDetailRespVO> get(Long id) {
        return repoProc.getOptional(id)
                .map(t -> {
                    var vo = CONVERT.do2DetailRespVO(t);
                    vo.setTypeName(super.udcValue(new SysEmpGroupType(t.getType())));
                    return vo;
                }).map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    @Override
    public ApiResult<PagingVO<EmpGroupEmpPageRespVO>> pageEmp(EmpGroupEmpPageQueryVO queryVO) {
        var pageData = empGroupEmpRepoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }
        // 查询员工账号
        var userIds = pageData.stream().map(EmpGroupEmpPageRespVO::getUserId).filter(Objects::nonNull).collect(Collectors.toSet());
        var usernameMap = super.tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.getUsernameAndId(userIds));
        pageData.each(t -> t.setUsername(usernameMap.get(t.getUserId())));

        return ApiResult.ok(pageData);
    }

    private SysEmpGroupDO convertForInsert(EmpGroupSaveVO saveVO) {
        var exists = repoProc.existsCode(saveVO.getCode(), null);
        Assert.isTrue(!exists, "编号已存在");

        if (saveVO.getSortNo() == null) {
            saveVO.setSortNo(1);
        }
        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        return CONVERT.saveVo2DO(saveVO);
    }

    private SysEmpGroupDO convertForUpdate(EmpGroupSaveVO saveVO) {
        var groupDO = repoProc.get(saveVO.getId());
        Assert.notNull(groupDO, "修改的数据不存在");

        if (!groupDO.getCode().equals(saveVO.getCode())) {
            var exists = repoProc.existsCode(saveVO.getCode(), null);
            Assert.isTrue(!exists, "编号已存在");
        }

        if (saveVO.getSortNo() == null) {
            saveVO.setSortNo(1);
        }
        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        return CONVERT.copySaveVo2DO(saveVO, groupDO);
    }
}
