package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserCreateDTO;
import com.elitescloud.cloudt.system.model.entity.SysUserTerminalDO;
import com.elitescloud.cloudt.system.service.IUserService;
import com.elitescloud.cloudt.system.service.UserMngService;
import com.elitescloud.cloudt.system.service.manager.UserQueryManager;
import com.elitescloud.cloudt.system.service.model.entity.SysUserTypeDO;
import com.elitescloud.cloudt.system.service.repo.SysUserTerminalRepo;
import com.elitescloud.cloudt.system.service.repo.SysUserTerminalRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserTypeRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 内部服务.
 *
 * @author Kaiser（wang shao）
 * 2022/10/27
 */
@Service
@Slf4j
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class IUserServiceImpl extends BaseServiceImpl implements IUserService {
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private UserTypeRepoProc userTypeRepoProc;
    @Autowired
    private SysUserTerminalRepo userTerminalRepo;
    @Autowired
    private SysUserTerminalRepoProc userTerminalRepoProc;

    @Autowired
    private UserQueryManager userQueryManager;

    @Autowired
    private UserMngService userMngService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long insertUser(UserCreateDTO createDTO) {
        var saveResult = userMngService.save(createDTO);
        if (!saveResult.isSuccess()) {
            throw new BusinessException(saveResult.getMsg());
        }
        return saveResult.getData();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long addUserType(Long id, Set<String> userTypes) {
        Assert.notNull(id, "用户ID为空");
        Assert.notEmpty(userTypes, "新增用户类型为空");

        // 获取用户已有类型
        var tenantId = super.currentTenantId();
        var existsTypes = userTypeRepoProc.getUserType(id, tenantId);
        // 过滤出需要新增的
        var typesToAdd = userTypes.stream()
                .filter(t -> !existsTypes.contains(t))
                .map(t -> {
                    SysUserTypeDO typeDO = new SysUserTypeDO();
                    typeDO.setUserId(id);
                    typeDO.setType(t);
                    typeDO.setSysTenantId(tenantId);
                    return typeDO;
                }).collect(Collectors.toList());
        if (!typesToAdd.isEmpty()) {
            userTypeRepoProc.save(typesToAdd);
        }
        return id;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long removeUserType(Long id, Set<String> userTypes) {
        Assert.notNull(id, "用户ID为空");
        Assert.notEmpty(userTypes, "用户类型为空");

        userTypeRepoProc.delete(id, userTypes, super.currentTenantId());

        return id;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long addUserTerminal(Long id, Set<String> terminals) {
        Assert.notNull(id, "用户ID为空");
        Assert.notEmpty(terminals, "新增用户终端为空");

        // 获取用户已有终端
        var existsTerminals = userTerminalRepoProc.getTerminalStrByUserId(id);
        // 过滤出需要新增的
        LocalDateTime timeNow = LocalDateTime.now();
        var terminalsToAdd = terminals.stream()
                .filter(t -> {
                    if (existsTerminals.contains(t)) {
                        // 已存在
                        return false;
                    }
                    return Terminal.parse(t) != null;
                })
                .map(t -> {
                    SysUserTerminalDO terminalDO = new SysUserTerminalDO();
                    terminalDO.setUserId(id);
                    terminalDO.setTerminalCode(t);
                    terminalDO.setTimeBind(timeNow);

                    return terminalDO;
                }).collect(Collectors.toList());
        if (!terminalsToAdd.isEmpty()) {
            userTerminalRepo.saveAll(terminalsToAdd);
        }
        return id;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long removeUserTerminal(Long id, Set<String> terminals) {
        Assert.notNull(id, "用户ID为空");
        Assert.notEmpty(terminals, "用户终端为空");

        userTerminalRepoProc.delete(id, terminals);

        return id;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateCasSyncResult(Map<String, Long> userMap) {
        for (Map.Entry<String, Long> entry : userMap.entrySet()) {
            if (StringUtils.hasText(entry.getKey())) {
                userRepoProc.updateCasSyncResult(entry.getKey(), entry.getValue());
            }
        }
        return true;
    }

    @Override
    public boolean existsUsername(String username, Long id) {
        Assert.hasText(username, "账号为空");

        return userRepoProc.existsUsername(username, id);
    }

    @Override
    public boolean existsMobile(String mobile, Long id) {
        Assert.hasText(mobile, "手机号为空");

        return userRepoProc.existsMobile(mobile, id);
    }

    @Override
    public boolean existsEmail(String email, Long id) {
        Assert.hasText(email, "邮箱为空");

        return userRepoProc.existsMobile(email, id);
    }

    @Override
    public Long getUniqueIdByUsername(String username) {
        Assert.hasText(username, "账号为空");

        return userQueryManager.getUniqueIdByUsername(username);
    }

    @Override
    public Long getUniqueIdByMobile(String mobile) {
        Assert.hasText(mobile, "手机号为空");

        return userQueryManager.getUniqueIdByMobile(mobile);
    }

    @Override
    public Long getUniqueIdByEmail(String email) {
        Assert.hasText(email, "邮箱为空");

        return userQueryManager.getUniqueIdByEmail(email);
    }

    @Override
    public SysUserBasicDTO getById(Long id) {
        Assert.notNull(id, "ID为空");

        return userRepoProc.getBasicDto(id);
    }
}
