package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.convert.MenuFavorConvert;
import com.elitescloud.cloudt.system.model.vo.query.menu.MenuFavorPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuFavorRespVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuFavorSaveVO;
import com.elitescloud.cloudt.system.service.MenuFavorMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysMenuFavorDO;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/31
 */
@Service
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class MenuFavorMngServiceImpl extends BaseServiceImpl implements MenuFavorMngService {
    private static final MenuFavorConvert CONVERT = MenuFavorConvert.INSTANCE;

    @Autowired
    private MenuFavorRepo menuFavorRepo;
    @Autowired
    private MenuFavorRepoProc menuFavorRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private MenuRepoProc menuRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> upsert(MenuFavorSaveVO saveVO) {
        // 转换为收藏菜单对象
        var currentUser = super.currentUser(true);
        boolean isAdd = saveVO.getId() == null;
        SysMenuFavorDO menuFavorDO = null;
        if (isAdd) {
            menuFavorDO = CONVERT.saveVo2Do(saveVO);
            menuFavorDO.setUserId(currentUser.getUserId());
        } else {
            menuFavorDO = menuFavorRepoProc.get(saveVO.getId());
            if (menuFavorDO == null) {
                return ApiResult.noData();
            }
            CONVERT.copySaveVo(saveVO, menuFavorDO);
        }

        // 设置默认值
        menuFavorDO.setSortNo(ObjectUtil.defaultIfNull(menuFavorDO.getSortNo(), 0));
        menuFavorDO.setEnabled(ObjectUtil.defaultIfNull(menuFavorDO.getEnabled(), true));

        // 保存信息
        menuFavorRepo.save(menuFavorDO);

        return ApiResult.ok(menuFavorDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        Assert.notNull(id, "ID为空");
        menuFavorRepoProc.delete(id);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> switchEnabled(Long id) {
        Boolean enabled = menuFavorRepoProc.getEnabled(id);
        boolean value = enabled == null || !enabled;
        menuFavorRepoProc.updateEnabled(value, id);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<MenuFavorRespVO> get(Long id) {
        Assert.notNull(id, "ID为空");
        var respVoOptional = menuFavorRepoProc.getOptional(id).map(CONVERT::do2DetailRespVo);
        if (respVoOptional.isEmpty()) {
            return ApiResult.noData();
        }
        var respVO = respVoOptional.get();

        // 判断是否自定义了菜单树，如果自定义了，则优先查询自定义的菜单名称
        var tenantId = super.currentTenantId();
        boolean custom = ObjectUtil.defaultIfNull(tenantMenuRepoProc.getEnabled(tenantId), false);
        if (custom) {
            var menu = tenantMenuTreeRepoProc.getByMenuCode(respVO.getMenuCode(), tenantId);
            Assert.notNull(menu, "菜单不存在");
            if (StringUtils.hasText(menu.getMenuName())) {
                respVO.setMenuName(menu.getMenuName());
                return ApiResult.ok(respVO);
            }
        }

        // 查询默认菜单名称
        var platformMenu = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.getByMenuCode(respVO.getMenuCode()));
        Assert.notNull(platformMenu, "菜单已不存在");
        respVO.setMenuName(platformMenu.getMenusName());

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<MenuFavorRespVO>> pageMng(MenuFavorPageQueryVO queryVO) {
        // 判断是否自定义了菜单树，如果自定义了，则优先查询自定义的菜单名称
        var currentUser = super.currentUser(true);
        var tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        boolean custom = ObjectUtil.defaultIfNull(tenantMenuRepoProc.getEnabled(tenantId), false);

        // 先查询菜单编码
        List<String> menuCodesFilter = null;
        if (StringUtils.hasText(queryVO.getMenuName())) {
            if (custom) {
                // 自定义了
                menuCodesFilter = tenantMenuTreeRepoProc.queryMenuCodeByMenuNameLike(queryVO.getMenuName(), tenantId);
            } else {
                menuCodesFilter = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenuCodeByMenuNameLike(queryVO.getMenuName()));
            }
            if (menuCodesFilter.isEmpty()) {
                return ApiResult.ok(PagingVO.empty());
            }
        }

        // 分页查询菜单
        var pageData = menuFavorRepoProc.pageMng(currentUser.getUserId(), queryVO, menuCodesFilter);
        if (pageData.isEmpty()) {
            return ApiResult.ok(PagingVO.<MenuFavorRespVO>builder().total(pageData.getTotal()).build());
        }
        // 查询菜单名称
        var menuCodes = pageData.getRecords().stream().map(SysMenuFavorDO::getMenuCode).collect(Collectors.toSet());
        var originalMenuNameMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.getMenuNameByMenuCode(menuCodes));
        var customMenuNameMap = custom ? tenantMenuTreeRepoProc.getMenuNameByMenuCode(menuCodes) : new HashMap<String, String>(0);

        var pageVo = pageData.map(t -> {
            t.setMenuName(customMenuNameMap.getOrDefault(t.getMenuCode(), originalMenuNameMap.get(t.getMenuCode())));
            return CONVERT.do2DetailRespVo(t);
        });


        return ApiResult.ok(pageVo);
    }
}
