package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.param.CodeNameParam;
import com.elitescloud.cloudt.common.constant.TenantConstant;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.convert.TenantMenuTreeConvert;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuTreeCustomRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.menu.MenuTreeRespVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuTreeSaveVO;
import com.elitescloud.cloudt.system.service.MenuMngService;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuTreeDO;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class MenuMngServiceImpl extends BaseServiceImpl implements MenuMngService {
    private static final TenantMenuTreeConvert CONVERT = TenantMenuTreeConvert.INSTANCE;

    @Autowired
    private TenantMenuRepo tenantMenuRepo;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepo tenantMenuTreeRepo;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private MenuRepoProc menusRepoProc;
    @Autowired
    private TenantAppRepoProc tenantAppRepoProc;

    @Autowired
    private PlatformAppProvider appProvider;


    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveTree(List<MenuTreeSaveVO> saveVOList) {
        // 组织菜单树
        var tenantId = currentTenantId();
        var menuTreeDoList = convertAndCheck(tenantId, saveVOList);

        // 先清除原有数据
        tenantMenuTreeRepoProc.deleteByTenantId(tenantId);

        // 再保存现有数据
        tenantMenuTreeRepo.saveAll(menuTreeDoList);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateEnabled() {
        // 获取现有配置
        var tenantId = currentTenantId();
        var tenantMenuDO = tenantMenuRepoProc.getByTenantId(tenantId);

        if (tenantMenuDO == null) {
            // 如果没有，则新建
            tenantMenuDO = new SysTenantMenuDO();
            tenantMenuDO.setSysTenantId(tenantId);
            tenantMenuDO.setEnabled(true);
            tenantMenuRepo.save(tenantMenuDO);
            return ApiResult.ok(true);
        }

        // 有则更新
        boolean enabled = !tenantMenuDO.getEnabled();
        tenantMenuRepoProc.updateEnabled(tenantMenuDO.getId(), enabled);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<List<MenuTreeRespVO>> getTreeDefault(Boolean tree) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        // 先获取应用
        Set<String> appCodes = currentUser.getTenant() == null ? null :
                tenantDataIsolateProvider.byDefaultDirectly(() -> tenantAppRepoProc.getAppCode(currentUser.getTenantId()));
        if (!currentUser.isOperation() && CollectionUtils.isEmpty(appCodes)) {
            // 租户尚未分配应用
            return ApiResult.ok(Collections.emptyList());
        }
        List<MenuTreeRespVO> respVOList = convertMenuTreeRespVO(appCodes);
        if (respVOList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 获取菜单
        List<MenuTreeRespVO> finalRespVOList = respVOList;
        tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.queryMenuByAppCode(appCodes, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                })
                .forEach(t -> {
                    MenuTreeRespVO respVO = new MenuTreeRespVO();
                    respVO.setMenuCode(t.getMenusCode());
                    respVO.setMenuName(t.getMenusName());
                    respVO.setNodeType(MenuTreeNodeType.MENU.getValue());
                    respVO.setParentMenuCode(t.getMenusParentCode());
                    if (CharSequenceUtil.isBlank(t.getMenusParentCode())) {
                        // 顶级的设置应用编码
                        respVO.setParentMenuCode(t.getMenusAppCode());
                    }
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setHidden(t.getDisplay() != null && !t.getDisplay());
                    respVO.setEnabled(t.getMenusState() == null || t.getMenusState());
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.getMenusOrder(), 1));
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(128));

                    finalRespVOList.add(respVO);
                });

        // 转树形
        if (BooleanUtil.isFalse(tree)) {
            return ApiResult.ok(respVOList);
        }
        TreeDataUtil<MenuTreeRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, MenuTreeRespVO::getMenuCode,
                MenuTreeRespVO::getParentMenuCode, MenuTreeRespVO::setChildren, Comparator.comparingInt(MenuTreeRespVO::getSortNo));
        respVOList = ((List<MenuTreeRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> MenuTreeNodeType.APP.getValue().equals(t.getNodeType()))
                .collect(Collectors.toList());

        return ApiResult.ok(respVOList);
    }

    @Override
    public ApiResult<List<MenuTreeCustomRespVO>> getTreeCustom(Boolean tree) {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        var tenantId = currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
        // 查询租户的菜单
        var menuTreeDOList = tenantMenuTreeRepoProc.queryByTenantId(tenantId);
        if (menuTreeDOList.isEmpty()) {
            // 尚未设置
            return ApiResult.ok(Collections.emptyList());
        }

        // 查询默认菜单
        var menuDOMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.queryMenuByAppCode(null, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                }).collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));
        var allApps = appProvider.all().stream().collect(Collectors.toMap(CodeNameParam::getCode, t -> t, (t1, t2) -> t1));

        // 组织vo
        List<MenuTreeCustomRespVO> menuVOList = new ArrayList<>(menuTreeDOList.size());
        MenuTreeCustomRespVO menuVO = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            menuVO = new MenuTreeCustomRespVO();
            menuVO.setMenuCode(menuTreeDO.getMenuCode());
            menuVO.setCustom(menuTreeDO.getCustom());
            menuVO.setMenuName(menuTreeDO.getMenuName());

            // 设置节点原始信息
            var nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            if (nodeType == null) {
                log.error("未知菜单节点类型：" + menuTreeDO.getNodeType());
                continue;
            }
            if (MenuTreeNodeType.MENU == nodeType) {
                MenuBO menuBO = menuDOMap.get(menuTreeDO.getMenuCode());
                if (menuBO == null) {
                    // 是否是自定义菜单
                    if (!menuTreeDO.getCustom()) {
                        log.info("菜单不存在：" + menuTreeDO.getMenuCode());
                        continue;
                    }
                    menuVO.setHidden(false);
                    menuVO.setEnabled(true);
                } else {
                    if (CharSequenceUtil.isBlank(menuVO.getMenuName())) {
                        menuVO.setMenuName(menuBO.getMenusName());
                    }
                    menuVO.setOriginalMenuName(menuBO.getMenusName());
                    menuVO.setOriginalMenusIcon(menuBO.getMenusIcon());

                    menuVO.setHidden(menuBO.getDisplay() != null && !menuBO.getDisplay());
                    menuVO.setEnabled(menuBO.getMenusState() == null || menuBO.getMenusState());
                }
            } else if (MenuTreeNodeType.APP == nodeType) {
                var app = allApps.get(menuTreeDO.getMenuCode());
                if (app == null) {
                    log.info("租户应用已不存在：{}", menuTreeDO.getMenuCode());
                    continue;
                }
                if (CharSequenceUtil.isBlank(menuVO.getMenuName())) {
                    menuVO.setMenuName(app.getName());
                }
                menuVO.setOriginalMenuName(app.getName());
                menuVO.setOriginalMenusIcon("");
                menuVO.setHidden(false);
                menuVO.setEnabled(true);
            } else {
                log.error("暂不支持的节点类型：{}", nodeType.getValue());
                continue;
            }
            menuVO.setNodeType(menuTreeDO.getNodeType());
            menuVO.setParentMenuCode(menuTreeDO.getParentMenuCode());
            menuVO.setMenusIcon(menuTreeDO.getIcon());
            menuVO.setSortNo(menuTreeDO.getSortNo());
            menuVO.setHasChildren(false);
            menuVO.setChildren(new ArrayList<>(128));

            menuVOList.add(menuVO);
        }

        // 树形结构
        if (BooleanUtil.isFalse(tree)) {
            return ApiResult.ok(menuVOList);
        }
        TreeDataUtil<MenuTreeCustomRespVO> treeDataUtil = new TreeDataUtil<>(menuVOList, MenuTreeCustomRespVO::getMenuCode,
                MenuTreeCustomRespVO::getParentMenuCode, MenuTreeCustomRespVO::setChildren, Comparator.comparing(MenuTreeCustomRespVO::getSortNo, Comparator.nullsLast(Integer::compareTo)));
        return ApiResult.ok((List<MenuTreeCustomRespVO>) treeDataUtil.getRoots());
    }

    @Override
    public ApiResult<Boolean> getEnabled() {
        var tenantId = currentTenantId();
        var enabled = tenantMenuRepoProc.getEnabled(tenantId);
        return ApiResult.ok(ObjectUtil.defaultIfNull(enabled, false));
    }

    private List<MenuTreeRespVO> convertMenuTreeRespVO(Set<String> appCodes) {
        // 先获取应用
        var allApps = appProvider.all();
        if (allApps.isEmpty()) {
            throw new BusinessException("未获取都有效的应用列表");
        }

        List<MenuTreeRespVO> respVOList = new ArrayList<>(256);
        if (CollectionUtils.isEmpty(appCodes)) {
            // 所有应用
            MenuTreeRespVO respVO = null;
            int sortNo = 1;
            for (CodeNameParam app : allApps) {
                respVO = new MenuTreeRespVO();
                respVO.setMenuCode(app.getCode());
                respVO.setMenuName(app.getName());
                respVO.setNodeType(MenuTreeNodeType.APP.getValue());
                respVO.setHidden(false);
                respVO.setEnabled(true);
                respVO.setSortNo(sortNo++);
                respVO.setHasChildren(false);
                respVO.setChildren(new ArrayList<>(128));

                respVOList.add(respVO);
            }
            return respVOList;
        }

        MenuTreeRespVO respVO = null;
        int sortNo = 1;
        for (var app : allApps) {
            if (!appCodes.contains(app.getCode())) {
                // 未分配当前应用
                continue;
            }

            respVO = new MenuTreeRespVO();
            respVO.setMenuCode(app.getCode());
            respVO.setMenuName(app.getName());
            respVO.setNodeType(MenuTreeNodeType.APP.getValue());
            respVO.setHidden(false);
            respVO.setEnabled(true);
            respVO.setSortNo(sortNo++);
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(128));

            respVOList.add(respVO);
        }

        return respVOList;
    }

    private List<SysTenantMenuTreeDO> convertAndCheck(Long tenantId, List<MenuTreeSaveVO> saveVOList) {
        if (CollectionUtils.isEmpty(saveVOList)) {
            return Collections.emptyList();
        }
        // 转为do列表
        List<SysTenantMenuTreeDO> menuTreeDOList = new ArrayList<>(256);
        convertToList(saveVOList, null, menuTreeDOList);

        var menuCodes = menuTreeDOList.stream().map(SysTenantMenuTreeDO::getMenuCode).collect(Collectors.toSet());
        // 检查参数
        MenuTreeNodeType nodeType = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            menuTreeDO.setSysTenantId(tenantId);

            if (StringUtils.hasText(menuTreeDO.getParentMenuCode()) && !menuCodes.contains(menuTreeDO.getParentMenuCode())) {
                // 去掉顶级菜单的parentCode
                menuTreeDO.setParentMenuCode("");
            }

            // 节点类型
            nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            Assert.notNull(nodeType, "未知节点类型" + menuTreeDO.getNodeType());

            // 是否自定义
            if (menuTreeDO.getCustom() == null) {
                menuTreeDO.setCustom(false);
            }
            if (nodeType == MenuTreeNodeType.APP) {
                menuTreeDO.setCustom(false);
            }
        }

        // 设置排序
        var menuTreeMap = menuTreeDOList.stream().collect(Collectors.groupingBy(SysTenantMenuTreeDO::getParentMenuCode));
        for (List<SysTenantMenuTreeDO> menus : menuTreeMap.values()) {
            int sortNo = 0;
            for (SysTenantMenuTreeDO menu : menus) {
                menu.setSortNo(sortNo++);
            }
        }

        return menuTreeDOList;
    }

    private void convertToList(List<MenuTreeSaveVO> saveVOList, String parentMenuCode, List<SysTenantMenuTreeDO> menuTreeDOList) {
        if (CollectionUtils.isEmpty(saveVOList)) {
            return;
        }
        if (parentMenuCode == null) {
            parentMenuCode = "";
        }

        int sortNo = 1;
        List<SysTenantMenuTreeDO> newList = new ArrayList<>(128);
        SysTenantMenuTreeDO menuTreeDO = null;
        for (MenuTreeSaveVO menuTreeSaveVO : saveVOList) {
            menuTreeDO = CONVERT.saveVo2Do(menuTreeSaveVO);
            if (StringUtils.hasText(parentMenuCode) || menuTreeDO.getParentMenuCode() == null) {
                menuTreeDO.setParentMenuCode(parentMenuCode);
            }
            menuTreeDO.setIcon(menuTreeSaveVO.getMenusIcon());
            menuTreeDO.setSortNo(sortNo);

            sortNo++;
            newList.add(menuTreeDO);

            // 如果有子列表，则遍历下级
            convertToList(menuTreeSaveVO.getChildren(), menuTreeSaveVO.getMenuCode(), menuTreeDOList);
        }
        menuTreeDOList.addAll(newList);
    }
}
