package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysBusinessUnit;
import com.elitescloud.cloudt.constant.SysFunctionType;
import com.elitescloud.cloudt.constant.SysProfitablyCenter;
import com.elitescloud.cloudt.constant.SysRegion;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.OrgType;
import com.elitescloud.cloudt.system.convert.OrgConvert;
import com.elitescloud.cloudt.system.dto.req.OrgUpsertDTO;
import com.elitescloud.cloudt.system.model.vo.query.org.OrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.OrgSaveVO;
import com.elitescloud.cloudt.system.service.OrgMngService;
import com.elitescloud.cloudt.system.service.manager.OrgMngManager;
import com.elitescloud.cloudt.system.service.manager.OrgTreeManager;
import com.elitescloud.cloudt.system.service.model.bo.SysOrgSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysOrgDO;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.OuRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/29
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class OrgMngServiceImpl extends BaseServiceImpl implements OrgMngService {
    private static final OrgConvert CONVERT = OrgConvert.INSTANCE;

    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private OuRepoProc ouRepoProc;
    @Autowired
    private OrgMngManager orgManager;
    @Autowired
    private OrgTreeManager orgTreeManager;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> upsert(OrgSaveVO saveVO) {
        // 转BO
        if (saveVO.getExecutive() == null) {
            saveVO.setExecutive(true);
        }
        SysOrgSaveBO saveBo = saveVO.getId() == null ? this.toSaveBoForAdd(saveVO) : this.toSaveBoForUpdate(saveVO);

        // 保存数据
        var orgDo = orgManager.upsert(saveBo);
        return ApiResult.ok(orgDo.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> upsert(OrgUpsertDTO upsertDTO) {
        // 转BO
        if (upsertDTO.getExecutive() == null) {
            upsertDTO.setExecutive(true);
        }
        SysOrgSaveBO saveBo = upsertDTO.getId() == null ? this.toSaveBoForAdd(upsertDTO) : this.toSaveBoForUpdate(upsertDTO);

        // 保存数据
        var orgDo = orgManager.upsert(saveBo);
        return ApiResult.ok(orgDo.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        orgManager.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Assert.notNull(id, "ID为空");
        // 获取现有状态
        var enabled = orgRepoProc.getEnabled(id);

        // 更新状态
        enabled = enabled == null || !enabled;

        return this.updateEnabled(id, enabled);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        Assert.notNull(id, "ID为空");

        // 更新状态
        enabled = enabled == null || enabled;
        orgManager.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> updateTreeRebuild(Long rootOrgId) {
        orgManager.rebuildTree(rootOrgId);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<List<OrgTreeNodeRespVO>> getOrgTree(Boolean tree) {
        var currentUser = super.currentUser(true);
        if (!hasPermission(currentUser)) {
            return ApiResult.fail("无权限");
        }

        Long orgIdBelong = null;
        if (currentUser.isTenantOrgAdmin()) {
            orgIdBelong = currentUser.getTenantOrgId();
            Assert.notNull(orgIdBelong, "未知所属租户组织");
        }
        var treeData = orgTreeManager.getTree(orgIdBelong, tree == null || tree, true, false);
        return ApiResult.ok(treeData);
    }

    @Override
    public ApiResult<OrgDetailRespVO> getDetail(Long id, String withParentType) {
        var org = orgRepoProc.get(id);
        if (org == null) {
            return ApiResult.noData();
        }

        var orgRespVO = CONVERT.do2DetailRespVO(org);

        // 上级组织
        if (org.getPId() != null) {
            orgRespVO.setParentId(org.getPId());
            orgRespVO.setParentName(orgRepoProc.getNameById(org.getPId()));
        }
        // 组织类型
        orgRespVO.setTypeName(super.udcValue(new OrgType(orgRespVO.getType())));
        // 公司
        if (org.getOuId() != null) {
            orgRespVO.setOuName(ouRepoProc.getOuName(org.getOuId()));
        }
        orgRespVO.setRegionName(super.udcValue(new SysRegion(orgRespVO.getRegion())));
        orgRespVO.setBusinessUnitName(super.udcValue(new SysBusinessUnit(orgRespVO.getBusinessUnit())));
        orgRespVO.setProfitablyCenterName(super.udcValue(new SysProfitablyCenter(orgRespVO.getProfitablyCenter())));
        orgRespVO.setFunctionTypeName(super.udcValue(new SysFunctionType(orgRespVO.getFunctionType())));

        // 获取指定的上级
        if (StringUtils.hasText(withParentType)) {
            var withParent = orgRepoProc.queryParentNameForType(List.of(id), withParentType, false).get(id);
            orgRespVO.setWithParent(withParent);
        }

        return ApiResult.ok(orgRespVO);
    }

    @Override
    public ApiResult<PagingVO<OrgPagedRespVO>> queryByPage(OrgPageQueryVO queryVO) {
        var pageData = orgRepoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }

        // 查询父节点名称
        var parentCodes = pageData.getRecords().stream().map(SysOrgDO::getParentCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        var parentNameMap = parentCodes.isEmpty() ? new HashMap<String, String>(0) :
                orgRepoProc.getNameByCode(parentCodes);

        // 查询公司名称
        var ouIds = pageData.stream().map(SysOrgDO::getOuId).filter(Objects::nonNull).collect(Collectors.toSet());
        var ouNameMap = ouIds.isEmpty() ? new HashMap<String, String>(0) :
                ouRepoProc.getNameMapById(ouIds);

        // udc组织类型
        var orgTypeMap = super.udcMap(new OrgType());
        var pageResult = pageData.map(t -> {
            var respVO = CONVERT.do2MngPageRespVO(t);
            respVO.setParentName(parentNameMap.get(t.getParentCode()));
            respVO.setOuName(ouNameMap.get(t.getOuId()));
            respVO.setTypeName(orgTypeMap.get(t.getType()));
            return respVO;
        });

        return ApiResult.ok(pageResult);
    }

    private SysOrgSaveBO toSaveBoForAdd(OrgSaveVO saveVO) {
        SysOrgSaveBO saveBo = CONVERT.saveVo2SaveBo(saveVO);

        // 上级组织
        if (StringUtils.hasText(saveVO.getParentCode())) {
            boolean exists = orgRepoProc.existsCode(saveVO.getParentCode());
            Assert.isTrue(exists, "上级组织不存在");
        } else if (saveVO.getParentId() != null && saveVO.getParentId() != BaseTreeModel.DEFAULT_PARENT) {
            String pCode = orgRepoProc.getCodeById(saveVO.getParentId());
            Assert.hasText(pCode, "上级组织不存在");
            saveBo.setParentCode(pCode);
        }

        return saveBo;
    }

    private SysOrgSaveBO toSaveBoForAdd(OrgUpsertDTO upsertDTO) {
        SysOrgSaveBO saveBo = CONVERT.upsertDTO2SaveBo(upsertDTO);

        // 上级组织
        if (upsertDTO.getParentId() != null && upsertDTO.getParentId() != BaseTreeModel.DEFAULT_PARENT) {
            String pCode = orgRepoProc.getCodeById(upsertDTO.getParentId());
            Assert.hasText(pCode, "上级组织不存在");
            saveBo.setParentCode(pCode);
        }

        return saveBo;
    }

    private SysOrgSaveBO toSaveBoForUpdate(OrgSaveVO saveVO) {
        var orgDO = orgRepoProc.get(saveVO.getId());
        Assert.notNull(orgDO, "组织信息不存在");

        SysOrgSaveBO saveBo = CONVERT.do2SaveBo(orgDO);
        CONVERT.copySaveVo2Bo(saveVO, saveBo);

        // 上级组织
        if (StringUtils.hasText(saveVO.getParentCode())) {
            if (!saveVO.getParentCode().equals(orgDO.getParentCode())) {
                boolean exists = orgRepoProc.existsCode(saveVO.getParentCode());
                Assert.isTrue(exists, "上级组织不存在");
            }
        } else if (saveVO.getParentId() != null && saveVO.getParentId() != BaseTreeModel.DEFAULT_PARENT) {
            if (saveVO.getParentId().longValue() == orgDO.getPId()) {
                saveBo.setParentCode(orgDO.getParentCode());
            } else {
                String pCode = orgRepoProc.getCodeById(saveVO.getParentId());
                Assert.hasText(pCode, "上级组织不存在");
                saveBo.setParentCode(pCode);
            }
        }

        return saveBo;
    }

    private SysOrgSaveBO toSaveBoForUpdate(OrgUpsertDTO upsertDTO) {
        var orgDO = orgRepoProc.get(upsertDTO.getId());
        Assert.notNull(orgDO, "组织信息不存在");

        SysOrgSaveBO saveBo = CONVERT.do2SaveBo(orgDO);
        CONVERT.copyUpsertDTO2Bo(upsertDTO, saveBo);

        // 上级组织
        if (upsertDTO.getParentId() != null && upsertDTO.getParentId() != BaseTreeModel.DEFAULT_PARENT) {
            if (upsertDTO.getParentId().longValue() == orgDO.getPId()) {
                saveBo.setParentCode(orgDO.getParentCode());
            } else {
                String pCode = orgRepoProc.getCodeById(upsertDTO.getParentId());
                Assert.hasText(pCode, "上级组织不存在");
                saveBo.setParentCode(pCode);
            }
        }

        return saveBo;
    }

    private boolean hasPermission(GeneralUserDetails currentUser) {
        if (currentUser == null) {
            return false;
        }
//        return currentUser.isSystemAdmin() || currentUser.isTenantAdmin() || currentUser.isTenantOrgAdmin();
        return true;
    }
}
