package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysBusinessUnit;
import com.elitescloud.cloudt.constant.SysFunctionType;
import com.elitescloud.cloudt.constant.SysProfitablyCenter;
import com.elitescloud.cloudt.constant.SysRegion;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.OrgType;
import com.elitescloud.cloudt.system.convert.OrgConvert;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.req.SysOrgPageQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgDetailRespDTO;
import com.elitescloud.cloudt.system.dto.resp.SysOrgPageRespDTO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgPagedRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;
import com.elitescloud.cloudt.system.service.OrgQueryService;
import com.elitescloud.cloudt.system.service.manager.OrgTreeManager;
import com.elitescloud.cloudt.system.service.model.entity.SysOrgDO;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.OuRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/10
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Slf4j
public class OrgQueryServiceImpl extends BaseServiceImpl implements OrgQueryService {
    private static final OrgConvert CONVERT = OrgConvert.INSTANCE;

    @Autowired
    private OrgRepoProc repoProc;
    @Autowired
    private OuRepoProc ouRepoProc;
    @Autowired
    private OrgTreeManager orgTreeManager;

    @Override
    public ApiResult<List<OrgTreeNodeRespVO>> getOrgTree(Boolean tree, Boolean employee, Boolean all) {
        // 默认返回树形
        tree = tree == null || tree;
        // 默认不包含员工
        employee = employee != null && employee;

        // 返回所属租户组织
        var currentUser = super.currentUser(true);

        Long orgIdBelong = null;
        if (!currentUser.isSystemAdmin() && !currentUser.isTenantAdmin()) {
            if (!Boolean.TRUE.equals(all)) {
                orgIdBelong = ObjectUtil.defaultIfNull(currentUser.getTenantOrgId(), currentUser.getOrgId());
                if (orgIdBelong == null) {
                    return ApiResult.ok(Collections.emptyList());
                }
            }
        }

        // 查询组织树
        var respVO = orgTreeManager.getTree(orgIdBelong, tree, false, employee);

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<OrgTreeNodeRespVO>> getOrgTreeAsync(Long pId, Boolean employee, Boolean all) {
        // 默认不包含员工
        employee = employee != null && employee;

        // 返回所属租户组织
        var currentUser = super.currentUser(true);
        if (!currentUser.isSystemAdmin() && !currentUser.isTenantAdmin()) {
            if (pId == null || pId == BaseTreeModel.DEFAULT_PARENT) {
                if (!Boolean.TRUE.equals(all)) {
                    pId = ObjectUtil.defaultIfNull(currentUser.getTenantOrgId(), currentUser.getOrgId());
                }
            }
        }

        // 查询组织树
        var respVO = orgTreeManager.getTreeAsync(pId, false, employee);

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<Long>> querySubOrgIds(Long id) {
        var ids = repoProc.getIdByPid(id);
        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<List<Long>> queryBelongOrgIds(Long id) {
        var ids = repoProc.getChildrenIdByPid(id);
        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<List<Long>> getIdsByOrgName(String orgName) {
        Assert.hasText(orgName, "组织名称为空");

        var ids = repoProc.getIdsByName(orgName);
        return ApiResult.ok(ids);
    }

    @Override
    public ApiResult<Long> getIdByOrgCode(String orgCode) {
        Assert.hasText(orgCode, "组织编码为空");

        var id = repoProc.getIdByCode(orgCode);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> queryCodeAndName(Set<Long> ids) {
        Assert.notEmpty(ids, "ID为空");

        var dataList = repoProc.queryIdCodeName(ids);
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> queryIdAndName(Set<String> codes) {
        Assert.notEmpty(codes, "编码为空");

        var dataList = repoProc.queryIdCodeNameByCodes(codes);
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<SysOrgBasicDTO> getById(Long id) {
        var dto = repoProc.getBasicDto(id);
        this.wrapBasicDTO(dto);
        return ApiResult.ok(dto);
    }

    @Override
    public ApiResult<SysOrgDetailRespDTO> getDetailById(Long id, String withParentType) {
        var orgDO = repoProc.get(id);
        if (orgDO == null) {
            return ApiResult.fail("组织不存在");
        }
        return ApiResult.ok(wrapDetailDTO(orgDO, withParentType));
    }

    @Override
    public ApiResult<SysOrgBasicDTO> getByCode(String code) {
        var dto = repoProc.getBasicDtoByCode(code);
        this.wrapBasicDTO(dto);
        return ApiResult.ok(dto);
    }

    @Override
    public ApiResult<SysOrgDetailRespDTO> getDetailByCode(String code, String withParentType) {
        var orgDO = repoProc.getByCode(code);
        if (orgDO == null) {
            return ApiResult.fail("组织不存在");
        }
        return ApiResult.ok(wrapDetailDTO(orgDO, withParentType));
    }

    @Override
    public ApiResult<SysOrgBasicDTO> getParentByCode(String code, String parentType) {
        if (CharSequenceUtil.isBlank(code)) {
            return ApiResult.fail("组织编码为空");
        }

        // 查询直接上级
        if (CharSequenceUtil.isBlank(parentType)) {
            return ApiResult.ok(repoProc.getParentBasicDtoByCode(code));
        }

        // 查询ID
        var id = repoProc.getIdByCode(code);
        if (id == null) {
            return ApiResult.fail("组织不存在");
        }
        // 查询上级
        var withParent = repoProc.queryParentNameForType(List.of(id), parentType, true).get(id);
        if (withParent == null) {
            return ApiResult.ok();
        }

        return ApiResult.ok(repoProc.getBasicDto(withParent.getId()));
    }

    @Override
    public ApiResult<Map<String, SysOrgBasicDTO>> getParentByCode(Set<String> codes, String parentType) {
        if (CollUtil.isEmpty(codes)) {
            return ApiResult.fail("组织编码为空");
        }

        // 查询直接上级
        if (CharSequenceUtil.isBlank(parentType)) {
            var parentCodeMap = repoProc.getParentCodeByCode(codes);
            if (parentCodeMap.isEmpty()) {
                return ApiResult.ok(Collections.emptyMap());
            }
            var parentMap = repoProc.getParentBasicDtoByCode(codes).stream().collect(Collectors.toMap(SysOrgBasicDTO::getCode, t -> t, (t1, t2) -> t1));
            Map<String, SysOrgBasicDTO> result = new HashMap<>(parentCodeMap.size());
            for (Map.Entry<String, String> entry : parentCodeMap.entrySet()) {
                var parent = parentMap.get(entry.getValue());
                if (parent == null) {
                    continue;
                }
                result.put(entry.getKey(), parent);
            }
            return ApiResult.ok(result);
        }

        // 查询ID
        var idMap = repoProc.getIdByCode(codes);
        if (idMap.isEmpty()) {
            return ApiResult.ok(Collections.emptyMap());
        }
        var parentMap = repoProc.queryParentNameForType(idMap.values(), parentType, true);
        if (parentMap.isEmpty()) {
            return ApiResult.ok(Collections.emptyMap());
        }
        var parentIds = parentMap.values().stream().map(IdCodeNameParam::getId).collect(Collectors.toSet());
        var parentDtoMap = repoProc.getBasicDtoList(parentIds).stream().collect(Collectors.toMap(SysOrgBasicDTO::getId, t -> t, (t1, t2) -> t1));
        Map<String, SysOrgBasicDTO> result = new HashMap<>(idMap.size());
        for (Map.Entry<String, Long> entry : idMap.entrySet()) {
            var parent = parentDtoMap.get(entry.getValue());
            if (parent == null) {
                continue;
            }
            result.put(entry.getKey(), parent);
        }
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<SysOrgBasicDTO>> queryList(SysOrgQueryDTO queryDTO) {
        var dataList = repoProc.queryList(queryDTO).stream().map(this::wrapBasicDTO).collect(Collectors.toList());
        return ApiResult.ok(dataList);
    }

    @Override
    public ApiResult<PagingVO<OrgPagedRespVO>> page(CommonOrgPageQueryVO queryVO) {
        var pageData = repoProc.pageQuery(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(CONVERT::do2PageRespVO));
        }

        var udcType = super.udcMap(new OrgType());
        // 父节点
        var parentCodes = pageData.stream()
                .map(SysOrgDO::getParentCode)
                .filter(StringUtils::hasText)
                .collect(Collectors.toSet());
        Map<String, String> nameMap = parentCodes.isEmpty() ? Collections.emptyMap() : repoProc.getNameByCode(parentCodes);

        var pageResult = pageData
                .map(t -> {
                    var vo = CONVERT.do2PageRespVO(t);
                    vo.setParentName(nameMap.get(t.getParentCode()));
                    vo.setTypeName(udcType.get(t.getType()));
                    return vo;
                });
        return ApiResult.ok(pageResult);
    }

    @Override
    public ApiResult<PagingVO<SysOrgPageRespDTO>> queryPage(SysOrgPageQueryDTO queryDTO) {
        var pageData = repoProc.pageQuery(queryDTO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(CONVERT::do2PageRespDTO));
        }

        var ids = pageData.getRecords().stream().map(SysOrgDO::getId).collect(Collectors.toList());

        var udcType = super.udcMap(new OrgType());
        if (StringUtils.hasText(queryDTO.getWithParentType())) {
            // 不包含的上级类型，则忽略
            if (!udcType.containsKey(queryDTO.getWithParentType())) {
                log.warn("不包含的组织上级类型：{}，将忽略", queryDTO.getWithParentType());
                queryDTO.setWithParentType(null);
            }
        }
        Map<Long, IdCodeNameParam> withParent = StringUtils.hasText(queryDTO.getWithParentType()) ?
                repoProc.queryParentNameForType(ids, queryDTO.getWithParentType(), Boolean.TRUE.equals(queryDTO.getEnabled())) :
                Collections.emptyMap();

        // 转换数据
        var pageResult = pageData.map(t -> {
            var vo = CONVERT.do2PageRespDTO(t);
            vo.setTypeName(udcType.get(vo.getType()));

            var tempParent = withParent.get(vo.getId());
            if (tempParent != null) {
                vo.setWithParentId(tempParent.getId());
                vo.setWithParentCode(tempParent.getCode());
                vo.setWithParentName(tempParent.getName());
            }
            return vo;
        });
        return ApiResult.ok(pageResult);
    }

    @Override
    public ApiResult<IdCodeNameParam> getParentOrgByType(Long id, String code, String parentType) {
        if (id == null && CharSequenceUtil.isBlank(code)) {
            // 默认获取当前用户的组织
            var currentUser = super.currentUser(false);
            if (currentUser != null) {
                id = currentUser.getOrgId();
            }
            if (id == null) {
                return ApiResult.fail("请选择组织");
            }
        }

        if (id == null) {
            id = repoProc.getIdByCode(code);
        }
        if (id == null) {
            return ApiResult.fail("组织不存在");
        }
        var parent = repoProc.queryParentNameForType(List.of(id), parentType, true).get(id);
        return ApiResult.ok(parent);
    }

    private SysOrgDetailRespDTO wrapDetailDTO(SysOrgDO orgDO, String withParentType) {
        var respDTO = CONVERT.do2DetailRespDTO(orgDO);

        // 上级组织
        if (orgDO.getPId() != null) {
            respDTO.setParentId(orgDO.getPId());
            respDTO.setParentName(repoProc.getNameById(orgDO.getPId()));
        }
        // 组织类型
        respDTO.setTypeName(super.udcValue(new OrgType(respDTO.getType())));
        // 公司
        if (respDTO.getOuId() != null) {
            respDTO.setOuName(ouRepoProc.getOuName(respDTO.getOuId()));
        }

        respDTO.setTypeName(super.udcValue(new OrgType(respDTO.getType())));
        respDTO.setRegionName(super.udcValue(new SysRegion(respDTO.getRegion())));
        respDTO.setBusinessUnitName(super.udcValue(new SysBusinessUnit(respDTO.getBusinessUnit())));
        respDTO.setProfitablyCenterName(super.udcValue(new SysProfitablyCenter(respDTO.getProfitablyCenter())));
        respDTO.setFunctionTypeName(super.udcValue(new SysFunctionType(respDTO.getFunctionType())));

        // 获取指定类型的上级
        if (CharSequenceUtil.isNotBlank(withParentType)) {
            var withParent = repoProc.queryParentNameForType(List.of(orgDO.getId()), withParentType, true).get(orgDO.getId());
            respDTO.setWithParent(withParent);
        }

        return respDTO;
    }

    private SysOrgBasicDTO wrapBasicDTO(SysOrgBasicDTO dto) {
        if (dto == null) {
            return null;
        }
        if (CharSequenceUtil.isBlank(dto.getParentCode())) {
            dto.setParentId(null);
        }
        return dto;
    }
}
