package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.provider.TenantOrgProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.RoleType;
import com.elitescloud.cloudt.system.convert.PermissionConverter;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.bo.PermissionParameterBO;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprApiCustomRuleRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprApiFieldRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.dpr.SysDprRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.PermissionMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.provider.dto.SysApiPermissionMetadataDTO;
import com.elitescloud.cloudt.system.service.PermissionQueryService;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.manager.PermissionMngManager;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.manager.RoleMngManager;
import com.elitescloud.cloudt.system.service.model.entity.SysDprRoleApiRowRuleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.vo.DprUdcVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/7
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class PermissionQueryServiceImpl extends BaseServiceImpl implements PermissionQueryService {

    @Autowired
    private ApiRepoProc apiRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private MenuRepoProc menusRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;
    @Autowired
    private MenuApiRepoProc menuApiRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc roleApiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;
    @Autowired
    private ApiParameterRepoProc apiParameterRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private TenantOrgRepoProc tenantOrgRepoProc;
    @Autowired
    private TenantOrgProvider tenantOrgProvider;

    @Autowired
    private PermissionMngManager permissionMngManager;
    @Autowired
    private PermissionQueryManager permissionQueryManager;
    @Autowired
    private RoleMngManager roleMngManager;

    @Override
    public ApiResult<List<RolePermissionRespVO>> getPermissionMenuByRole(Long roleId, String appCode, Boolean tree) {
        Assert.notNull(roleId, "角色ID为空");

        tree = tree == null || tree;
        var menuList = permissionMngManager.getPermissionMenuByRole(roleId, appCode, false, true, tree);
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<PermissionDetailRespVO> getPermissionDetailByCode(String code) {
        Assert.hasText(code, "权限编码为空");

        // 是否自定义菜单树
        var tenantId = super.currentTenantId();
        var enabledCustom = ObjectUtil.defaultIfNull(tenantMenuRepoProc.getEnabled(tenantId), false);

        // 获取信息
        var respVo = enabledCustom ? this.detailForCustom(code, tenantId) : this.detailForDefault(code, tenantId);
        if (respVo == null) {
            return ApiResult.ok(this.detailForApp(code));
        }

        // 详细信息
        if (StrUtil.isBlank(respVo.getNodeType())) {
            return ApiResult.fail("未知节点类型");
        }
        if (MenuTreeNodeType.APP.getValue().equals(respVo.getNodeType())) {
            // 节点是应用
            respVo = this.detailForApp(code);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.MENU.getValue().equals(respVo.getNodeType())) {
            // 节点是菜单
            this.fillApp(respVo);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.ACTION.getValue().equals(respVo.getNodeType())) {
            // 节点是按钮
            this.fillApp(respVo);
            this.fillMenu(respVo);
            return ApiResult.ok(respVo);
        }
        if (MenuTreeNodeType.API.getValue().equals(respVo.getNodeType())) {
            // 节点是API
            this.fillApp(respVo);
            this.fillMenu(respVo);
            return ApiResult.ok(respVo);
        }

        return ApiResult.fail("不支持的节点类型：" + respVo.getNodeType());
    }

    @Override
    public ApiResult<List<SysApiPermissionMetadataDTO>> queryApiMetadata(String appCode) {
        Assert.hasText(appCode, "应用编码为空");

        // 先查询应用下的权限
        var permissionMetadataList = tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.queryPermissionMetadata(appCode));
        if (permissionMetadataList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 判断是否有自定义菜单树
        var tenantId = super.currentTenantId();
        var enabledCustom = tenantMenuRepoProc.getEnabled(tenantId);
        // 查询权限与角色关联
        var rolePermissionMap = rolePermissionRepoProc.queryRoleCodeForApi(ObjectUtil.defaultIfNull(enabledCustom, false));

        for (SysApiPermissionMetadataDTO dto : permissionMetadataList) {
            dto.setRoleCodes(rolePermissionMap.getOrDefault(dto.getPermissionCode(), Collections.emptySet()));
        }

        return ApiResult.ok(permissionMetadataList);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRanges() {
        var result = Arrays.stream(DprSysInternallyEnum.values())
                .map(DprSysInternallyEnum::getUdcVO)
                .sorted(Comparator.comparing(DprUdcVO::getValSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> new CodeNameParam(t.getUdcVal(), t.getValDesc()))
                .collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRuleCondition() {
        var result = Arrays.stream(DprRuleConditionEnum.values())
                .map(DprRuleConditionEnum::getUdcVO)
                .sorted(Comparator.comparing(DprUdcVO::getValSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> new CodeNameParam(t.getUdcVal(), t.getValDesc()))
                .collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<CodeNameParam>> dataPermissionRuleValueMethod() {
        var result = Stream.of(
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM,
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS,
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS
                )
                .map(DprRuleValueTypeEnum::getUdcVO)
                .sorted(Comparator.comparing(DprUdcVO::getValSortNo, Comparator.nullsLast(Integer::compareTo)))
                .map(t -> new CodeNameParam(t.getUdcVal(), t.getValDesc()))
                .collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<SysMenuApiRespVO>> getApi(String menuCode) {
        Assert.hasText(menuCode, "菜单编码为空");

        Map<String, String> appNameMap = new HashMap<>(32);
        var apiList = tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    for (IdCodeNameParam param : appRepoProc.allParams(null)) {
                        appNameMap.put(param.getCode(), param.getName());
                    }

                    return menuApiRepoProc.queryApiOfMenu(menuCode);
                }).stream()
                .map(t -> {
                    SysMenuApiRespVO respVO = new SysMenuApiRespVO();
                    respVO.setId(t.getId());
                    respVO.setAppCode(t.getAppCode());
                    respVO.setAppName(appNameMap.get(t.getAppCode()));
                    respVO.setName(t.getApiName());
                    respVO.setCode(t.getApiCode());
                    respVO.setRequestType(t.getRequestType());
                    respVO.setApiPath(t.getApiPath());
                    respVO.setApiDescribe(t.getApiDescribe());
                    respVO.setMenusCode(menuCode);

                    return respVO;
                }).collect(Collectors.toList());
        return ApiResult.ok(apiList);
    }

    @Override
    public ApiResult<List<Long>> getMenuIdOfApi(Long apiId, String appCode) {
        Assert.notNull(apiId, "接口ID为空");
        var menuIds = menuApiRepoProc.getMenuIdByApiId(apiId, appCode);
        return ApiResult.ok(menuIds);
    }

    @Override
    public ApiResult<List<PermissionMenuRespVO>> getMenuDetailOfApi(Long apiId, String appCode) {
        Assert.notNull(apiId, "接口ID为空");

        // 获取菜单列表
        var menuList = menuApiRepoProc.getMenuDetailByApiId(apiId, appCode).stream()
                .map(PermissionConverter.INSTANCE::convertMenuRespVO)
                .collect(Collectors.toList());
        if (menuList.isEmpty()) {
            return ApiResult.ok(menuList);
        }

        // 应用信息
        var appMap = appRepoProc.all().stream().collect(Collectors.toMap(SysPlatformAppDO::getAppCode, SysPlatformAppDO::getAppName, (t1, t2) -> t1));

        // 获取上级菜单
        var parentMenuCodes = menuList.stream().map(PermissionMenuRespVO::getMenusParentCode).filter(StringUtils::hasText).collect(Collectors.toSet());
        Map<String, String> parentMenuNameMap = parentMenuCodes.isEmpty() ? Collections.emptyMap() : menusRepoProc.getMenuNameByMenuCode(parentMenuCodes);

        // 翻译名称
        for (PermissionMenuRespVO respVO : menuList) {
            respVO.setMenusParentName(parentMenuNameMap.get(respVO.getMenusParentCode()));
            respVO.setMenusAppName(appMap.get(respVO.getMenusAppCode()));
        }

        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<SysDprRespVO> getApiDataPermission(Long roleId, String menuCode, String apiCode) {
        Assert.notNull(roleId, "角色ID为空");
        Assert.hasText(menuCode, "菜单编码为空");
        Assert.hasText(apiCode, "接口编码为空");

        // 查询自定义规则
        var respVO = queryRoleCustomRule(roleId, menuCode, apiCode);
        // 查询字段权限
        respVO.setFieldList(this.queryRoleApiFields(roleId, menuCode, apiCode));
        // 查询入参
        respVO.setArgList(this.queryReqArguments(apiCode));

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<Set<Long>> getUserIdByRoleCode(String roleCode) {
        Assert.hasText(roleCode, "角色编码为空");

        var userIds = permissionQueryManager.queryUserIdOfRole(roleCode);
        return ApiResult.ok(userIds);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRoles(String tenantCode) {
        RoleType type = null;
        String typeId = null;

        AtomicReference<RoleType> roleTypeAtomicReference = new AtomicReference<>();
        AtomicReference<String> typeIdReference = new AtomicReference<>();
        roleMngManager.callByType((t, tId) -> {
            roleTypeAtomicReference.set(t);
            typeIdReference.set(tId);
            return null;
        });
        type = roleTypeAtomicReference.get();
        typeId = typeIdReference.get();

        // 默认获取当前租户
        if (CharSequenceUtil.isBlank(tenantCode)) {
            var roleList = roleRepoProc.listQuery(type, typeId, null, null);
            return ApiResult.ok(roleList);
        }

        // 查询指定租户的
        var tenant = tenantClientProvider.getTenantByCode(tenantCode);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }
        var roleList = tenantDataIsolateProvider.byTenantDirectly(() -> roleRepoProc.listQuery(RoleType.TENANT, tenantClientProvider.getSessionTenant().getId().toString(),
                null, null), tenant);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<List<IdCodeNameParam>> listAllRolesByTenantOrg(String tenantOrgCode) {
        Long orgId = null;
        if (StringUtils.hasText(tenantOrgCode)) {
            orgId = tenantOrgRepoProc.getOrgIdByCode(tenantOrgCode, false);
            if (orgId == null) {
                return ApiResult.fail("租户组织不存在或已禁用");
            }
        } else {
            // 获取当前用户的租户组织
            orgId = tenantOrgProvider.getSessionTenantOrgId();
            if (orgId == null) {
                return ApiResult.ok(Collections.emptyList());
            }
        }

        var roleList = roleRepoProc.listQuery(RoleType.TENANT_ORG, orgId.toString(), null, null);
        return ApiResult.ok(roleList);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> listUserByRole(String roleCode) {
        Assert.hasText(roleCode, "角色编码为空");
        var userIds = permissionQueryManager.queryUserIdOfRole(roleCode);
        if (CollUtil.isEmpty(userIds)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var userList = tenantDataIsolateProvider.byDefaultDirectly(() -> userRepoProc.getBasicDto(userIds));
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<DataPermissionTreeNodeRespVO>> dataPermissionTree(Long roleId, String appCode, Boolean tree) {
        tree = tree == null || tree;
        // 先查询菜单树
        var menuList = permissionMngManager.getDataPermissionByRole(roleId, appCode, true, tree);
        return ApiResult.ok(menuList);
    }

    @Override
    public ApiResult<SysDprRoleApiRowColumnRuleDTO> getDataPermissionOfCurrentUser() {
        var roleRuleDTO = permissionQueryManager.getDataPermissionOfCurrentUser();
        return ApiResult.ok(roleRuleDTO);
    }

    private List<SysDprApiFieldRespVO> queryReqArguments(String apiCode) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiParameterRepoProc.queryInParamByApiCode(apiCode)).stream()
                .map(t -> {
                    SysDprApiFieldRespVO respVO = new SysDprApiFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());
                    respVO.setEnabled(false);

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<SysDprApiFieldRespVO> queryRoleApiFields(Long roleId, String menuCode, String apiCode) {
        var roleFieldMap = apiFieldsRepoProc.queryByApi(roleId, menuCode, apiCode).stream()
                .collect(Collectors.toMap(PermissionParameterBO::getFieldName, t -> t, (t1, t2) -> t1));

        // 查询出参
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiParameterRepoProc.queryOutParamByApiCode(apiCode)).stream()
                .map(t -> {
                    SysDprApiFieldRespVO respVO = new SysDprApiFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());

                    var roleField = roleFieldMap.get(t.getCode());
                    if (roleField == null) {
                        // 未设置
                        respVO.setFieldApiVisible(false);
                        respVO.setFieldFormVisible(false);
                        respVO.setFieldFormUpdate(false);
                        respVO.setEnabled(false);
                        return respVO;
                    }
                    respVO.setId(roleField.getId());
                    respVO.setFieldApiVisible(Boolean.TRUE.equals(roleField.getFieldApiVisible()));
                    respVO.setFieldFormVisible(Boolean.TRUE.equals(roleField.getFieldFormVisible()));
                    respVO.setFieldFormUpdate(Boolean.TRUE.equals(roleField.getFieldFormUpdate()));
                    respVO.setEnabled(true);

                    return respVO;
                }).collect(Collectors.toList());
    }

    private SysDprRespVO queryRoleCustomRule(Long roleId, String menuCode, String apiCode) {
        SysDprRespVO respVO = new SysDprRespVO();

        var ruleDoList = roleApiRowRuleRepoProc.queryByApi(roleId, menuCode, apiCode);
        if (ruleDoList.isEmpty()) {
            // 尚未配置
            respVO.setCustomRuleList(Collections.emptyList());
            return respVO;
        }

        List<SysDprApiCustomRuleRespVO> customRuleRespVoList = new ArrayList<>(ruleDoList.size());
        respVO.setCustomRuleList(customRuleRespVoList);
        for (SysDprRoleApiRowRuleDO ruleDO : ruleDoList) {
            if (Boolean.TRUE.equals(ruleDO.getDataRange())) {
                // 数据范围
                respVO.setRange(DprSysInternallyEnum.valueOf(ruleDO.getDprSysInternally()));
                continue;
            }
            SysDprApiCustomRuleRespVO vo = new SysDprApiCustomRuleRespVO();
            vo.setId(ruleDO.getId());
            vo.setDprRuleName(ruleDO.getDprRuleName());
            vo.setDprRuleField(ruleDO.getDprRuleField());
            if (StringUtils.hasText(ruleDO.getDprRuleCondition())) {
                vo.setDprRuleCondition(DprRuleConditionEnum.valueOf(ruleDO.getDprRuleCondition()));
            }
            if (StringUtils.hasText(ruleDO.getDprRuleValueType())) {
                vo.setDprRuleValueType(DprRuleValueTypeEnum.valueOf(ruleDO.getDprRuleValueType()));
            }
            vo.setDataSet(ruleDO.getDataSet());
            vo.setDprRuleValue(ruleDO.getDprRuleValue());
            if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS == vo.getDprRuleValueType() && StringUtils.hasText(ruleDO.getDprRuleValue())) {
                vo.setDprRuleValueName(DprSysInternallyEnum.valueOf(ruleDO.getDprRuleValue()).getValueDescription());
            }
            if (StringUtils.hasText(ruleDO.getDprRuleValueName())) {
                vo.setDprRuleValueName(ruleDO.getDprRuleValueName());
            }
            vo.setDprRuleDeclare(ruleDO.getDprRuleDeclare());
            vo.setBs1(ruleDO.getBs1());
            vo.setBs2(ruleDO.getBs2());
            vo.setBs3(ruleDO.getBs3());
            customRuleRespVoList.add(vo);
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForCustom(String code, Long tenantId) {
        var menuTree = tenantMenuTreeRepoProc.getByMenuCode(code, tenantId);
        if (menuTree == null) {
            return this.detailForDefault(code, tenantId);
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(menuTree.getId());
        respVO.setCode(menuTree.getMenuCode());
        respVO.setName(menuTree.getMenuName());
        respVO.setNodeType(menuTree.getNodeType());
        respVO.setParentCode(menuTree.getParentMenuCode());

        if (StringUtils.hasText(menuTree.getParentMenuCode())) {
            // 有父节点
            var menuTreeParent = tenantMenuTreeRepoProc.getByMenuCode(menuTree.getParentMenuCode(), tenantId);
            if (menuTreeParent != null) {
                respVO.setParentId(menuTreeParent.getId());
                respVO.setParentName(menuTreeParent.getMenuName());
            }
        }

        if (StrUtil.isBlank(menuTree.getMenuName())) {
            // 获取名称
            var menuName = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(code));
            respVO.setName(menuName);
            respVO.setApiName(menuName);
        }
        if (StrUtil.isBlank(respVO.getParentName()) && StrUtil.isNotBlank(respVO.getParentCode())) {
            var menuName = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(respVO.getParentCode()));
            respVO.setParentName(menuName);
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForDefault(String code, Long tenantId) {
        var menuBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(code));
        if (menuBO == null) {
            // 没有找到菜单，则尝试获取API
            return this.detailForApi(code);
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(menuBO.getId());
        respVO.setCode(menuBO.getMenusCode());
        respVO.setName(menuBO.getMenusName());

        // 节点类型
        var nodeType = MenuTreeNodeType.valueOfPlatformMenu(menuBO.getNodeType());
        Assert.notNull(nodeType, "不支持的节点类型");
        respVO.setNodeType(nodeType.getValue());

        respVO.setParentCode(menuBO.getMenusParentCode());
        respVO.setAppCode(menuBO.getMenusAppCode());
        respVO.setAction(menuBO.getMenusRoute());

        if (StrUtil.isNotBlank(menuBO.getMenusParentCode())) {
            // 父菜单
            var menuParentBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(menuBO.getMenusParentCode()));
            if (menuParentBO != null) {
                respVO.setParentId(menuParentBO.getId());
                respVO.setParentCode(menuParentBO.getMenusCode());
                respVO.setParentName(menuParentBO.getMenusName());

                var parentNodeType = MenuTreeNodeType.valueOfPlatformMenu(menuParentBO.getNodeType());
                if (parentNodeType == MenuTreeNodeType.MENU) {
                    respVO.setMenuCode(menuParentBO.getMenusCode());
                    respVO.setMenuName(menuParentBO.getMenusName());
                }
            }
        }

        return respVO;
    }

    private PermissionDetailRespVO detailForApi(String code) {
        var apiDo = tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.getByCode(code));
        if (apiDo == null) {
            return null;
        }

        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(apiDo.getId());
        respVO.setCode(apiDo.getPermissonCode());
        respVO.setName(apiDo.getPermissonName());
        respVO.setNodeType(MenuTreeNodeType.API.getValue());

        // 获取菜单
        respVO.setParentCode(apiDo.getMenusCode());
        if (StringUtils.hasText(apiDo.getMenusCode())) {
            var menuBO = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getByMenuCode(apiDo.getMenusCode()));
            if (menuBO != null) {
                respVO.setParentId(menuBO.getId());
                respVO.setParentName(menuBO.getMenusName());

                respVO.setMenuCode(menuBO.getMenusCode());
                respVO.setMenuName(menuBO.getMenusName());
            }
        }

        respVO.setAppCode(apiDo.getAppCode());
        respVO.setAction(apiDo.getPermissonPath());

        return respVO;
    }

    private PermissionDetailRespVO detailForApp(String code) {
        var appDO = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getByCode(code));
        if (appDO == null) {
            return null;
        }
        PermissionDetailRespVO respVO = new PermissionDetailRespVO();
        respVO.setId(appDO.getId());
        respVO.setCode(appDO.getAppCode());
        respVO.setName(appDO.getAppName());
        respVO.setNodeType(MenuTreeNodeType.APP.getValue());

        return respVO;
    }

    private void fillApp(PermissionDetailRespVO respVO) {
        String appCode = respVO.getAppCode();
        // 查询应用编码
        if (StrUtil.isBlank(appCode)) {
            appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> menusRepoProc.getMenuNameByMenuCode(respVO.getCode()));
            if (StrUtil.isBlank(appCode)) {
                return;
            }
            respVO.setAppCode(appCode);
        }

        // 查询应用
        var app = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getByCode(respVO.getAppCode()));
        if (app == null) {
            return;
        }
        respVO.setAppId(app.getId());
        respVO.setAppName(app.getAppName());
    }

    private void fillMenu(PermissionDetailRespVO respVO) {
        if (StrUtil.isNotBlank(respVO.getMenuCode())) {
            return;
        }

        String menuCode = respVO.getParentCode();
        if (StrUtil.isBlank(menuCode)) {
            return;
        }

        // 查询菜单
        var menu = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var temp = menusRepoProc.getByMenuCode(menuCode);
            while (temp != null) {
                var tempType = MenuTreeNodeType.valueOfPlatformMenu(temp.getNodeType());
                if (tempType == MenuTreeNodeType.MENU) {
                    // 已是菜单
                    break;
                }
                if (StrUtil.isBlank(temp.getMenusParentCode())) {
                    // 已无父节点
                    temp = null;
                    break;
                }

                temp = menusRepoProc.getByMenuCode(temp.getMenusParentCode());
            }
            return temp;
        });
        if (menu == null) {
            return;
        }
        respVO.setMenuCode(menu.getMenusCode());
        respVO.setMenuName(menu.getMenusName());
    }
}
