package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.convert.RoleGroupConvert;
import com.elitescloud.cloudt.system.model.vo.query.role.RoleGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleGroupPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleGroupSaveVO;
import com.elitescloud.cloudt.system.service.RoleGroupMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleGroupDO;
import com.elitescloud.cloudt.system.service.repo.RoleGroupRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/11/24
 */
@Service
public class RoleGroupMngServiceImpl implements RoleGroupMngService {

    @Autowired
    private RoleGroupRepoProc roleGroupRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> upsert(RoleGroupSaveVO saveVO) {
        SysRoleGroupDO groupDO = null;
        if (saveVO.getId() == null) {
            if (CharSequenceUtil.isNotBlank(saveVO.getGroupCode()) && roleGroupRepoProc.existsCode(saveVO.getGroupCode())) {
                return ApiResult.fail("分组编码已存在");
            }
            groupDO = RoleGroupConvert.INSTANCE.saveVo2DO(saveVO);
        } else {
            groupDO = roleGroupRepoProc.get(saveVO.getId());
            if (groupDO == null) {
                return ApiResult.fail("数据不存在");
            }
            if (CharSequenceUtil.isNotBlank(saveVO.getGroupCode()) && !saveVO.getGroupCode().equals(groupDO.getGroupCode())) {
                if (roleGroupRepoProc.existsCode(saveVO.getGroupCode())) {
                    return ApiResult.fail("分组编码已存在");
                }
            }
            RoleGroupConvert.INSTANCE.copySaveVO(saveVO, groupDO);
        }

        groupDO.setEnabled(ObjectUtil.defaultIfNull(groupDO.getEnabled(), true));

        // 保存数据
        roleGroupRepoProc.save(groupDO);
        return ApiResult.ok(groupDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }
        if (!roleGroupRepoProc.exists(id)) {
            return ApiResult.ok(id);
        }

        roleGroupRepoProc.delete(id);
        // 删除角色的关联关系
        roleRepoProc.clearGroupId(id);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }
        if (!roleGroupRepoProc.exists(id)) {
            return ApiResult.ok(id);
        }

        roleGroupRepoProc.updateEnabled(id, enabled);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<RoleGroupDetailRespVO> getDetail(Long id) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }

        var roleGroup = roleGroupRepoProc.get(id);
        if (roleGroup == null) {
            return ApiResult.noData();
        }

        var respVO = RoleGroupConvert.INSTANCE.do2DetailRespVO(roleGroup);
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<RoleGroupPageRespVO>> pageMng(RoleGroupPageQueryVO queryVO) {
        var pageData = roleGroupRepoProc.pageMng(queryVO).map(RoleGroupConvert.INSTANCE::do2PageRespVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 统计角色数量
        var ids = pageData.stream().map(RoleGroupPageRespVO::getId).collect(Collectors.toList());
        var countMap = roleRepoProc.countByGroupId(ids);
        if (!countMap.isEmpty()) {
            pageData.each(t -> {
                t.setRoleNum(countMap.getOrDefault(t.getId(), 0L));
            });
        }

        return ApiResult.ok(pageData);
    }
}
