package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformNextNumberDO;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNextNumberRepo;
import com.elitescloud.cloudt.system.convert.SeqNumConvert;
import com.elitescloud.cloudt.system.model.vo.query.sequence.SeqNumPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sequence.SeqNumPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sequence.SeqNumRespVO;
import com.elitescloud.cloudt.system.model.vo.save.sequence.SeqNumSaveVO;
import com.elitescloud.cloudt.system.service.SeqNumMngService;
import com.elitescloud.cloudt.system.service.repo.SeqNextNumRepoProc;
import com.elitescloud.cloudt.system.service.repo.SeqRuleDtlRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/14
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class SeqNumMngServiceImpl extends BaseServiceImpl implements SeqNumMngService {
    private static final SeqNumConvert CONVERT = SeqNumConvert.INSTANCE;

    @Autowired
    private SysPlatformNextNumberRepo nextNumberRepo;
    @Autowired
    private SeqNextNumRepoProc seqNumRepoProc;
    @Autowired
    private SeqRuleDtlRepoProc seqNumRuleDtlRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(SeqNumSaveVO saveVO) {
        var numberDO = saveVO.getId() == null ? checkForAdd(saveVO) :
                checkForUpdate(saveVO);
        nextNumberRepo.save(numberDO);

        return ApiResult.ok(numberDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Boolean enabled = seqNumRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;

        seqNumRepoProc.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        var numberDO = seqNumRepoProc.get(id);
        if (numberDO == null) {
            return ApiResult.noData();
        }
        if (ObjectUtil.defaultIfNull(numberDO.getInternal(), true)) {
            return ApiResult.fail("内置数据不可删除");
        }

        boolean isInUse = seqNumRuleDtlRepoProc.isInUseForNextNum(numberDO.getAppCode(), numberDO.getCode());
        if (isInUse) {
            return ApiResult.fail("该编号正在使用中，请先修改关联的发号规则");
        }

        seqNumRepoProc.delete(id);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<SeqNumRespVO> get(Long id) {
        var seqNum = seqNumRepoProc.get(id);
        if (seqNum == null) {
            return ApiResult.noData();
        }

        var respVO = CONVERT.do2DetailRespVO(seqNum);
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<SeqNumPageRespVO>> pageMng(SeqNumPageQueryVO queryVO) {
        var pageData = seqNumRepoProc.pageMng(queryVO)
                .map(CONVERT::do2MngRespVO);

        return ApiResult.ok(pageData);
    }

    private SysPlatformNextNumberDO checkForAdd(SeqNumSaveVO saveVO) {
        boolean exists = seqNumRepoProc.existsCode(saveVO.getAppCode(), saveVO.getCode());
        Assert.isTrue(!exists, "编号已存在");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        // 默认数据
        SysPlatformNextNumberDO nextNumberDO = CONVERT.saveVo2Do(saveVO);
        nextNumberDO.setVersion(1);
        nextNumberDO.setInternal(false);
        return nextNumberDO;
    }

    private SysPlatformNextNumberDO checkForUpdate(SeqNumSaveVO saveVO) {
        SysPlatformNextNumberDO old = seqNumRepoProc.get(saveVO.getId());
        Assert.notNull(old, "修改的数据不存在");

        saveVO.setAppCode(old.getAppCode());
        if (!CharSequenceUtil.equals(old.getCode(), saveVO.getCode())) {
            boolean exists = seqNumRepoProc.existsCode(saveVO.getAppCode(), saveVO.getCode());
            Assert.isTrue(!exists, "编号已存在");
        }

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(ObjectUtil.defaultIfNull(old.getEnabled(), true));
        }

        CONVERT.copySaveVo2Do(saveVO, old);
        return old;
    }
}
