package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNumberRuleDtlRepo;
import com.elitescloud.cloudt.platform.service.repo.number.SysPlatformNumberRuleRepo;
import com.elitescloud.cloudt.system.convert.SeqRuleConvert;
import com.elitescloud.cloudt.system.factory.seq.SeqNumFactory;
import com.elitescloud.cloudt.system.model.vo.query.sequence.SeqRulePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sequence.SeqRuleDetailsRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sequence.SeqRulePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.sequence.SeqRuleSaveVO;
import com.elitescloud.cloudt.system.service.SeqRuleMngService;
import com.elitescloud.cloudt.system.service.repo.SeqNextNumRepoProc;
import com.elitescloud.cloudt.system.service.repo.SeqRuleDtlRepoProc;
import com.elitescloud.cloudt.system.service.repo.SeqRuleRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/11/14
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class SeqRuleMngServiceImpl extends BaseServiceImpl implements SeqRuleMngService {
    private static final SeqRuleConvert CONVERT = SeqRuleConvert.INSTANCE;

    @Autowired
    private SeqNextNumRepoProc seqNumRepoProc;
    @Autowired
    private SysPlatformNumberRuleRepo numberRuleRepo;
    @Autowired
    private SeqRuleRepoProc seqRuleRepoProc;
    @Autowired
    private SysPlatformNumberRuleDtlRepo numberRuleDtlRepo;
    @Autowired
    private SeqRuleDtlRepoProc seqRuleDtlRepoProc;

    @Autowired
    private SeqNumFactory seqNumFactory;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> update(Long id, SeqRuleSaveVO saveVO) {
        // 保存规则
        var ruleDO = seqRuleRepoProc.get(id);
        if (ruleDO == null) {
            return ApiResult.noData();
        }
        Assert.notEmpty(saveVO.getDetailsList(), "规则明细列表为空");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }
        CONVERT.copySaveVo(saveVO, ruleDO);
        numberRuleRepo.save(ruleDO);

        // 更新规则明细
        seqRuleDtlRepoProc.deleteByRuleId(id);
        AtomicInteger sortNo = new AtomicInteger(1);
        var detailDoList = saveVO.getDetailsList().stream().map(t -> {
            var detailDO = CONVERT.saveVo2Do(t);
            detailDO.setRuleId(ruleDO.getId());
            detailDO.setAppCode(ruleDO.getAppCode());
            if (t.getSeq() == null) {
                detailDO.setSeq(sortNo.getAndIncrement());
            }

            if (CharSequenceUtil.equals(detailDO.getNumberType(), SysNumType.NN.name())) {
                Assert.isTrue(detailDO.getNnLen() != null && detailDO.getNnLen() > 0, "取号类型是[下一编号]时自增序号宽度不能小于0");
                // 判断自增序号是否存在
                var exists = seqNumRepoProc.existsCode(detailDO.getAppCode(), detailDO.getNumberPattern());
                Assert.isTrue(exists, "选择的下一编号不存在");
            }

            return detailDO;
        }).collect(Collectors.toList());
        numberRuleDtlRepo.saveAll(detailDoList);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Boolean enabled = seqRuleRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;

        seqRuleRepoProc.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<SeqRuleDetailsRespVO> get(Long id) {
        var ruleDO = seqRuleRepoProc.get(id);
        if (ruleDO == null) {
            return ApiResult.noData();
        }

        var ruleRespVO = CONVERT.do2DetailRespVO(ruleDO);

        // 查询规则明细
        var ruleDtlDoList = seqRuleDtlRepoProc.queryByRuleId(id);
        var ruleDtlVoList = ruleDtlDoList.stream()
                .map(t -> {
                    var vo = CONVERT.do2RespVO(t);
                    vo.setNumberTypeName(this.convertNumberTypeName(t.getNumberType()));
                    return vo;
                })
                .collect(Collectors.toList());

        ruleRespVO.setDetailsList(ruleDtlVoList);

        return ApiResult.ok(ruleRespVO);
    }

    @Override
    public ApiResult<PagingVO<SeqRulePageRespVO>> pageMng(SeqRulePageQueryVO queryVO) {
        var pageData = seqRuleRepoProc.pageMng(queryVO)
                .map(CONVERT::do2MngRespVO);
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<String> getSample(Long id, List<String> runtimeValues) {
        String sampleCode = seqNumFactory.generateSampleCode(id, runtimeValues);
        return ApiResult.ok(sampleCode);
    }

    private String convertNumberTypeName(String numberType) {
        var type = SysNumType.parse(numberType);
        return type == null ? null : type.getDescription();
    }
}
