package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.SysAlertType;
import com.elitescloud.cloudt.system.model.entity.SysAlertConfigDO;
import com.elitescloud.cloudt.system.model.entity.SysAlertDO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertConfigBaseRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertConfigWxWorkRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertConfigBaseSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertSaveVO;
import com.elitescloud.cloudt.system.service.SysAlertMngService;
import com.elitescloud.cloudt.system.service.SysAlertService;
import com.elitescloud.cloudt.system.service.alert.AlertProvider;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/26
 */
@Slf4j
@Service
public class SysAlertMngServiceImpl extends BaseAlertServiceImpl implements SysAlertMngService {

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveAlert(AlertSaveVO saveVO) {
        SysAlertDO alertDO = null;
        try {
            alertDO = this.checkForSave(saveVO);
        } catch (Exception e) {
            log.info("保存预警配置失败：{}", e.getMessage());
            return ApiResult.fail("保存失败，" + e.getMessage());
        }
        alertRepoProc.save(alertDO);

        // 企业微信配置
        SysAlertConfigDO config = this.convertConfigDO(saveVO.getWxWorkSaveVO(), saveVO.getBusinessType(), SysAlertType.WX_WORK);
        if (config != null) {
            alertConfigRepoProc.save(config);
        }

        return ApiResult.ok(alertDO.getId());
    }

    @Override
    public ApiResult<AlertRespVO> getAlert(String businessType) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return ApiResult.fail("业务类型为空");
        }
        SysAlertDO alertDO = alertRepoProc.getByBusinessType(businessType);
        if (alertDO == null) {
            return ApiResult.ok(null);
        }

        Map<String, SysAlertConfigDO> configMap = alertConfigRepoProc.getByBusinessType(alertDO.getBusinessType())
                .stream().collect(Collectors.toMap(SysAlertConfigDO::getAlertType, t -> t, (t1, t2) -> t1));

        AlertRespVO respVO = new AlertRespVO();
        respVO.setId(alertDO.getId());
        respVO.setBusinessType(alertDO.getBusinessType());
        respVO.setEnabled(alertDO.getEnabled());
        respVO.setWxWorkRespVO((AlertConfigWxWorkRespVO) this.convertWxWorkConfig(configMap.get(SysAlertType.WX_WORK.name()), SysAlertType.WX_WORK));
        respVO.setDescription(alertDO.getDescription());

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<CodeNameParam>> getTmplParam(String businessType) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return ApiResult.fail("业务类型为空");
        }

        List<CodeNameParam> codeNameParams = new ArrayList<>(32);
        switch (businessType) {
            // 三方接口API
            case SysAlertService.BUSINESS_TYPE_THIRD_API:
                codeNameParams.add(new CodeNameParam("{businessType}", "业务类型"));
                codeNameParams.add(new CodeNameParam("{direction}", "调用方向"));
                codeNameParams.add(new CodeNameParam("{businessKey}", "关键参数"));
                codeNameParams.add(new CodeNameParam("{reqTime}", "请求时间"));
                codeNameParams.add(new CodeNameParam("{respTime}", "响应时间"));
                codeNameParams.add(new CodeNameParam("{sourceApp}", "源系统"));
                codeNameParams.add(new CodeNameParam("{targetApp}", "目标系统"));
                codeNameParams.add(new CodeNameParam("{url}", "接口地址"));
                codeNameParams.add(new CodeNameParam("{failMsg}", "失败信息"));
                break;
            // SQL执行
            case SysAlertService.BUSINESS_TYPE_SQL_EXECUTE:
                codeNameParams.add(new CodeNameParam("{success}", "执行状态"));
                codeNameParams.add(new CodeNameParam("{failMsg}", "失败信息"));
                codeNameParams.add(new CodeNameParam("{result}", "检查结果"));
                codeNameParams.add(new CodeNameParam("{rowNum}", "记录数量"));
                codeNameParams.add(new CodeNameParam("{operateType}", "操作类型"));
                codeNameParams.add(new CodeNameParam("{sql}", "执行SQL"));
                codeNameParams.add(new CodeNameParam("{sqlName}", "SQL名称"));
                codeNameParams.add(new CodeNameParam("{sqlCode}", "SQL编码"));
                codeNameParams.add(new CodeNameParam("{category}", "分类"));
                codeNameParams.add(new CodeNameParam("{dsName}", "数据源名称"));
                codeNameParams.add(new CodeNameParam("{startTime}", "开始时间"));
                codeNameParams.add(new CodeNameParam("{finishTime}", "结束时间"));
                codeNameParams.add(new CodeNameParam("{costTime}", "耗时"));
                break;
            default:
                return ApiResult.fail("暂不支持");
        }
        return ApiResult.ok(codeNameParams);
    }

    private SysAlertDO checkForSave(AlertSaveVO saveVO) {
        SysAlertDO alertDO = alertRepoProc.getByBusinessType(saveVO.getBusinessType());
        if (saveVO.getId() == null) {
            Assert.isNull(alertDO, "预警类型已存在");

            alertDO = new SysAlertDO();
            alertDO.setBusinessType(saveVO.getBusinessType());
        } else {
            Assert.isTrue(alertDO != null && alertDO.getId().equals(saveVO.getId()), "预警类型不能修改");
        }
        Objects.requireNonNull(alertDO);

        alertDO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), true));
        alertDO.setDescription(saveVO.getDescription());

        return alertDO;
    }

    private SysAlertConfigDO convertConfigDO(AlertConfigBaseSaveVO wxWorkSaveVO, String businessType, SysAlertType alertType) {
        if (wxWorkSaveVO == null) {
            return null;
        }

        SysAlertConfigDO configDO = alertConfigRepoProc.getByBusinessType(businessType, alertType.name());
        if (wxWorkSaveVO.getId() == null) {
            Assert.isNull(configDO, "企业微信配置已存在");

            configDO = new SysAlertConfigDO();
            configDO.setBusinessType(businessType);
            configDO.setAlertType(alertType.name());
        }
        configDO.setEnabled(ObjectUtil.defaultIfNull(wxWorkSaveVO.getEnabled(), true));
        configDO.setDescription(wxWorkSaveVO.getDescription());

        AlertProvider alertProvider = super.getAlertProvider(alertType);
        Assert.notNull(alertProvider, "暂不支持：" + alertType.name());

        configDO.setCfgJson(alertProvider.toString(wxWorkSaveVO));

        return configDO;
    }

    private AlertConfigBaseRespVO convertWxWorkConfig(SysAlertConfigDO configDO, SysAlertType alertType) {
        if (configDO == null) {
            return null;
        }

        AlertProvider alertProvider = super.getAlertProvider(alertType);
        Assert.notNull(alertProvider, "暂不支持：" + alertType.name());
        var respVO = alertProvider.parse(configDO.getCfgJson());

        respVO.setId(configDO.getId());
        respVO.setEnabled(configDO.getEnabled());
        respVO.setDescription(configDO.getDescription());

        return respVO;
    }
}
