package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.SysAlertType;
import com.elitescloud.cloudt.system.model.entity.SysAlertConfigDO;
import com.elitescloud.cloudt.system.service.SysAlertService;
import com.elitescloud.cloudt.system.service.alert.AlertProvider;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.function.BiConsumer;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/26
 */
@Slf4j
@Component
public class SysAlertServiceImpl extends BaseAlertServiceImpl implements SysAlertService {

    @Override
    public ApiResult<Boolean> sendAlertByTmpl(String businessType, String category, Map<String, Object> tmplParams) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return ApiResult.fail("业务类型为空");
        }
        if (CollUtil.isEmpty(tmplParams)) {
            return ApiResult.fail("模板参数为空");
        }

        this.send(businessType, (cfg, alertProvider) -> alertProvider.sendByTmpl(cfg, tmplParams, category));

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> sendAlert(String businessType, String category, String content) {
        if (CharSequenceUtil.isBlank(businessType)) {
            return ApiResult.fail("业务类型为空");
        }
        if (CharSequenceUtil.isBlank(content)) {
            return ApiResult.fail("发送内容为空");
        }

        this.send(businessType, (cfg, alertProvider) -> alertProvider.send(cfg, content, category));

        return ApiResult.ok(true);
    }

    private void send(String businessType, BiConsumer<String, AlertProvider> consumer) {
        var alertDO = alertRepoProc.getByBusinessType(businessType);
        Assert.notNull(alertDO, "预警未配置");
        Assert.isTrue(Boolean.TRUE.equals(alertDO.getEnabled()), "未启用预警配置");

        var configList = alertConfigRepoProc.getByBusinessType(businessType);
        Assert.notEmpty(configList, "未配置有效的预警途径");

        SysAlertType alertType = null;
        AlertProvider alertProvider = null;
        for (SysAlertConfigDO configDO : configList) {
            if (Boolean.FALSE.equals(configDO.getEnabled())) {
                log.info("未启用预警：{}，{}", businessType, configDO.getAlertType());
                continue;
            }
            alertType = SysAlertType.parse(configDO.getAlertType());
            Assert.notNull(alertType, "未知预警类型：" + configDO.getAlertType());
            alertProvider = super.getAlertProvider(alertType);
            Assert.notNull(alertProvider, "不支持的预警类型：" + configDO.getAlertType());

            try {
                consumer.accept(configDO.getCfgJson(), alertProvider);
            } catch (Exception e) {
                log.error("发送预警异常：", e);
            }
        }
    }
}
