package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysNumType;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.entity.*;
import com.elitescloud.cloudt.system.convert.UdcConvert;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.model.bo.*;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/12
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class SysTenantBasicDataServiceImpl extends BaseServiceImpl implements SysTenantBasicDataService {
    private static final UdcConvert CONVERT_UDC = UdcConvert.INSTANCE;

    @Autowired
    private UdcRepoProc udcRepoProc;
    @Autowired
    private UdcValueRepoProc udcValueRepoProc;
    @Autowired
    private SeqNextNumRepoProc nextNumRepoProc;
    @Autowired
    private SeqRuleRepoProc seqRuleRepoProc;
    @Autowired
    private SeqRuleDtlRepoProc seqRuleDtlRepoProc;
    @Autowired
    private SysTenantAppRepoProc tenantAppRepoProc;

    @Override
    public ApiResult<Boolean> syncUdc(Long tenantId) {
        var tenant = tenantClientProvider.getTenant(tenantId);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }
        var appCodes = tenantAppRepoProc.getAppCode(tenantId);
        if (appCodes.isEmpty()) {
            return ApiResult.fail("租户尚未绑定应用");
        }

        for (String appCode : appCodes) {
            int page = 0, pageSize = 20;
            log.info("向租户【{}】同步UDC【{}】...", tenantId, appCode);
            var pageUdc = queryUdc(page, pageSize, appCode);
            while (!pageUdc.isEmpty()) {
                // 向租户同步
                syncUdcToTenant(tenant, appCode, pageUdc.getRecords());

                page++;
                pageUdc = queryUdc(page, pageSize, appCode);
            }
        }

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> syncUdc(String appCode, String udcCode) {
        if (CharSequenceUtil.hasBlank(appCode, udcCode)) {
            return ApiResult.fail("参数不能为空");
        }

        var udcBO = udcRepoProc.getBoByAppCodeAndUdcCode(appCode, udcCode);
        var udcValueBoList = udcValueRepoProc.listBoByUdc(appCode, udcCode, true);
        if (udcBO == null || CollUtil.isEmpty(udcValueBoList)) {
            // 删掉UDC
            CompletableFuture.runAsync(() -> tenantDataIsolateProvider.byAllTenant(() -> {
                executeDeleteUdc(appCode, udcCode);

                // 删掉缓存
                udcProvider.clearCache(appCode, udcCode);
                return null;
            }));
            return ApiResult.ok(true);
        }
        udcBO.setValueList(udcValueBoList);
        CompletableFuture.runAsync(() -> tenantDataIsolateProvider.byAllTenant(() -> {
            executeSyncUdc(appCode, List.of(udcBO));
            // 删掉缓存
            udcProvider.clearCache(appCode, udcCode);
            return null;
        }));

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> syncSequence(Long tenantId) {
        var tenant = tenantClientProvider.getTenant(tenantId);
        if (tenant == null) {
            return ApiResult.fail("租户不存在");
        }
        var appCodes = tenantAppRepoProc.getAppCode(tenantId);
        if (appCodes.isEmpty()) {
            return ApiResult.fail("租户尚未绑定应用");
        }

        for (String appCode : appCodes) {
            int page = 0, pageSize = 20;
            log.info("向租户【{}】同步下一编号【{}】...", tenantId, appCode);
            var pageNextNum = queryNextNum(page, pageSize, appCode);
            while (!pageNextNum.isEmpty()) {
                // 向租户同步
                syncNextNumToTenant(tenant, appCode, pageNextNum.getRecords());

                page++;
                pageNextNum = queryNextNum(page, pageSize, appCode);
            }

            log.info("向租户【{}】同步发号规则【{}】...", tenantId, appCode);
            page = 0;
            var pageRule = querySeqRule(page, pageSize, appCode);
            while (!pageRule.isEmpty()) {
                // 向租户同步
                syncSeqRuleToTenant(tenant, appCode, pageRule.getRecords());

                page++;
                pageRule = querySeqRule(page, pageSize, appCode);
            }
        }

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> syncSequence(String appCode, String ruleCode) {
        if (CharSequenceUtil.hasBlank(appCode, ruleCode)) {
            return ApiResult.fail("参数不能为空");
        }

        var rule = seqRuleRepoProc.getBo(appCode, ruleCode);
        if (rule == null) {
            // 不存在，则删除
            CompletableFuture.runAsync(() -> tenantDataIsolateProvider.byAllTenant(() -> {
                executeDeleteSeqRule(appCode, ruleCode);
                return null;
            }));
            return ApiResult.ok(true);
        }

        // 查询规则明细
        var detailList = seqRuleDtlRepoProc.queryBoByRuleId(rule.getId());
        rule.setRuleDtlList(detailList);
        var nextNumberCodes = detailList.stream()
                .filter(t -> CharSequenceUtil.equals(t.getNumberType(), SysNumType.NN.name()))
                .map(SysSeqRuleDtlBO::getNumberPattern)
                .filter(StringUtils::hasText)
                .collect(Collectors.toSet());
        rule.setNextNumberList(nextNumRepoProc.getBoByCode(appCode, nextNumberCodes));
        // 开始同步
        CompletableFuture.runAsync(() -> tenantDataIsolateProvider.byAllTenant(() -> {
            executeSyncSeqRule(rule);
            return null;
        }));

        return ApiResult.ok(true);
    }

    private PagingVO<SysSeqNextNumberBO> queryNextNum(Integer page, Integer pageSize, String appCode) {
        return nextNumRepoProc.pageBo(page, pageSize, appCode);
    }

    private void syncNextNumToTenant(SysTenantDTO tenant, String appCode, List<SysSeqNextNumberBO> nextNumberSystem) {
        tenantDataIsolateProvider.byTenantDirectly(() -> {
            // 查询已存在的编码
            var existsCodes = nextNumRepoProc.allCodes(appCode);
            // 过滤出需要新增的
            var nextNumberDoList = nextNumberSystem.stream()
                    .filter(t -> !existsCodes.contains(t.getCode()))
                    .map(t -> {
                        SysPlatformNextNumberDO nextNumberDO = new SysPlatformNextNumberDO();
                        nextNumberDO.setAppCode(t.getAppCode());
                        nextNumberDO.setCode(t.getCode());
                        nextNumberDO.setName(t.getName());
                        nextNumberDO.setStep(t.getStep());
                        nextNumberDO.setNextNumber(1L);
                        nextNumberDO.setNnPeriod(t.getNnPeriod());
                        nextNumberDO.setVersion(1);
                        nextNumberDO.setEnabled(ObjectUtil.defaultIfNull(t.getEnabled(), true));
                        nextNumberDO.setInternal(true);
                        nextNumberDO.setRemark(t.getRemark());

                        return nextNumberDO;
                    }).collect(Collectors.toList());
            if (!nextNumberDoList.isEmpty()) {
                nextNumRepoProc.save(nextNumberDoList);
            }

            return null;
        }, tenant);
    }

    private PagingVO<SysSeqRuleBO> querySeqRule(Integer page, Integer pageSize, String appCode) {
        var pageRule = seqRuleRepoProc.pageBo(page, pageSize, appCode);
        if (pageRule.isEmpty()) {
            return pageRule;
        }

        // 查询规则明细
        var ruleIds = pageRule.getRecords().stream()
                .map(SysSeqRuleBO::getId)
                .collect(Collectors.toSet());
        var ruleDtlMap = seqRuleDtlRepoProc.listBo(ruleIds);
        pageRule.each(t -> t.setRuleDtlList(ObjectUtil.defaultIfNull(ruleDtlMap.get(t.getId()), Collections.emptyList())));

        return pageRule;
    }

    private void syncSeqRuleToTenant(SysTenantDTO tenant, String appCode, List<SysSeqRuleBO> ruleSystem) {
        tenantDataIsolateProvider.byTenantDirectly(() -> {
            // 先同步规则
            var existsCodes = seqRuleRepoProc.allCodes(appCode);

            // 过滤出未同步的
            for (SysSeqRuleBO sysSeqRuleBO : ruleSystem) {
                if (existsCodes.contains(sysSeqRuleBO.getRuleCode()) || CollUtil.isEmpty(sysSeqRuleBO.getRuleDtlList())) {
                    continue;
                }

                // 保存规则
                SysPlatformNumberRuleDO numberRuleDO = new SysPlatformNumberRuleDO();
                numberRuleDO.setAppCode(sysSeqRuleBO.getAppCode());
                numberRuleDO.setRuleCode(sysSeqRuleBO.getRuleCode());
                numberRuleDO.setRuleName(sysSeqRuleBO.getRuleName());
                numberRuleDO.setSampleCode(sysSeqRuleBO.getSampleCode());
                numberRuleDO.setEnabled(ObjectUtil.defaultIfNull(sysSeqRuleBO.getEnabled(), true));
                numberRuleDO.setRemark(sysSeqRuleBO.getRemark());
                numberRuleDO.setTenantCustom(false);
                seqRuleRepoProc.save(numberRuleDO);

                // 保存规则明细
                AtomicInteger seq = new AtomicInteger(1);
                var numberRuleDtlList = sysSeqRuleBO.getRuleDtlList().stream()
                        .map(t -> {
                            SysPlatformNumberRuleDtlDO ruleDtlDO = new SysPlatformNumberRuleDtlDO();
                            ruleDtlDO.setAppCode(t.getAppCode());
                            ruleDtlDO.setRuleId(numberRuleDO.getId());
                            ruleDtlDO.setSeq(ObjectUtil.defaultIfNull(t.getSeq(), seq.getAndIncrement()));
                            ruleDtlDO.setNumberType(t.getNumberType());
                            ruleDtlDO.setNumberPattern(t.getNumberPattern());
                            ruleDtlDO.setNnLen(t.getNnLen());
                            ruleDtlDO.setRemark(t.getRemark());

                            return ruleDtlDO;
                        }).collect(Collectors.toList());
                seqRuleDtlRepoProc.save(numberRuleDtlList);
            }

            return null;
        }, tenant);
    }

    private void executeSyncSeqRule(SysSeqRuleBO seqRuleBO) {
        var ruleDO = seqRuleRepoProc.get(seqRuleBO.getAppCode(), seqRuleBO.getRuleCode());
        if (ruleDO == null) {
            ruleDO = new SysPlatformNumberRuleDO();
            ruleDO.setAppCode(seqRuleBO.getAppCode());
            ruleDO.setRuleCode(seqRuleBO.getRuleCode());
            ruleDO.setRuleName(seqRuleBO.getRuleName());
            ruleDO.setSampleCode(seqRuleBO.getSampleCode());
            ruleDO.setEnabled(ObjectUtil.defaultIfNull(seqRuleBO.getEnabled(), true));
            ruleDO.setRemark(seqRuleBO.getRemark());
            ruleDO.setTenantCustom(false);
            seqRuleRepoProc.save(ruleDO);
        }
        if (Boolean.TRUE.equals(ruleDO.getTenantCustom())) {
            // 租户自定义的跳过
            return;
        }
        ruleDO.setRuleName(seqRuleBO.getRuleName());
        ruleDO.setSampleCode(seqRuleBO.getSampleCode());
        ruleDO.setEnabled(seqRuleBO.getEnabled());
        seqRuleRepoProc.save(ruleDO);

        // 同步明细
        AtomicInteger seq = new AtomicInteger(1);
        seqRuleDtlRepoProc.deleteByRuleId(ruleDO.getId());
        SysPlatformNumberRuleDO finalRuleDO = ruleDO;
        var numberRuleDtlList = seqRuleBO.getRuleDtlList().stream()
                .map(t -> {
                    SysPlatformNumberRuleDtlDO ruleDtlDO = new SysPlatformNumberRuleDtlDO();
                    ruleDtlDO.setAppCode(t.getAppCode());
                    ruleDtlDO.setRuleId(finalRuleDO.getId());
                    ruleDtlDO.setSeq(ObjectUtil.defaultIfNull(t.getSeq(), seq.getAndIncrement()));
                    ruleDtlDO.setNumberType(t.getNumberType());
                    ruleDtlDO.setNumberPattern(t.getNumberPattern());
                    ruleDtlDO.setNnLen(t.getNnLen());
                    ruleDtlDO.setRemark(t.getRemark());

                    return ruleDtlDO;
                }).collect(Collectors.toList());
        seqRuleDtlRepoProc.save(numberRuleDtlList);

        // 同步下一编号
        if (CollUtil.isNotEmpty(seqRuleBO.getNextNumberList())) {
            var nextNumberCodes = seqRuleBO.getNextNumberList().stream().map(SysSeqNextNumberBO::getCode).collect(Collectors.toSet());
            var nextNumberMap = nextNumRepoProc.getByCode(seqRuleBO.getAppCode(), nextNumberCodes)
                    .stream().collect(Collectors.toMap(SysPlatformNextNumberDO::getCode, t -> t, (t1, t2) -> t1));

            List<SysPlatformNextNumberDO> nextNumberList = new ArrayList<>();
            for (SysSeqNextNumberBO sysSeqNextNumberBO : seqRuleBO.getNextNumberList()) {
                SysPlatformNextNumberDO nextNumberDO = nextNumberMap.get(sysSeqNextNumberBO.getCode());
                if (nextNumberDO == null) {
                    nextNumberDO = new SysPlatformNextNumberDO();
                    nextNumberDO.setAppCode(sysSeqNextNumberBO.getAppCode());
                    nextNumberDO.setCode(sysSeqNextNumberBO.getCode());
                    nextNumberDO.setNextNumber(1L);
                    nextNumberDO.setVersion(1);
                }
                nextNumberDO.setName(sysSeqNextNumberBO.getName());
                nextNumberDO.setNnPeriod(sysSeqNextNumberBO.getNnPeriod());
                nextNumberDO.setStep(sysSeqNextNumberBO.getStep());
                nextNumberDO.setEnabled(ObjectUtil.defaultIfNull(sysSeqNextNumberBO.getEnabled(), true));
                nextNumberDO.setRemark(sysSeqNextNumberBO.getRemark());
                nextNumberDO.setInternal(true);

                nextNumberList.add(nextNumberDO);
            }
            nextNumRepoProc.save(nextNumberList);
        }
    }

    private void executeDeleteSeqRule(String appCode, String ruleCode) {
        var ruleId = seqRuleRepoProc.getIdByRuleCode(appCode, ruleCode);
        if (ruleId == null) {
            return;
        }
        seqRuleDtlRepoProc.deleteByRuleId(ruleId);
        seqRuleRepoProc.delete(ruleId);
    }

    private PagingVO<SysUdcBO> queryUdc(Integer page, Integer pageSize, String appCode) {
        var pageUdc = udcRepoProc.pageAll(page, pageSize, appCode);
        if (pageUdc.isEmpty()) {
            return pageUdc;
        }

        // 查询UDC值
        var udcCodes = pageUdc.getRecords().stream()
                .map(SysUdcBO::getUdcCode)
                .filter(StringUtils::hasText)
                .collect(Collectors.toSet());
        var udcValueMap = udcValueRepoProc.queryBoByUdc(appCode, udcCodes, true)
                .stream()
                .collect(Collectors.groupingBy(SysUdcValueBO::getUdcCode));
        pageUdc.each(t -> t.setValueList(ObjectUtil.defaultIfNull(udcValueMap.get(t.getUdcCode()), Collections.emptyList())));

        return pageUdc;
    }

    private void syncUdcToTenant(SysTenantDTO tenant, String appCode, List<SysUdcBO> udcSystem) {
        tenantDataIsolateProvider.byTenantDirectly(() -> {
            // 开始同步
            executeSyncUdc(appCode, udcSystem);
            // 清理缓存
            udcProvider.clearCache(appCode, null);
            return null;
        }, tenant);
    }

    private void executeDeleteUdc(String appCode, String udcCode) {
        udcRepoProc.delete(appCode, udcCode);
        udcValueRepoProc.delete(appCode, udcCode);
    }

    private void executeSyncUdc(String appCode, List<SysUdcBO> udcSystem) {
        var udcCodes = udcSystem.stream().map(SysUdcBO::getUdcCode).collect(Collectors.toSet());
        // 查询已存在的
        var existsUdcMap = udcRepoProc.queryByUdcCode(appCode, udcCodes).stream()
                .collect(Collectors.toMap(SysPlatformUdcDO::getUdcCode, t -> t, (t1, t2) -> t1));
        var existsUdcValueMap = udcValueRepoProc.queryByUdc(appCode, udcCodes, true)
                .stream().collect(Collectors.groupingBy(SysPlatformUdcValueDO::getUdcCode));

        List<SysPlatformUdcDO> udcList = new ArrayList<>(udcCodes.size());
        List<SysPlatformUdcValueDO> udcValueList = new ArrayList<>(128);
        Set<Long> udcValueIdToDel = new HashSet<>(128);
        for (SysUdcBO sysUdcBO : udcSystem) {
            // UDC
            SysPlatformUdcDO udcDO = existsUdcMap.get(sysUdcBO.getUdcCode());
            if (udcDO == null) {
                // 不存在，则新增
                udcDO = new SysPlatformUdcDO();
            }
            var udcId = udcDO.getId();
            var udcName = udcDO.getUdcName();
            CONVERT_UDC.copyUdcBO(sysUdcBO, udcDO);
            udcDO.setId(udcId);
            if (CharSequenceUtil.isNotBlank(udcName)) {
                udcDO.setUdcName(udcName);
            }
            udcList.add(udcDO);

            // UDC值
            var valueList = copyUdcValue(sysUdcBO.getValueList(), existsUdcValueMap.getOrDefault(sysUdcBO.getUdcCode().trim(), new ArrayList<>(16)));
            udcValueList.addAll(valueList);

            // 需要删除的UDC值
            udcValueIdToDel.addAll(filterUdcValueIdToDel(sysUdcBO.getValueList(), valueList));
        }

        // 保存UDC
        udcRepoProc.save(udcList);
        // 保存UDC值
        udcValueRepoProc.save(udcValueList);
        // 需要删除的UDC值
        if (!udcValueIdToDel.isEmpty()) {
            udcValueRepoProc.delete(udcValueIdToDel);
        }
    }

    private List<SysPlatformUdcValueDO> copyUdcValue(List<SysUdcValueBO> valueBOS, List<SysPlatformUdcValueDO> valueDOS) {
        var valueMap = valueDOS.stream()
                .collect(Collectors.toMap(t -> t.getUdcValueCode().trim(), t -> t, (t1, t2) -> t1));
        List<SysPlatformUdcValueDO> valueList = new ArrayList<>(valueDOS.size());
        Set<String> existsCodes = new HashSet<>();
        for (SysUdcValueBO valueBO : valueBOS) {
            SysPlatformUdcValueDO valueDO = valueMap.get(valueBO.getUdcValueCode().trim());
            if (valueDO == null) {
                // 没有则新增
                valueDO = new SysPlatformUdcValueDO();
                CONVERT_UDC.copyUdcValueBO(valueBO, valueDO);
                valueDO.setAllowDefault(true);
            } else {
                // 修改值
                valueDO.setUdcValueName(valueBO.getUdcValueName());
                valueDO.setUdcOrder(ObjectUtil.defaultIfNull(valueBO.getUdcOrder(), 1));
                valueDO.setAllowStart(ObjectUtil.defaultIfNull(valueDO.getAllowStart(), valueBO.getAllowStart()));
                valueDO.setUdcValueDescribe(CharSequenceUtil.blankToDefault(valueDO.getUdcValueDescribe(), valueBO.getUdcValueDescribe()));
                valueDO.setAllowDefault(true);
                if (StringUtils.hasText(valueBO.getParentUdcValueCode())) {
                    valueDO.setParentUdcValueCode(valueBO.getParentUdcValueCode());
                }
                valueDO.setValueAliasName(CharSequenceUtil.blankToDefault(valueDO.getValueAliasName(), valueBO.getValueAliasName()));
            }
            valueList.add(valueDO);

            existsCodes.add(valueDO.getUdcValueCode());
        }

        // 需要删除的
        for (SysPlatformUdcValueDO valueDO : valueDOS) {
            if (valueDO.getAllowDefault() == null || !valueDO.getAllowDefault()) {
                continue;
            }
            if (existsCodes.contains(valueDO.getUdcValueCode())) {
                continue;
            }
            valueList.add(valueDO);
        }

        return valueList;
    }

    private Set<Long> filterUdcValueIdToDel(List<SysUdcValueBO> valueBOS, List<SysPlatformUdcValueDO> valueDOS) {
        var valueCodes = valueBOS.stream().map(SysUdcValueBO::getUdcValueCode).collect(Collectors.toSet());
        Set<Long> toDel = new HashSet<>(8);
        for (SysPlatformUdcValueDO valueDO : valueDOS) {
            if (!BooleanUtil.isTrue(valueDO.getAllowDefault())) {
                // 非系统内置
                continue;
            }
            if (!valueCodes.contains(valueDO.getUdcValueCode())) {
                toDel.add(valueDO.getId());
            }
        }
        return toDel;
    }
}
