package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.SysVersionConvert;
import com.elitescloud.cloudt.system.model.entity.SysVersionDO;
import com.elitescloud.cloudt.system.model.vo.query.version.SysVersionPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.version.SysVersionRespVO;
import com.elitescloud.cloudt.system.model.vo.save.version.SysVersionChangeStatusVO;
import com.elitescloud.cloudt.system.model.vo.save.version.SysVersionSaveVO;
import com.elitescloud.cloudt.system.service.SysVersionService;
import com.elitescloud.cloudt.system.service.repo.SysVersionRepoProc;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;

@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysVersionServiceImpl implements SysVersionService {
    private static final String VERSION_KEY = "cloudt:system:std_version";

    @Autowired
    private SysVersionRepoProc sysVersionRepoProc;

    @Autowired
    private RedisUtils redisUtils;
    private final Cache<String, String> localCache;

    public SysVersionServiceImpl() {
        localCache = Caffeine.newBuilder()
                .maximumSize(5)
                .expireAfterWrite(Duration.ofMinutes(20))
                .build();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Object> save(SysVersionSaveVO save) {
        //查询是否有已经启用的数据
        this.checkVersionStatus(save.getId(), save.getStatus(), save.getVersionNo());
        SysVersionDO sysVersion = this.excuteSaveVersion(save);
        sysVersionRepoProc.saveVersion(sysVersion);

        this.saveCacke(sysVersionRepoProc.getActiveVersion());
        return ApiResult.ok();
    }

    @Override
    public PagingVO<SysVersionRespVO> page(SysVersionPageQueryVO query) {
        return sysVersionRepoProc.page(query);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void del(Long id) {
        sysVersionRepoProc.del(id);

        this.saveCacke(sysVersionRepoProc.getActiveVersion());
    }


    public void changeStatus(SysVersionChangeStatusVO changeStatusVO) {
        this.checkVersionStatus(changeStatusVO.getId(), changeStatusVO.getStatus(), null);
        SysVersionDO sysVersion = sysVersionRepoProc.getById(changeStatusVO.getId());
        sysVersion.setStatus(changeStatusVO.getStatus());
        sysVersionRepoProc.saveVersion(sysVersion);

        this.saveCacke(sysVersionRepoProc.getActiveVersion());
    }

    @Override
    public ApiResult<SysVersionRespVO> getCurrentVersionDetail() {
        SysVersionRespVO resp = sysVersionRepoProc.getActiveVersionDetail();
        return ApiResult.ok(resp);
    }

    @Override
    public ApiResult<String> findActiveVersion() {
        // 先从本地缓存获取
        var version = localCache.getIfPresent(VERSION_KEY);
        if (CharSequenceUtil.isNotBlank(version)) {
            return ApiResult.ok(version);
        }

        // 从redis获取
        version = (String) redisUtils.get(VERSION_KEY);
        if (CharSequenceUtil.isNotBlank(version)) {
            localCache.put(VERSION_KEY, version);
            return ApiResult.ok(version);
        }

        // 从数据库查询
        version = sysVersionRepoProc.getActiveVersion();

        if (version == null) {
            version = "0";
        }
        saveCacke(version);

        return ApiResult.ok(version);
    }

    private void saveCacke(String version) {
        if (CharSequenceUtil.isBlank(version)) {
            redisUtils.set(VERSION_KEY, "0");
            localCache.put(VERSION_KEY, "0");
            return;
        }

        redisUtils.set(VERSION_KEY, version);
        localCache.put(VERSION_KEY, version);
    }

    private void checkVersionStatus(Long id, Boolean status, String versionNo) {
        if (Boolean.TRUE.equals(status)) {
            Long count = sysVersionRepoProc.getStatusCount(id, status);
            if (count > 0) {
                throw new BusinessException("版本管理只能有一个启用");
            }
        }
        if (StringUtils.isNotBlank(versionNo)) {
            Long versionCount = sysVersionRepoProc.getVersionNoCount(id, versionNo);
            if (versionCount > 0) {
                throw new BusinessException("版本号已存在");
            }
        }
    }

    private SysVersionDO excuteSaveVersion(SysVersionSaveVO save) {
        SysVersionDO sysVersion = null;
        if (ObjectUtils.isNotEmpty(save.getId())) {
            sysVersion = sysVersionRepoProc.get(save.getId());

            sysVersion.setVersionNo(save.getVersionNo());
            sysVersion.setChangeContent(save.getChangeContent());
            sysVersion.setReleaseTime(save.getReleaseTime());
            sysVersion.setStatus(save.getStatus());
            sysVersion.setDescribeMsg(save.getDescribeMsg());
        } else {
            sysVersion = SysVersionConvert.INSTANCE.saveToDo(save);

        }
        return sysVersion;
    }
}
