package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ZipUtil;
import cn.hutool.poi.excel.cell.CellUtil;
import com.alibaba.fastjson.JSONObject;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.core.support.rate.CommonRate;
import com.elitescloud.boot.core.support.rate.CommonRateBuilder;
import com.elitescloud.boot.provider.CloudtIdCreator;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.util.FileUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.model.entity.SysBasicRecordDO;
import com.elitescloud.cloudt.system.model.vo.save.devops.ExportBasicDataSaveVO;
import com.elitescloud.cloudt.system.service.SystemDataService;
import com.elitescloud.cloudt.system.service.common.constant.BasicRecordTypeEnum;
import com.elitescloud.cloudt.system.service.devops.init.*;
import com.elitescloud.cloudt.system.service.repo.BasicRecordRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.WorkbookUtil;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.StopWatch;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Service
@Slf4j
public class SystemDataServiceImpl extends BaseServiceImpl implements SystemDataService, InitializingBean {
    private static final String SUFFIX_ZIP = "zip";
    private static final String SUFFIX_XLSX = "xlsx";
    private static final String FILENAME_MANIFEST = "MANIFEST.properties";
    private static final String PROP_EXPORT_FILE = "_exportFile";
    private static final String PROP_EXPORT_TIME = "_exportTime";
    private static final String PROP_TENANT_ID = "_tenantId";
    private static final String PROP_STD_VERSION = "_stdVersion";
    private final List<AbstractBasicDataInitProvider> exportProviders = new ArrayList<>(64);
    private final List<AbstractBasicDataInitProvider> importProviders = new ArrayList<>(64);
    private String checkResult = null;

    @Autowired
    private BasicRecordRepoProc recordRepoProc;
    @Autowired
    private NamedParameterJdbcTemplate jdbcTemplate;
    @Autowired
    private PlatformTransactionManager transactionManager;
    @Autowired
    private CloudtIdCreator idCreator;
    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private FileService<String> fileService;

    @Override
    public void afterPropertiesSet() throws Exception {
        CompletableFuture.runAsync(() -> {
            initExportProviders();
            validateExportProvider(exportProviders);

            initImportProviders();
            validateExportProvider(importProviders);
        }).whenComplete((r, e) -> {
            if (e != null) {
                log.error("基础数据initProvider检查异常：", e);
                checkResult = e.getMessage();
            }
        });
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> exportData(ExportBasicDataSaveVO saveVO) {
        if (checkResult != null) {
            return ApiResult.fail("导出失败，系统正在初始化中");
        }

        // 保存导出记录
        SysBasicRecordDO recordDO = new SysBasicRecordDO();
        recordDO.setRecordType(BasicRecordTypeEnum.EXPORT);
        recordDO.setAllData(Boolean.TRUE.equals(saveVO.getAllData()));
        recordDO.setDataVersion(saveVO.getDataVersion());
        recordDO.setDataType(saveVO.getDataType());
        recordDO.setTenantCode(CharSequenceUtil.blankToDefault(saveVO.getTenantCode(), systemProperties.getStdProject().getTenantCode()));
        recordDO.setOperateTime(LocalDateTime.now());
        recordDO.setDescription(saveVO.getDescription());
        recordRepoProc.save(recordDO);

        // 开始导出
        var rate = CommonRateBuilder.builder("导出系统数据")
                .total(Integer.MAX_VALUE)
                .current(1)
                .key(recordDO.getId().toString())
                .build();
        taskExecutor.execute(() -> this.startExport(recordDO, rate));

        return ApiResult.ok(recordDO.getId());
    }

    @Override
    public ApiResult<Boolean> importData(MultipartFile file, Boolean incremental) {
        if (file == null || file.isEmpty()) {
            return ApiResult.fail("导入文件为空");
        }
        if (checkResult != null) {
            return ApiResult.fail("暂不支持导入：" + checkResult);
        }
        var importFileName = file.getOriginalFilename();
        var suffix = FileUtil.getSuffix(importFileName);
        var isSupport = SUFFIX_ZIP.equals(suffix) || SUFFIX_XLSX.equals(suffix);
        if (!isSupport) {
            return ApiResult.fail("解析导入文件失败，请使用导出的原文件");
        }

        // 转成临时文件
        File tempFile = null;
        try {
            tempFile = FileUtil.createTempFile();
            file.transferTo(tempFile);
        } catch (IOException e) {
            log.error("转存临时文件异常：", e);
            return ApiResult.fail("导入失败：" + e.getMessage());
        }

        // 解析导入的文件
        StopWatch watch = new StopWatch();
        watch.start();
        try {
            ImportedData importData = this.analyzeImportFile(tempFile, incremental, suffix);
            this.importFromExcel(importData);
        } catch (Exception e) {
            log.error("导入数据异常：", e);
            return ApiResult.fail("导入数据异常：" + e.getMessage());
        } finally {
            tempFile.delete();
            watch.stop();
            log.info("导入结束，耗时：{}s", watch.getTotalTimeSeconds());
        }

        return ApiResult.ok(true);
    }

    private void initExportProviders() {
        exportProviders.add(new PlatformAppInitProvider());
        exportProviders.add(new PlatformMenusInitProvider());
        exportProviders.add(new PlatformAdminMenusInitProvider());
        exportProviders.add(new PlatformApiManageInitProvider());
        exportProviders.add(new PlatformApiParameterInitProvider());
        exportProviders.add(new PlatformMenusApiInitProvider());
        exportProviders.add(new PlatformNextNumberInitProvider());
        exportProviders.add(new PlatformNumberRuleInitProvider());
        exportProviders.add(new PlatformNumberRuleDtlInitProvider());
        exportProviders.add(new PlatformUdcInitProvider());
        exportProviders.add(new PlatformUdcValueInitProvider());
        exportProviders.add(new PlatformAreaInitProvider());
        exportProviders.add(new PlatformCurrencyInitProvider());
        exportProviders.add(new TaxRateInitProvider());
        exportProviders.add(new TmplInitProvider());
        exportProviders.add(new CurrencyRateInitProvider());
        exportProviders.add(new SettingInitProvider());
        exportProviders.add(new MsgTemplateInitProvider());
        exportProviders.add(new MsgTemplateConfigInitProvider());
    }

    private void initImportProviders() {
        importProviders.add(new PlatformAppInitProvider());
        importProviders.add(new PlatformMenusInitProvider()
                .thenImport(new PlatformApiManageInitProvider().thenImport(new PlatformApiParameterInitProvider(), new PlatformMenusApiInitProvider()))
        );
        importProviders.add(new PlatformAdminMenusInitProvider());
        importProviders.add(new PlatformNextNumberInitProvider());
        importProviders.add(new PlatformNumberRuleInitProvider().thenImport(new PlatformNumberRuleDtlInitProvider()));
        importProviders.add(new PlatformUdcInitProvider());
        importProviders.add(new PlatformUdcValueInitProvider());
        importProviders.add(new PlatformAreaInitProvider());
        importProviders.add(new PlatformCurrencyInitProvider());
        importProviders.add(new TaxRateInitProvider());
        importProviders.add(new TmplInitProvider());
        importProviders.add(new CurrencyRateInitProvider());
        importProviders.add(new SettingInitProvider());
        importProviders.add(new MsgTemplateInitProvider().thenImport(new MsgTemplateConfigInitProvider()));
    }

    private Long queryTenantId(String tenantCode) {
        return jdbcTemplate.queryForObject("select id from sys_tenant where tenant_code = :tenant_code",
                Map.of("tenant_code", tenantCode), Long.class);
    }

    private List<Long> allTenantId() {
        var allTenants = tenantClientProvider.getAllTenants();
        if (allTenants.isEmpty()) {
            return Collections.emptyList();
        }
        return allTenants.stream().map(SysTenantDTO::getId).collect(Collectors.toList());
    }

    private String sheetName(AbstractBasicDataInitProvider provider) {
        return provider.tableName();
    }

    private void validateExportProvider(List<AbstractBasicDataInitProvider> providers) {
        Set<String> existsProviderNames = new HashSet<>();
        for (AbstractBasicDataInitProvider provider : providers) {
            // 校验是否重复
            var dataTypeName = provider.typeName();
            Assert.notBlank(dataTypeName, () -> new IllegalArgumentException(provider.getClass().getName() + "的typeName为空"));
            if (existsProviderNames.contains(dataTypeName)) {
                throw new IllegalArgumentException("存在重复的数据初始化：" + dataTypeName);
            }
            existsProviderNames.add(dataTypeName);

            // 校验sheetName
            var sheetName = sheetName(provider);
            try {
                WorkbookUtil.validateSheetName(sheetName);
            } catch (Exception e) {
                throw new IllegalArgumentException("sheet名称不合法：" + e.getMessage());
            }

            Assert.notEmpty(provider.fields(), () -> new IllegalArgumentException(provider.getClass().getName() + "的fields为空"));

            Assert.notEmpty(provider.fieldTitles(), () -> new IllegalArgumentException(provider.getClass().getName() + "的fieldTitles为空"));

            // 校验是否有死循环
            if (CollUtil.isNotEmpty(provider.thenImport())) {
                validateChildren(Collections.emptySet(), provider.thenImport());
            }
        }
    }

    private void validateChildren(Set<String> parentNames, List<AbstractBasicDataInitProvider> children) {
        for (AbstractBasicDataInitProvider child : children) {
            var childName = child.getClass().getName();
            if (parentNames.contains(childName)) {
                throw new IllegalArgumentException(childName + "可能出现循环调用");
            }
            if (CollUtil.isNotEmpty(child.thenImport())) {
                var tempParentNames = new HashSet<>(parentNames);
                tempParentNames.add(childName);
                this.validateChildren(tempParentNames, child.thenImport());
            }
        }
    }

    private ImportedData analyzeImportFile(File importFile, Boolean incremental, String fileSuffix) throws Exception {
        ImportedData importedData = new ImportedData();
        importedData.setIncremental(incremental == null || incremental);

        if (SUFFIX_XLSX.equals(fileSuffix)) {
            // 上传的excel文件
            importedData.setManifest(new Properties());
            importedData.setWorkbook(new XSSFWorkbook(importFile));
            importedData.setOtherFiles(Collections.emptyMap());

            return importedData;
        }

        // 解压文件
        File zipDir = null;
        try {
            zipDir = ZipUtil.unzip(importFile);
        } catch (UtilException e) {
            log.error("解压上传文件异常：", e);
            throw new IllegalArgumentException("解压上传文件异常：" + e.getMessage());
        }

        // 资源清单
        var manifest = this.loadManifest(zipDir);
        importedData.setManifest(manifest);

        // 导入的excel文件
        Workbook exportFile = this.loadImportFile(zipDir, manifest);
        importedData.setWorkbook(exportFile);

        // 其它文件
        importedData.setOtherFiles(new HashMap<>(128));
        Set<String> excludeFiles = Set.of(FILENAME_MANIFEST, manifest.getProperty(PROP_EXPORT_FILE));
        this.loadOtherFiles(zipDir, "", excludeFiles, importedData.getOtherFiles());

        return importedData;
    }

    private void loadOtherFiles(File dir, String keyPrefix, Set<String> excludeFileNames, Map<String, File> fileMap) {
        var childFiles = dir.listFiles((dir1, name) -> excludeFileNames == null || !excludeFileNames.contains(name));
        if (ArrayUtil.isEmpty(childFiles)) {
            return;
        }
        for (File childFile : childFiles) {
            if (childFile.isDirectory()) {
                this.loadOtherFiles(childFile, keyPrefix + childFile.getName() + File.separator, null, fileMap);
                continue;
            }

            fileMap.put(keyPrefix + childFile.getName(), childFile);
        }
    }

    private Workbook loadImportFile(File zipDir, Properties manifest) {
        var exportFileName = manifest.getProperty(PROP_EXPORT_FILE);
        if (CharSequenceUtil.isBlank(exportFileName)) {
            throw new IllegalArgumentException("文件异常，缺少资源清单");
        }
        File exportFile = new File(zipDir, exportFileName);
        if (!exportFile.exists()) {
            throw new IllegalArgumentException("文件异常，资源清单不存在");
        }

        try {
            return new XSSFWorkbook(exportFile);
        } catch (Exception e) {
            log.error("加载excel文件异常：", e);
            throw new IllegalArgumentException("加载导入文件异常");
        }
    }

    private Properties loadManifest(File zipDir) {
        Properties properties = new Properties();
        try (FileReader fileReader = new FileReader(new File(zipDir, FILENAME_MANIFEST))) {
            properties.load(fileReader);
        } catch (IOException e) {
            log.error("加载资源清单失败：", e);
            throw new IllegalArgumentException("加载资源清单失败");
        }
        return properties;
    }

    private void importFromExcel(ImportedData importData) throws Exception {
        // 加入线程队列
        List<CompletableFuture<Void>> threads = new ArrayList<>();
        var tenantIdList = this.allTenantId();
        for (AbstractBasicDataInitProvider provider : importProviders) {
            if (!provider.isTenantData()) {
                // 非租户
                var thread = createThreadToImportSheet(importData, provider, TenantConstant.DEFAULT_TENANT_ID);
                threads.add(thread);
                continue;
            }

            if (!tenantIdList.isEmpty()) {
                // 所有租户
                for (Long tenantId : tenantIdList) {
                    var thread = createThreadToImportSheet(importData, provider, tenantId);
                    threads.add(thread);
                }
            }
        }
        CompletableFuture.allOf(threads.toArray(CompletableFuture[]::new))
                .exceptionally(e -> {
                    log.error("导入基础数据异常：", e);
                    return null;
                })
                .get();
    }

    private CompletableFuture<Void> createThreadToImportSheet(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                                              Long tenantId) {
        return CompletableFuture.runAsync(() -> createImportRunner(importData, dataInitProvider, Collections.emptyList(), tenantId))
                .whenComplete((r, e) -> {
                    if (e != null) {
                        log.error("导入{}异常：", dataInitProvider.typeName(), e);
                    }
                    dataInitProvider.afterExecute(e);
                });
    }

    private List<Map<String, Object>> createImportRunner(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                                         List<Map<String, Object>> preDataList, Long tenantId) {
        log.info("开始处理导入数据：{}", dataInitProvider.typeName());
        var workbook = importData.getWorkbook();
        var sheetName = this.sheetName(dataInitProvider);
        var sheet = workbook.getSheet(sheetName);
        if (sheet == null) {
            log.info("不存在sheet：{}", sheetName);
            return Collections.emptyList();
        }
        // 读取属性
        var fieldMap = this.readFields(sheet);
        if (fieldMap.isEmpty()) {
            log.info("为读取到属性：{}", sheetName);
            return Collections.emptyList();
        }

        // 从excel里读取数据
        var dataList = this.readDataFromSheet(sheet, fieldMap);
        if (dataList.isEmpty()) {
            log.info("数据为空：{}", sheetName);
            return Collections.emptyList();
        }
        var fields = dataInitProvider.fields();
        String sqlInsert = this.generateSqlInsert(fieldMap, fields, dataInitProvider.tableName());
        String sqlUpdate = this.generateSqlUpdate(fieldMap, fields, dataInitProvider.tableName());
        if (CharSequenceUtil.hasBlank(sqlInsert, sqlUpdate)) {
            log.info("未生成有效SQL：{}，{}，{}", dataInitProvider.typeName(), sqlInsert, sqlUpdate);
            return dataList;
        }

        // 当前所有数据
        var currentData = this.queryCurrentData(importData.isIncremental(), dataInitProvider, tenantId);

        // 开始更新数据库
        TransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        TransactionStatus transactionStatus = transactionManager.getTransaction(transactionDefinition);
        Throwable throwable = null;
        try {
            // 导入前准备
            dataInitProvider.prepareForImport(importData, tenantId);

            this.executeImportData(importData, dataInitProvider, currentData, dataList, preDataList, tenantId,
                    sqlInsert, sqlUpdate);
            transactionManager.commit(transactionStatus);

            log.info("处理导入数据{}成功", dataInitProvider.typeName());
        } catch (Throwable e) {
            log.error("更新数据异常：", e);
            throwable = e;
            transactionManager.rollback(transactionStatus);
            return dataList;
        } finally {
            dataInitProvider.afterExecute(throwable);
        }

        // 依赖项的处理
        var children = dataInitProvider.thenImport();
        if (CollUtil.isNotEmpty(children)) {
            for (AbstractBasicDataInitProvider child : children) {
                createImportRunner(importData, child, dataList, tenantId);
            }
        }
        return dataList;
    }

    private void executeImportData(ImportedData importData, AbstractBasicDataInitProvider dataInitProvider,
                                   List<Map<String, Object>> currentDataList, List<Map<String, Object>> importDataList,
                                   List<Map<String, Object>> preDataList, Long tenantId, String sqlInsert, String sqlUpdate) {
        if (!importData.isIncremental()) {
            // 删除目前的
            String sqlDel = "delete from " + dataInitProvider.tableName() + " where tenant_id = :tenant_id";
            jdbcTemplate.update(sqlDel, Map.of("tenant_id", tenantId));
        }

        List<Map<String, Object>> sqlValuesAdd = new ArrayList<>(50);
        List<Map<String, Object>> sqlValuesUpdate = new ArrayList<>(50);

        AbstractBasicDataInitProvider.UpdateType updateType = null;
        LocalDateTime nowTime = LocalDateTime.now();
        for (Map<String, Object> data : importDataList) {
            data.put("tenant_id", tenantId);
            data.put("modify_time", nowTime);

            updateType = dataInitProvider.convertForImport(data, currentDataList, preDataList);
            if (updateType == null || updateType == AbstractBasicDataInitProvider.UpdateType.IGNORE) {
                // 忽略
                continue;
            }
            if (updateType == AbstractBasicDataInitProvider.UpdateType.ADD) {
                data.put("id", idCreator.create());
                data.put("create_time", nowTime);
                sqlValuesAdd.add(normalizeSqlParam(data));
                if (sqlValuesAdd.size() == 20) {
                    jdbcTemplate.batchUpdate(sqlInsert, sqlValuesAdd.toArray(Map[]::new));
                    sqlValuesAdd.clear();
                }
                continue;
            }

            sqlValuesUpdate.add(normalizeSqlParam(data));
            if (data.get("id") == null) {
                log.info("更新失败数据：{}", JSONObject.toJSONString(data));
                throw new IllegalStateException(dataInitProvider.typeName() + "更新数据时ID为空");
            }
            if (sqlValuesUpdate.size() == 20) {
                jdbcTemplate.batchUpdate(sqlUpdate, sqlValuesUpdate.toArray(Map[]::new));
                sqlValuesUpdate.clear();
            }
        }

        if (!sqlValuesAdd.isEmpty()) {
            jdbcTemplate.batchUpdate(sqlInsert, sqlValuesAdd.toArray(Map[]::new));
            sqlValuesAdd.clear();
        }
        if (!sqlValuesUpdate.isEmpty()) {
            jdbcTemplate.batchUpdate(sqlUpdate, sqlValuesUpdate.toArray(Map[]::new));
            sqlValuesUpdate.clear();
        }
    }

    private Map<String, Object> normalizeSqlParam(Map<String, Object> param) {
        Map<String, Object> newParam = new HashMap<>(param);
        for (Map.Entry<String, Object> entry : param.entrySet()) {
            if (entry.getValue() instanceof Boolean) {
                newParam.put(entry.getKey(), ((Boolean) entry.getValue()) ? 1 : 0);
                continue;
            }
            if (entry.getValue() instanceof String && CharSequenceUtil.isBlank(entry.getValue().toString())) {
                newParam.put(entry.getKey(), null);
                continue;
            }

            newParam.put(entry.getKey(), entry.getValue());
        }
        return newParam;
    }

    private List<Map<String, Object>> queryCurrentData(boolean incremental, AbstractBasicDataInitProvider dataInitProvider,
                                                       Long tenantId) {
        if (!incremental) {
            // 不是增量即全量，直接删除原所有数据，无需查询当前数据
            return Collections.emptyList();
        }
        return this.queryData(dataInitProvider, tenantId);
    }

    private String generateSqlInsert(Map<String, Integer> fieldMap, List<String> fields, String tableName) {
        var tableColumns = fieldMap.entrySet().stream()
                .filter(t -> fields.contains(t.getKey()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        if (tableColumns.isEmpty()) {
            return null;
        }
        if (!tableColumns.contains("id")) {
            tableColumns.add(0, "id");
        }
        if (!tableColumns.contains("tenant_id")) {
            tableColumns.add(1, "tenant_id");
        }
        if (!tableColumns.contains("create_time")) {
            tableColumns.add("create_time");
        }

        String sqlValue = tableColumns.stream().map(t -> ":" + t).collect(Collectors.joining(", "));

        return "insert into " + tableName + "(" + String.join(", ", tableColumns) + ") values (" + sqlValue + ")";
    }

    private String generateSqlUpdate(Map<String, Integer> fieldMap, List<String> fields, String tableName) {
        var tableColumns = fieldMap.entrySet().stream()
                .filter(t -> fields.contains(t.getKey()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
        if (tableColumns.isEmpty()) {
            return null;
        }
        if (tableColumns.contains("create_time")) {
            tableColumns.remove("create_time");
        }

        String sqlValue = tableColumns.stream()
                .map(t -> t + " = :" + t).collect(Collectors.joining(", "));

        return "update " + tableName + " set " + sqlValue + " where id = :id";
    }

    private List<Map<String, Object>> readDataFromSheet(Sheet sheet, Map<String, Integer> fieldMap) {
        var lastRowNum = sheet.getLastRowNum();
        if (lastRowNum < 2) {
            return Collections.emptyList();
        }

        List<Map<String, Object>> dataList = new ArrayList<>(lastRowNum + 1);
        var fieldEntry = fieldMap.entrySet();

        Row row = null;
        Cell cell = null;
        Object cellValue = null;
        Map<String, Object> rowData = null;
        boolean hasData = false;
        for (int rowNum = 2; rowNum <= lastRowNum; rowNum++) {
            row = sheet.getRow(rowNum);
            rowData = new HashMap<>(fieldMap.size());
            hasData = false;

            // 读取行数据
            for (Map.Entry<String, Integer> entry : fieldEntry) {
                cell = row.getCell(entry.getValue());
                if (cell == null) {
                    continue;
                }
                cellValue = getCellValue(cell);
                if (cellValue == null) {
                    continue;
                }

                rowData.put(entry.getKey(), cellValue);
                hasData = true;
            }
            if (hasData) {
                dataList.add(rowData);
            }
        }
        return dataList;
    }

    private Object getCellValue(Cell cell) {
        var cellType = cell.getCellTypeEnum();
        return CellUtil.getCellValue(cell, cellType, true);
    }

    private void startExport(SysBasicRecordDO recordDO, CommonRate rate) {
        var tenantCode = recordDO.getTenantCode();
        // 准备工作目录
        var workDir = FileUtil.createTempDir(CharSequenceUtil.format("系统数据-{}", DatetimeUtil.currentTimeLong()));
        // 导出并压缩
        File zipFile = null;
        try {
            Assert.notBlank(tenantCode, "缺少配置标产租户编码");
            Long tenantId = this.queryTenantId(tenantCode);
            Assert.notNull(tenantId, "{}对应的租户不存在", tenantCode);
            log.info("导出所在工作目录:{}", workDir.getAbsolutePath());

            // 开始导出
            Properties props = new Properties();
            props.put(PROP_STD_VERSION, recordDO.getDataVersion());
            this.exportToExcel(workDir, tenantId, props, rate);
            zipFile = ZipUtil.zip(workDir, StandardCharsets.UTF_8);

            // 上传导出的文件
            var uploadResult = fileService.upload(zipFile);
            Assert.isTrue(uploadResult.isSuccess(), "上传导出文件失败，" + uploadResult.getMsg());
            recordRepoProc.updateExportSuccessfully(recordDO.getId(), uploadResult.getData().getFileCode(), props.toString());

            rate.complete(uploadResult.getData().getFileCode());
        } catch (Exception e) {
            recordRepoProc.updateFailResult(recordDO.getId(), ExceptionsUtil.stacktraceToString(e));
            rate.exceptionally(e.getMessage());
            log.error("导出系统的初始化数据异常：", e);
        } finally {
            FileUtil.del(workDir);
            FileUtil.del(zipFile);
        }
    }

    private void exportToExcel(File workDir, Long tenantId, Properties props, CommonRate rate) throws Exception {
        // 创建导出文件
        File excelFile = new File(workDir, workDir.getName() + "." + SUFFIX_XLSX);
        var createResult = excelFile.createNewFile();
        Assert.isTrue(createResult, "创建导出文件失败");

        props.setProperty(PROP_EXPORT_FILE, excelFile.getName());
        props.setProperty(PROP_TENANT_ID, tenantId.toString());

        try (var workbook = new SXSSFWorkbook(); var output = new FileOutputStream(excelFile)) {
            // 加入线程队列
            List<CompletableFuture<Void>> threads = new ArrayList<>();
            for (AbstractBasicDataInitProvider provider : exportProviders) {
                var thread = createThreadToExportSheet(workbook, provider, workDir, tenantId, rate);
                threads.add(thread);
            }
            CompletableFuture.allOf(threads.toArray(CompletableFuture[]::new))
                    .exceptionally(e -> {
                        log.error("导出基础数据异常：", e);
                        return null;
                    })
                    .get();

            workbook.write(output);
        }

        // 资源文件清单
        props.setProperty(PROP_EXPORT_TIME, DatetimeUtil.currentTimeStr());
        File manifest = new File(workDir, FILENAME_MANIFEST);
        try (var fileWriter = new FileWriter(manifest)) {
            props.store(fileWriter, "resource list, exported by cloudt");
        }
    }

    private CompletableFuture<Void> createThreadToExportSheet(Workbook workbook, AbstractBasicDataInitProvider dataInitProvider,
                                                              File workDir, Long tenantIdReal, CommonRate rate) {
        var sheetName = this.sheetName(dataInitProvider);
        var sheet = workbook.createSheet(sheetName);

        return CompletableFuture.runAsync(() -> {
            var dataTypeName = sheet.getSheetName();
            var tenantId = dataInitProvider.isTenantData() ? tenantIdReal : TenantConstant.DEFAULT_TENANT_ID;

            sheet.setDefaultColumnWidth(22);

            log.info("开始导出{}...", dataTypeName);
            // 表格头部
            var fieldTitles = dataInitProvider.fieldTitles();
            exportRowTitle(sheet, fieldTitles);

            dataInitProvider.prepareExport(workDir);

            // 查询数据
            var dataList = queryData(dataInitProvider, tenantId);
            if (!dataList.isEmpty()) {
                int rowIndex = 2;
                int cellIndex = 0;
                Row row = null;
                Cell cell = null;
                var cellStyle = createCommonStyle(sheet);

                var fields = fieldTitles.keySet();
                for (var rowData : dataList) {
                    // 过滤和转换数据
                    var realData = dataInitProvider.convertForExport(rowData, null);
                    if (CollUtil.isEmpty(realData)) {
                        continue;
                    }
                    row = sheet.createRow(rowIndex);

                    cellIndex = 0;
                    for (String field : fields) {
                        cell = row.createCell(cellIndex);
                        cell.setCellValue(convertFieldValue(realData.get(field)));
                        cell.setCellStyle(cellStyle);

                        cellIndex++;
                    }

                    rowIndex++;
                }
            }

            rate.increaseRate(dataList.size());
            log.info("导出{}结束", dataTypeName);
        }).whenComplete((r, e) -> {
            if (e != null) {
                log.error("导出{}异常：", dataInitProvider.typeName(), e);
            }
            dataInitProvider.afterExecute(e);
        });
    }

    private void exportRowTitle(Sheet sheet, Map<String, String> fieldTitles) {
        var cellStyle = createTitleStyle(sheet);

        // 创建标题
        var row = sheet.createRow(0);
        int cellIndex = 0;
        Cell cell = null;
        for (String value : fieldTitles.values()) {
            cell = row.createCell(cellIndex);
            cell.setCellValue(value);

            // 样式
            cell.setCellStyle(cellStyle);

            cellIndex++;
        }
        // 创建属性
        var fields = fieldTitles.keySet();
        row = sheet.createRow(1);
        cellIndex = 0;
        for (String value : fields) {
            cell = row.createCell(cellIndex);
            cell.setCellValue(value);
            cell.setCellStyle(cellStyle);

            cellIndex++;
        }
    }

    private Map<String, Integer> readFields(Sheet sheet) {
        var row = sheet.getRow(1);
        if (row == null) {
            return Collections.emptyMap();
        }

        var lastCellNum = row.getLastCellNum();
        Map<String, Integer> fieldMap = new HashMap<>(lastCellNum + 1);
        Cell cell = null;
        String cellValue = null;
        for (int colNum = 0; colNum <= lastCellNum; colNum++) {
            cell = row.getCell(colNum);
            if (cell == null) {
                continue;
            }
            cellValue = cell.getStringCellValue();
            if (cellValue != null) {
                cellValue = cellValue.trim();
            }
            if (CharSequenceUtil.isBlank(cellValue)) {
                continue;
            }
            fieldMap.put(cellValue, colNum);
        }

        return fieldMap;
    }

    private CellStyle createCommonStyle(Sheet sheet) {
        CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
        var font = sheet.getWorkbook().createFont();
        cellStyle.setFont(font);
        cellStyle.setAlignment(HorizontalAlignment.LEFT);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setWrapText(true);

        return cellStyle;
    }

    private CellStyle createTitleStyle(Sheet sheet) {
        CellStyle cellStyle = sheet.getWorkbook().createCellStyle();
        var font = sheet.getWorkbook().createFont();
        cellStyle.setFont(font);
        cellStyle.setAlignment(HorizontalAlignment.CENTER);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setWrapText(true);

        font.setBold(true);
        return cellStyle;
    }

    private List<Map<String, Object>> queryData(AbstractBasicDataInitProvider dataInitProvider, Long tenantId) {
        var fields = new ArrayList<>(dataInitProvider.fields());
        if (!fields.contains("id")) {
            fields.add("id");
        }
        String sql = "select " + String.join(", ", fields) + " from " + dataInitProvider.tableName();

        // where 条件
        Map<String, Object> sqlParams = new HashMap<>(8);
        List<String> conditionList = new ArrayList<>(8);

        if (tenantId != null) {
            conditionList.add("tenant_id = :tenant_id");
            sqlParams.put("tenant_id", tenantId);
        }

        conditionList.add("delete_flag = :delete_flag");
        sqlParams.put("delete_flag", 0);

        sql += " where ";
        sql += String.join(" and ", conditionList);

        // 排序字段
        var orderFieldList = dataInitProvider.orderItems();
        if (CollUtil.isNotEmpty(orderFieldList)) {
            var order = orderFieldList.stream()
                    .filter(t -> CharSequenceUtil.isNotBlank(t.getColumn()))
                    .map(t -> t.getColumn() + " " + (t.isAsc() ? "asc" : "desc"))
                    .collect(Collectors.joining(", "));
            if (CharSequenceUtil.isNotBlank(order)) {
                sql += (" order by " + order);
            }
        }

        var dataList = jdbcTemplate.queryForList(sql, sqlParams);
        return ObjectUtil.defaultIfNull(dataList, Collections.emptyList());
    }

    private String convertFieldValue(Object value) {
        if (value == null) {
            return null;
        }
        if (value instanceof LocalDateTime) {
            return DatetimeUtil.FORMATTER_DATETIME.format((LocalDateTime) value);
        }
        if (value instanceof LocalDate) {
            return DatetimeUtil.FORMATTER_DATE.format((LocalDate) value);
        } else if (value instanceof Date) {
            return DatetimeUtil.FORMAT_DATETIME.format((Date) value);
        }

        return value.toString();
    }

    public static class ImportedData {
        private boolean incremental;
        private Properties manifest;
        private Workbook workbook;
        private Map<String, File> otherFiles;

        public boolean isIncremental() {
            return incremental;
        }

        public void setIncremental(boolean incremental) {
            this.incremental = incremental;
        }

        public Properties getManifest() {
            return manifest;
        }

        public void setManifest(Properties manifest) {
            this.manifest = manifest;
        }

        public Workbook getWorkbook() {
            return workbook;
        }

        public void setWorkbook(Workbook workbook) {
            this.workbook = workbook;
        }

        public Map<String, File> getOtherFiles() {
            return otherFiles;
        }

        public void setOtherFiles(Map<String, File> otherFiles) {
            this.otherFiles = otherFiles;
        }
    }
}
