package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.common.PwdStrategyTypeEnum;
import com.elitescloud.boot.auth.cas.provider.PwdStrategyTransferHelper;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepoProc;
import com.elitescloud.cloudt.system.model.vo.CommonTenantAppVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAppPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRateQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonAreaTreeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonTaxRateRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.system.PwdStrategyRespVO;
import com.elitescloud.cloudt.system.service.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class SystemQueryServiceImpl extends BaseServiceImpl implements SystemQueryService {

    @Autowired
    private SysPlatformAppRepoProc appRepoProc;
    @Autowired
    private AreaQueryService areaQueryService;
    @Autowired
    private CurrencyQueryService currencyQueryService;
    @Autowired
    private TaxRateQueryService taxRateQueryService;
    @Autowired
    private CurrencyRateQueryService currencyRateQueryService;
    @Autowired(required = false)
    private PwdStrategyTransferHelper pwdStrategyTransferHelper;

    @Override
    public ApiResult<List<CommonTenantAppVO>> listApp() {
        if (super.isTenantUser()) {
            // 如果是租户下的用户，则返回租户的应用列表，否则返回所有
            // 获取租户的应用
            var tenant = super.currentTenant();
            if (tenant == null || CollectionUtils.isEmpty(tenant.getAppCodes())) {
                return ApiResult.ok(Collections.emptyList());
            }

            var codeMap = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allEnabled())
                    .stream()
                    .collect(Collectors.toMap(CodeNameParam::getCode, CodeNameParam::getName, (t1, t2) -> t1));
            var appList = tenant.getAppCodes().stream()
                    .filter(codeMap::containsKey)
                    .map(t -> new CommonTenantAppVO(t, codeMap.get(t)))
                    .collect(Collectors.toList());
            return ApiResult.ok(appList);
        }

        // 获取所有
        var appList = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allEnabled())
                .stream()
                .map(t -> new CommonTenantAppVO(t.getCode(), t.getName()))
                .collect(Collectors.toList());
        return ApiResult.ok(appList);
    }

    @Override
    public ApiResult<PagingVO<CommonTenantAppVO>> pageApp(CommonAppPageQueryVO queryVO) {
        var listData = listApp().getData();
        if (CollectionUtils.isEmpty(listData)) {
            return ApiResult.ok(PagingVO.empty());
        }

        // 转为分页信息
        var pageRequest = queryVO.getPageRequest();
        int offset = (int) pageRequest.getOffset();
        var recordList = CollUtil.sub(listData, offset, offset + pageRequest.getPageSize());
        var pageData = PagingVO.<CommonTenantAppVO>builder()
                .total(listData.size()).records(recordList)
                .build();
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaQueryVO queryVO) {
        return areaQueryService.listArea(queryVO);
    }

    @Override
    public ApiResult<List<CodeNameParam>> listCurrency() {
        return currencyQueryService.listCurrency();
    }

    @Override
    public ApiResult<List<CommonTaxRateRespVO>> listTaxRate(String taxRateNo, String taxRateType) {
        return taxRateQueryService.queryList(taxRateNo, taxRateType);
    }

    @Override
    public ApiResult<Double> queryCurrentRate(CurrencyRateQueryVO queryVO) {
        return currencyRateQueryService.findRatio(queryVO);
    }

    @Override
    public ApiResult<List<PwdStrategyRespVO>> listPwdStrategyForCreateAccount() {
        if (pwdStrategyTransferHelper == null) {
            return ApiResult.fail("系统异常，请稍后再试");
        }

        var queryResult = pwdStrategyTransferHelper.listByType(PwdStrategyTypeEnum.PWD_UPDATE);
        if (Boolean.FALSE.equals(queryResult.getSuccess())) {
            return ApiResult.fail(queryResult.getMsg());
        }
        var strategyList = queryResult.getData().stream().map(t -> {
            PwdStrategyRespVO respVO = new PwdStrategyRespVO();
            respVO.setStrategyCode(t.getStrategyCode());
            respVO.setStrategyName(t.getStrategyName());
            respVO.setParams(ObjectUtil.defaultIfNull(t.getParams(), () -> new String[0]));

            return respVO;
        }).collect(Collectors.toList());

        return ApiResult.ok(strategyList);
    }
}
