package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.model.vo.query.org.TenantOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.TenantOrgDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.TenantOrgRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.OrgTenantSaveVO;
import com.elitescloud.cloudt.system.service.TenantOrgMngService;
import com.elitescloud.cloudt.system.service.manager.UserQueryManager;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserTypeDO;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantOrgRepo;
import com.elitescloud.cloudt.system.service.repo.TenantOrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserTypeRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/30
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class TenantOrgMngServiceImpl extends BaseServiceImpl implements TenantOrgMngService {

    @Autowired
    private TenantOrgRepo tenantOrgRepo;
    @Autowired
    private TenantOrgRepoProc tenantOrgRepoProc;
    @Autowired
    private UserTypeRepoProc userTypeRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;

    @Autowired
    private UserQueryManager userQueryManager;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveTenantOrg(OrgTenantSaveVO saveVO) {
        // 先根据组织ID查询已有数据
        var tenantOrg = tenantOrgRepoProc.getByOrgId(saveVO.getOrgId());
        Long oldAdminId = null;
        // 保存租户组织
        if (tenantOrg == null) {
            // 不存在则创建
            tenantOrg = new SysTenantOrgDO();
        } else {
            oldAdminId = tenantOrg.getAdminId();
        }
        tenantOrg.setOrgId(saveVO.getOrgId());
        tenantOrg.setAdminId(saveVO.getAdminUserId());
        tenantOrg.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), true));
        tenantOrgRepo.save(tenantOrg);

        // 更新组织管理员类型
        updateAdminUserType(oldAdminId, saveVO.getAdminUserId());

        return ApiResult.ok(tenantOrg.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        var tenantOrgDO = tenantOrgRepoProc.get(id);
        if (tenantOrgDO == null) {
            return ApiResult.noData();
        }

        tenantOrgRepoProc.delete(id);
        if (tenantOrgDO.getAdminId() != null) {
            // 删除组织管理员类型
            var tenantId = super.currentTenantId();
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                userTypeRepoProc.delete(tenantOrgDO.getAdminId(), Set.of(UserType.ADMIN_ORG.getValue()), tenantId);
                return null;
            });
        }

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Boolean enabled = tenantOrgRepoProc.getEnabled(id);
        enabled = enabled == null || !enabled;

        tenantOrgRepoProc.updateEnabled(id, enabled);
        return ApiResult.ok();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateAdmin(Long id, Long adminId) {
        var oldAdminId = tenantOrgRepoProc.getAdminId(id);
        if (oldAdminId != null && Objects.equals(oldAdminId, adminId)) {
            // 管理员未变更
            return ApiResult.ok(id);
        }

        tenantOrgRepoProc.updateAdminId(id, adminId);

        // 更新管理员类型
        updateAdminUserType(oldAdminId, adminId);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<TenantOrgDetailRespVO> getDetail(Long id) {
        // 查询租户组织
        var tenantOrgDO = tenantOrgRepoProc.get(id);
        if (tenantOrgDO == null) {
            return ApiResult.noData();
        }
        TenantOrgDetailRespVO respVO = new TenantOrgDetailRespVO();
        respVO.setId(tenantOrgDO.getId());
        respVO.setOrgId(tenantOrgDO.getOrgId());
        respVO.setEnabled(tenantOrgDO.getEnabled());

        // 获取组织信息
        var org = orgRepoProc.getBasicDto(tenantOrgDO.getOrgId());
        if (org != null) {
            respVO.setOrgCode(org.getCode());
            respVO.setOrgName(org.getName());
        }
        SysUserBasicDTO admin = null;
        if (tenantOrgDO.getAdminId() != null) {
            admin = userQueryManager.getBasic(tenantOrgDO.getAdminId());
            if (admin != null) {
                respVO.setAdminUserId(admin.getId());
                respVO.setAdminUsername(admin.getUsername());
                respVO.setAdminUser(admin.getFullName());
                respVO.setMobile(admin.getMobile());
            }
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<List<TenantOrgRespVO>> getList() {
        // 查询租户组织
        var tenantOrgs = tenantOrgRepoProc.all();

        // 转vo返回
        var tenantOrgList = tenantOrgDo2VoList(tenantOrgs);

        return ApiResult.ok(tenantOrgList);
    }

    @Override
    public ApiResult<PagingVO<TenantOrgRespVO>> pageQuery(TenantOrgPageQueryVO queryVO) {
        // 查询租户组织
        var pageTenantOrgDO = tenantOrgRepoProc.pageMng(queryVO);
        if (pageTenantOrgDO.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }

        // 转vo
        var tenantOrgList = tenantOrgDo2VoList(pageTenantOrgDO.getRecords());
        var pageData = PagingVO.<TenantOrgRespVO>builder().total(pageTenantOrgDO.getTotal()).records(tenantOrgList).build();
        return ApiResult.ok(pageData);
    }

    private List<TenantOrgRespVO> tenantOrgDo2VoList(List<SysTenantOrgDO> orgDoList) {
        if (orgDoList.isEmpty()) {
            return Collections.emptyList();
        }
        // 查询组织和管理员信息
        var orgIds = orgDoList.stream()
                .map(SysTenantOrgDO::getOrgId)
                .collect(Collectors.toSet());
        var adminIds = orgDoList.stream()
                .map(SysTenantOrgDO::getAdminId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        var orgMap = orgRepoProc.getBasicDtoList(orgIds).stream()
                .collect(Collectors.toMap(SysOrgBasicDTO::getId, t -> t, (t1, t2) -> t1));
        var adminMap = userQueryManager.getBasic(adminIds).stream()
                .collect(Collectors.toMap(SysUserBasicDTO::getId, t -> t, (t1, t2) -> t1));

        // 返回信息
        return orgDoList.stream().map(t -> {
            TenantOrgRespVO respVO = new TenantOrgRespVO();
            respVO.setId(t.getId());
            respVO.setOrgId(t.getOrgId());
            var org = orgMap.get(t.getOrgId());
            if (org != null) {
                respVO.setOrgCode(org.getCode());
                respVO.setOrgName(org.getName());
            }
            if (t.getAdminId() != null) {
                respVO.setAdminUserId(t.getAdminId());
                var admin = adminMap.get(t.getAdminId());
                if (admin != null) {
                    respVO.setAdminUsername(admin.getUsername());
                    respVO.setAdminUser(admin.getFullName());
                    respVO.setMobile(admin.getMobile());
                }
            }
            respVO.setEnabled(t.getEnabled());
            respVO.setCreateTime(t.getCreateTime());
            respVO.setModifyTime(t.getModifyTime());

            return respVO;
        }).collect(Collectors.toList());
    }

    private void updateAdminUserType(Long oldAdminId, Long adminUserId) {
        // 移除老的
        if (oldAdminId != null && !Objects.equals(oldAdminId, adminUserId)) {
            var tenantId = super.currentTenantId();
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                userTypeRepoProc.delete(oldAdminId, Set.of(UserType.ADMIN_ORG.getValue()), tenantId);
                return null;
            });
        }

        // 增加新的
        if (adminUserId != null && !Objects.equals(oldAdminId, adminUserId)) {
            // 判断用户是否属于当前租户
            var currentTenantId = super.currentTenantId();
            Assert.isTrue(userQueryManager.belongToTenant(adminUserId, currentTenantId), "指定用户不属于当前租户");

            SysUserTypeDO userTypeDO = new SysUserTypeDO();
            userTypeDO.setUserId(adminUserId);
            userTypeDO.setType(UserType.ADMIN_ORG.getValue());
            userTypeRepoProc.save(userTypeDO);
        }
    }
}
