package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.FileByteVO;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.TmplConvert;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplQueryParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplPageQueryRespVO;
import com.elitescloud.cloudt.system.service.TmplQueryService;
import com.elitescloud.cloudt.system.service.repo.SysTmplRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/20/2023
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class TmplQueryServiceImpl extends TmplBaseServiceImpl implements TmplQueryService {

    @Autowired
    private SysTmplRepoProc tmplRepoProc;

    @Override
    public ApiResult<PagingVO<TmplPageQueryRespVO>> search(TmplQueryParam queryParam) {
        if (CharSequenceUtil.isBlank(queryParam.getAppCode())) {
            return ApiResult.ok(PagingVO.empty());
        }
        var result = tmplRepoProc.pageQuery(queryParam)
                .map(TmplConvert.INSTANCE::do2PageQueryVo);
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<SysTmplDTO> getByCode(String code) {
        var tmplDO = tmplRepoProc.getByCode(code);
        if (tmplDO == null) {
            return ApiResult.fail("数据不存在");
        }

        if (!BooleanUtil.isTrue(tmplDO.getEnabled())) {
            return ApiResult.fail("模板未启用");
        }

        var dto = TmplConvert.INSTANCE.vo2Dto(tmplDO);

        // 字段属性
        List<List<String>> attributes = null;
        if (CharSequenceUtil.isNotBlank(tmplDO.getAttributeList())) {
            attributes = super.json2Obj(tmplDO.getAttributeList(), new TypeReference<>() {
            });
        }
        dto.setAttributes(ObjectUtil.defaultIfNull(attributes, Collections.emptyList()));
        if (!dto.getAttributes().isEmpty() && dto.getAttributes().size() >= dto.getFieldTypeRow()) {
            dto.setAttributeFields(dto.getAttributes().get(dto.getFieldTypeRow() - 1));
        }

        // sheet页
        if (CharSequenceUtil.isNotBlank(tmplDO.getSheetNos())) {
            dto.setSheetNoList(Arrays.stream(tmplDO.getSheetNos().split(",")).map(Integer::parseInt).collect(Collectors.toList()));
        }

        // 加入缓存
        cache(dto);

        return ApiResult.ok(dto);
    }

    @Override
    public ApiResult<String> getFileCodeByCode(String code) {
        String fileCode = tmplRepoProc.getFileCodeByCode(code);
        return ApiResult.ok(fileCode);
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadByFileCode(String fileCode) {
        return super.streamByFileCode(fileCode);
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadByCode(String code) {
        if (CharSequenceUtil.isBlank(code)) {
            return ResponseEntity.badRequest().build();
        }
        String fileCode = tmplRepoProc.getFileCodeByCode(code);
        return super.streamByFileCode(fileCode);
    }

    @Override
    public ApiResult<FileByteVO> getTmplFileByCode(String code) {
        if (CharSequenceUtil.isBlank(code)) {
            return ApiResult.fail("模板编码为空");
        }

        var fileCode = tmplRepoProc.getFileCodeByCode(code);
        if (CharSequenceUtil.isBlank(fileCode)) {
            return ApiResult.fail("未找到有效模板文件");
        }

        var byteVO = super.getFileVO(fileCode);
        return ApiResult.ok(byteVO);
    }
}
