package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformUdcDO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformUdcValueDO;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformUdcValueRepo;
import com.elitescloud.cloudt.system.convert.UdcConvert;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcValueRespVO;
import com.elitescloud.cloudt.system.model.vo.save.udc.UdcSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.udc.UdcValueSaveVO;
import com.elitescloud.cloudt.system.service.UdcMngService;
import com.elitescloud.cloudt.system.service.repo.UdcRepoProc;
import com.elitescloud.cloudt.system.service.repo.UdcValueRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class UdcMngServiceImpl extends BaseServiceImpl implements UdcMngService {
    private static final UdcConvert CONVERT = UdcConvert.INSTANCE;

    @Autowired
    private UdcRepoProc udcRepoProc;
    @Autowired
    private UdcValueRepoProc udcValueRepoProc;
    @Autowired
    private SysPlatformUdcValueRepo udcValueRepo;
    @Autowired
    private UdcProvider udcProvider;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(UdcSaveVO saveVO) {
        // 查询修改的UDC
        var udcDO = udcRepoProc.get(saveVO.getId());
        if (udcDO == null) {
            return ApiResult.noData();
        }
        var valueDOList = udcValueRepoProc.listByUdc(udcDO.getAppCode(), udcDO.getUdcCode(), true);

        // 检查修改项
        var idsToDel = checkValueForSave(udcDO, valueDOList, saveVO.getValueList());

        // 保存值
        udcValueRepo.saveAll(valueDOList);

        // 删除值
        if (!idsToDel.isEmpty()) {
            udcValueRepoProc.delete(idsToDel);
        }

        // 清理缓存
        clearCache(udcDO.getAppCode(), udcDO.getUdcCode());
        return ApiResult.ok(saveVO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveValue(Long id, UdcValueSaveVO saveVO) {
        // 查询修改的UDC
        var udcDO = udcRepoProc.get(id);
        if (udcDO == null) {
            return ApiResult.noData();
        }
        var valueDOList = udcValueRepoProc.listByUdc(udcDO.getAppCode(), udcDO.getUdcCode(), true);

        // 检查内容
        var valueDO = checkValueForSave(udcDO, valueDOList, saveVO);

        // 保存值
        udcValueRepo.save(valueDO);

        // 清理缓存
        clearCache(udcDO.getAppCode(), udcDO.getUdcCode());
        return ApiResult.ok(valueDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateValueEnabled(Long valueId) {
        var valueDO = udcValueRepoProc.get(valueId);
        if (valueDO == null) {
            // 已不存在
            return ApiResult.ok(valueId);
        }

        if (ObjectUtil.defaultIfNull(valueDO.getAllowDefault(), true)) {
            // 内置的，判断是否可删除
            var udcDO = udcRepoProc.getByAppCodeAndUdcCode(valueDO.getAppCode(), valueDO.getUdcCode());
            if (udcDO == null) {
                return ApiResult.fail("未查询到UDC数据");
            }
            if (!ObjectUtil.defaultIfNull(udcDO.getAllowUpdate(), true)) {
                return ApiResult.fail("不允许修改值");
            }
        }

        // 修改状态
        boolean enabled = valueDO.getAllowStart() == null || !valueDO.getAllowStart();
        udcValueRepoProc.updateEnabled(valueId, enabled);

        // 清理缓存
        clearCache(valueDO.getAppCode(), valueDO.getUdcCode());
        return ApiResult.ok(valueId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> deleteValue(Long valueId) {
        var valueDO = udcValueRepoProc.get(valueId);
        if (valueDO == null) {
            // 已不存在
            return ApiResult.ok(valueId);
        }

        if (ObjectUtil.defaultIfNull(valueDO.getAllowDefault(), true)) {
            // 系统内置，不可删除
            return ApiResult.fail("系统内置的不可删除");
        }

        udcValueRepoProc.delete(valueId);

        // 清理缓存
        clearCache(valueDO.getAppCode(), valueDO.getUdcCode());
        return ApiResult.ok(valueId);
    }

    @Override
    public ApiResult<PagingVO<UdcPageRespVO>> page(UdcPageQueryVO queryVO) {
        var pageData = udcRepoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }

        var pageResult = pageData.map(CONVERT::do2PageRespVO);
        return ApiResult.ok(pageResult);
    }

    @Override
    public ApiResult<UdcDetailRespVO> detail(Long id) {
        return udcRepoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);

                    // 查询值列表
                    var valueDoList = udcValueRepoProc.listByUdc(t.getAppCode(), t.getUdcCode(), true);
                    respVO.setValueList(CONVERT.valueDo2RespVoList(valueDoList));
                    for (UdcValueRespVO udcValueRespVO : respVO.getValueList()) {
                        if (udcValueRespVO.getUdcOrder() == null) {
                            udcValueRespVO.setUdcOrder(0);
                        }
                        udcValueRespVO.setAllowDefault(ObjectUtil.defaultIfNull(udcValueRespVO.getAllowDefault(), true));
                        udcValueRespVO.setAllowStart(ObjectUtil.defaultIfNull(udcValueRespVO.getAllowStart(), true));
                    }

                    return ApiResult.ok(respVO);
                }).orElse(ApiResult.noData());
    }

    private void clearCache(String appCode, String udcCode) {
        udcProvider.clearCache(appCode, udcCode);
    }

    private SysPlatformUdcValueDO checkValueForSave(SysPlatformUdcDO udcDO, List<SysPlatformUdcValueDO> valueDOList, UdcValueSaveVO valueSaveVO) {
        var valueCodes = valueDOList.stream().map(SysPlatformUdcValueDO::getUdcValueCode).collect(Collectors.toSet());
        if (valueSaveVO.getId() == null) {
            // 顺序
            if (valueSaveVO.getUdcOrder() == null) {
                Integer order = valueDOList.isEmpty() ? 1 : ObjectUtil.defaultIfNull(valueDOList.get(valueDOList.size() - 1).getUdcOrder(), 1) + 1;
                valueSaveVO.setUdcOrder(order);
            }
            if (valueSaveVO.getAllowStart() == null) {
                valueSaveVO.setAllowStart(true);
            }
            // 值编码
            if (valueCodes.contains(valueSaveVO.getUdcValueCode())) {
                throw new IllegalArgumentException("值编码已存在");
            }
            Assert.isTrue(ObjectUtil.defaultIfNull(udcDO.getAllowAddValue(), true), "不允许新增值");

            return initNewValue(udcDO, valueSaveVO);
        }

        // 修改值
        SysPlatformUdcValueDO valueDO = valueDOList.stream().filter(t -> t.getId().longValue() == valueSaveVO.getId()).findFirst().orElse(null);
        Assert.notNull(valueDO, "修改的值不存在");
        if (ObjectUtil.defaultIfNull(valueDO.getAllowDefault(), true)) {
            // 内置的值
            Assert.isTrue(valueDO.getUdcValueCode().equals(valueSaveVO.getUdcValueCode()), "系统内置的不可修改值编码");
            if (!ObjectUtil.defaultIfNull(udcDO.getAllowUpdate(), true)) {
                // 不允许编辑
                Assert.isTrue(Objects.equals(valueDO.getUdcValueName(), valueSaveVO.getUdcValueName()), "系统内置的值不可编辑显示名称");
                Assert.isTrue(Objects.equals(valueDO.getAllowStart(), valueSaveVO.getAllowStart()), "系统内置的值不可编辑启用状态");
            }
        } else {
            if (!valueSaveVO.getUdcValueCode().equals(valueDO.getUdcValueCode())) {
                Assert.isTrue(valueCodes.contains(valueSaveVO.getUdcValueCode()), "值编码已存在");
            }
            if (valueSaveVO.getAllowStart() == null) {
                valueSaveVO.setAllowStart(true);
            }
        }

        if (valueSaveVO.getUdcOrder() == null) {
            valueSaveVO.setUdcOrder(ObjectUtil.defaultIfNull(valueDO.getUdcOrder(), 1));
        }

        // 判断名称是否重复
        boolean nameExists = valueDOList.stream().filter(t -> !t.getId().equals(valueSaveVO.getId())).anyMatch(t -> t.getUdcValueName().equals(valueSaveVO.getUdcValueName()));
        Assert.isTrue(!nameExists, "值名称已存在");

        CONVERT.copyUdcValue(valueSaveVO, valueDO);

        return valueDO;
    }

    private Set<Long> checkValueForSave(SysPlatformUdcDO udcDO, List<SysPlatformUdcValueDO> valueDOList, List<UdcValueSaveVO> valueVONewList) {
        Assert.notEmpty(valueVONewList, "值不能为空");

        var valueDoMap = valueDOList.stream()
                .collect(Collectors.toMap(SysPlatformUdcValueDO::getId, t -> t, (t1, t2) -> t1));
        List<SysPlatformUdcValueDO> newValueDoList = new ArrayList<>(16);
        Set<String> codes = new HashSet<>(16);
        Set<String> names = new HashSet<>(16);
        int index = 0;
        for (UdcValueSaveVO saveVO : valueVONewList) {
            index++;
            saveVO.setUdcOrder(ObjectUtil.defaultIfNull(saveVO.getUdcOrder(), index));
            if (saveVO.getAllowStart() == null) {
                saveVO.setAllowStart(true);
            }

            // 值编码不可重复
            Assert.isTrue(!codes.contains(saveVO.getUdcValueCode()), "存在重复的值编码");
            codes.add(saveVO.getUdcValueCode());
            Assert.isTrue(!names.contains(saveVO.getUdcValueName()), "存在重复的值名称");
            names.add(saveVO.getUdcValueName());

            if (saveVO.getId() == null) {
                // 新增值
                Assert.isTrue(!BooleanUtil.isFalse(udcDO.getAllowAddValue()), "不允许新增值");

                var newValue = initNewValue(udcDO, saveVO);
                newValueDoList.add(newValue);
                continue;
            }
            // 修改值
            var valueDO = valueDoMap.get(saveVO.getId());
            Assert.notNull(valueDO, "修改的值不存在");
            if (ObjectUtil.defaultIfNull(valueDO.getAllowDefault(), true)) {
                // 内置的值
                Assert.isTrue(valueDO.getUdcValueCode().equals(saveVO.getUdcValueCode()), "系统内置的不可修改值编码");
                if (!ObjectUtil.defaultIfNull(udcDO.getAllowUpdate(), true)) {
                    // 不允许编辑
                    Assert.isTrue(Objects.equals(valueDO.getUdcValueName(), saveVO.getUdcValueName()), "系统内置的值不可编辑显示名称");
                    Assert.isTrue(Objects.equals(valueDO.getAllowStart(), saveVO.getAllowStart()), "系统内置的值不可编辑启用状态");
                }
            }
            CONVERT.copyUdcValue(saveVO, valueDO);
        }
        valueDOList.addAll(newValueDoList);

        // 获取删除的
        Set<Long> idToDel = new HashSet<>(16);
        for (SysPlatformUdcValueDO valueDO : valueDOList) {
            if (codes.contains(valueDO.getUdcValueCode())) {
                continue;
            }

            Assert.isTrue(ObjectUtil.defaultIfNull(valueDO.getAllowDefault(), true), "系统内置的值不可删除");
            idToDel.add(valueDO.getId());
        }

        return idToDel;
    }

    private SysPlatformUdcValueDO initNewValue(SysPlatformUdcDO udcDO, UdcValueSaveVO saveVO) {
        SysPlatformUdcValueDO valueDO = CONVERT.valueSaveVo2DO(saveVO);
        valueDO.setAppCode(udcDO.getAppCode());
        valueDO.setUdcCode(udcDO.getUdcCode());

        valueDO.setAllowDefault(false);
        return valueDO;
    }
}
