package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.UdcConvert;
import com.elitescloud.cloudt.system.dto.SysUdcDTO;
import com.elitescloud.cloudt.system.model.bo.SysUdcBO;
import com.elitescloud.cloudt.system.model.bo.SysUdcValueBO;
import com.elitescloud.cloudt.system.service.UdcQueryService;
import com.elitescloud.cloudt.system.service.repo.UdcRepoProc;
import com.elitescloud.cloudt.system.service.repo.UdcValueRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/24
 */
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Service
@Log4j2
public class UdcQueryServiceImpl extends BaseServiceImpl implements UdcQueryService {
    private static final UdcConvert CONVERT = UdcConvert.INSTANCE;

    @Autowired
    private UdcRepoProc udcRepoProc;
    @Autowired
    private UdcValueRepoProc udcValueRepoProc;

    @Override
    public ApiResult<SysUdcDTO> get(String appCode, String udcCode) {
        if (CharSequenceUtil.hasBlank(appCode, udcCode)) {
            return ApiResult.fail("应用编码或UDC编码为空");
        }
        // 先查询值
        var valueBoList = udcValueRepoProc.listBoByUdc(appCode, udcCode, false);
        if (valueBoList.isEmpty()) {
            return ApiResult.noData();
        }
        // 查询UDC项
        var udcBo = udcRepoProc.getBoByAppCodeAndUdcCode(appCode, udcCode);
        if (udcBo == null) {
            return ApiResult.noData();
        }

        // 转dto
        var udcDTO = bo2Dto(udcBo, valueBoList);

        return ApiResult.ok(udcDTO);
    }

    @Override
    public ApiResult<List<SysUdcDTO>> listByUdcCode(String appCode, Set<String> udcCodes) {
        if (!StringUtils.hasText(appCode)) {
            return ApiResult.fail("应用编码为空");
        }

        // 查询值
        var valueBoList = udcValueRepoProc.queryBoByUdc(appCode, udcCodes, true);
        if (valueBoList.isEmpty()) {
            return ApiResult.noData();
        }
        // 转换dto
        var udcDtoList = udcValue2Dto(valueBoList);

        return ApiResult.ok(udcDtoList);
    }

    private SysUdcDTO.UdcValue valueBo2Dto(SysUdcValueBO valueBO) {
        SysUdcDTO.UdcValue value = new SysUdcDTO.UdcValue();
        value.setUdcValueCode(valueBO.getUdcValueCode());
        value.setUdcValueName(CharSequenceUtil.blankToDefault(valueBO.getValueAliasName(), valueBO.getUdcValueName()));
        value.setUdcOrder(ObjectUtil.defaultIfNull(valueBO.getUdcOrder(), 0));
        value.setUdcValueDescribe(valueBO.getUdcValueDescribe());
        value.setEnabled(ObjectUtil.defaultIfNull(valueBO.getAllowStart(), true));
        value.setParentUdcValueCode(valueBO.getParentUdcValueCode());

        return value;
    }

    private SysUdcDTO udcBo2Dto(SysUdcBO udcBO) {
        SysUdcDTO udcDTO = new SysUdcDTO();
        udcDTO.setAppCode(udcBO.getAppCode());
        udcDTO.setUdcCode(udcBO.getUdcCode());
        udcDTO.setUdcName(udcBO.getUdcName());
        udcDTO.setUdcDescribe(udcBO.getUdcDescribe());
        udcDTO.setParentUdcCode(udcBO.getParentUdcCode());

        return udcDTO;
    }

    private SysUdcDTO bo2Dto(SysUdcBO udcBO, List<SysUdcValueBO> valueBoList) {
        var udcDto = udcBo2Dto(udcBO);
        var valueDtoList = valueBoList.stream()
                .map(this::valueBo2Dto)
                .sorted(Comparator.comparingInt(SysUdcDTO.UdcValue::getUdcOrder))
                .collect(Collectors.toList());
        udcDto.setValueList(valueDtoList);

        return udcDto;
    }

    private List<SysUdcDTO> udcValue2Dto(List<SysUdcValueBO> valueBos) {
        if (CollectionUtils.isEmpty(valueBos)) {
            return Collections.emptyList();
        }

        List<SysUdcDTO> dtoList = new ArrayList<>(valueBos.size());
        // 根据应用分组
        var udcAppCodeMap = valueBos.stream().collect(Collectors.groupingBy(SysUdcValueBO::getAppCode));
        for (Map.Entry<String, List<SysUdcValueBO>> appCodeEntry : udcAppCodeMap.entrySet()) {
            // 根据UDC编码分组，查询对应的UDC信息
            var udcCodeMap = appCodeEntry.getValue().stream().collect(Collectors.groupingBy(SysUdcValueBO::getUdcCode));
            var udcBoMap = udcRepoProc.queryBoByUdcCode(appCodeEntry.getKey(), udcCodeMap.keySet()).stream()
                    .collect(Collectors.toMap(SysUdcBO::getUdcCode, t -> t, (t1, t2) -> t1));

            var udcDtoList = udcCodeMap.entrySet().stream()
                    .filter(t -> udcBoMap.containsKey(t.getKey()))
                    .map(t -> bo2Dto(udcBoMap.get(t.getKey()), t.getValue())).collect(Collectors.toList());
            dtoList.addAll(udcDtoList);
        }
        return dtoList;
    }
}
