package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.SysSendVerifyCodeVO;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.core.support.verifycode.VerifyCodeManager;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.util.EncryptUtils;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.dto.req.UserCreateDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.user.UserSaveVO;
import com.elitescloud.cloudt.system.param.SysPasswordUpdate;
import com.elitescloud.cloudt.system.service.UserMngService;
import com.elitescloud.cloudt.system.service.model.bo.SysUserTypeBO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.service.param.SysAccountUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysPasswordUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysUserAvatarUpdateParam;
import com.elitescloud.cloudt.system.service.param.SysVerifyCodeParam;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 用户管理服务.
 *
 * @author Kaiser（wang shao）
 * 2022/9/22
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT, supportOperation = true)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class UserMngServiceImpl extends UserBaseServiceImpl implements UserMngService {

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(UserSaveVO saveVO) {
        // 转换参数
        var saveBo = CONVERT.vo2SaveBo(saveVO);
        saveBo.setAreaBO(CONVERT_AREA.vo2Bo(saveVO.getAreaVO()));

        // 保存用户信息
        var user = userManager.upsert(saveBo);
        return ApiResult.ok(user.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(UserCreateDTO basicDTO) {
        var saveBo = CONVERT.dto2SaveBo(basicDTO);

        saveBo.setTerminals(basicDTO.getTerminalsAll());
        if (CollUtil.isEmpty(basicDTO.getUserTypesAll()) && CollUtil.isNotEmpty(basicDTO.getTypesAll())) {
            saveBo.setUserTypes(basicDTO.getTypesAll().stream().map(SysUserTypeBO::new).collect(Collectors.toSet()));
        }
        if (CollUtil.isEmpty(basicDTO.getUserTypesAdd()) && CollUtil.isNotEmpty(basicDTO.getTypesAdd())) {
            saveBo.setUserTypesAdd(basicDTO.getTypesAdd().stream().map(SysUserTypeBO::new).collect(Collectors.toSet()));
        }

        var user = userManager.upsert(saveBo);
        var userId = user.getId();
        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (!hasPermission(currentUser)) {
            return ApiResult.fail("无权限");
        }

        // 查询当前启用状态
        Boolean enabled = null;
        if (!enabledTenant() || currentUser.isOperation()) {
            // 未启用租户或当前是运营机构
            enabled = userRepoProc.getEnabled(id);
        } else {
            enabled = tenantUserRepoProc.getEnabled(currentUser.getTenantId(), id);
        }

        // 修改启用状态
        enabled = enabled != null && !enabled;
        userManager.updateEnabled(id, enabled);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }
        enabled = enabled != null && !enabled;

        userManager.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updatePassword(Long id) {
        userManager.resetPassword(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updatePassword(Long id, String pwd) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }

        if (CharSequenceUtil.isBlank(pwd)) {
            pwd = CommonConstant.INIT_PWD;
        }
        userManager.updatePassword(id, pwd, null, true);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<Long> updatePassword(SysPasswordUpdate passwordUpdate) {
        if (passwordUpdate.getUserId() == null) {
            return ApiResult.fail("用户ID为空");
        }
        var currentPassword = passwordUpdate.getCurrentPassword();
        if (CharSequenceUtil.isBlank(currentPassword)) {
            return ApiResult.fail("当前密码为空");
        }
        try {
            currentPassword = EncryptUtils.decodeBase64(currentPassword);
        } catch (UnsupportedEncodingException e) {
            log.info("解密当前密码异常：{}", currentPassword, e);
            throw new BusinessException("原始密码不正确，加密错误");
        }

        var newPassword = passwordUpdate.getNewPassword();
        if (CharSequenceUtil.isBlank(newPassword)) {
            return ApiResult.fail("新密码为空");
        }
        try {
            newPassword = EncryptUtils.decodeBase64(newPassword);
        } catch (UnsupportedEncodingException e) {
            log.info("解密新密码异常：{}", newPassword, e);
            throw new BusinessException("新密码加密错误");
        }

        userManager.updatePassword(passwordUpdate.getUserId(), newPassword, currentPassword, false);

        return ApiResult.ok(passwordUpdate.getUserId());
    }

    @Override
    public ApiResult<String> sendVerifyCodeForUpdatePassword(SysSendVerifyCodeVO verifyCodeParam) {
        var verifyCode = verifyCodeManager.send(VerifyCodeManager.BUSINESS_TYPE_UPDATE_PWD, verifyCodeParam);
        if (CharSequenceUtil.isNotBlank(verifyCode)) {
            return ApiResult.ok("测试模式验证码：" + verifyCode);
        }

        return ApiResult.ok();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updatePasswordByVerifyCode(SysPasswordUpdateParam updateParam) {
        var verifyMsg = verifyCodeManager.verify(VerifyCodeManager.BUSINESS_TYPE_UPDATE_PWD, updateParam.getAccount(), updateParam.getVerifyCode());
        if (verifyMsg != null) {
            return ApiResult.fail(verifyMsg);
        }

        // 根据账号查询用户ID并修改密码
        var accountType = CharSequenceUtil.blankToDefault(updateParam.getAccountType(), VerifyCodeManager.ACCOUNT_TYPE_MOBILE);
        List<Long> userIdList = null;
        switch (accountType) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                userIdList = userRepoProc.getIdByMobile(updateParam.getAccount());
                break;
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                userIdList = userRepoProc.getIdByEmail(updateParam.getAccount());
                break;
            default:
                return ApiResult.fail("账号类型不支持");
        }

        if (userIdList.isEmpty()) {
            return ApiResult.fail("未查询到账号信息，请确认" + convertAccountType(accountType) + "正确");
        }
        if (userIdList.size() > 1) {
            return ApiResult.fail("查询到多条账号信息，请联系管理员");
        }
        var userId = userIdList.get(0);

        userManager.updatePassword(userId, updateParam.getPassword(), null, false);

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateMobile(Long id, String mobile) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }

        userManager.updateMobile(id, mobile);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEmail(Long id, String email) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }

        userManager.updateEmail(id, email);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<String> sendVerifyCodeForUpdateAccount(SysSendVerifyCodeVO verifyCodeParam) {
        var accountType = CharSequenceUtil.blankToDefault(verifyCodeParam.getAccountType(), VerifyCodeManager.ACCOUNT_TYPE_MOBILE);

        String verifyCode = null;
        switch (accountType) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                verifyCode = verifyCodeManager.send(VerifyCodeManager.BUSINESS_TYPE_UPDATE_MOBILE, verifyCodeParam);
                break;
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                verifyCode = verifyCodeManager.send(VerifyCodeManager.BUSINESS_TYPE_UPDATE_EMAIL, verifyCodeParam);
                break;
            default:
                return ApiResult.fail("不支持的账号类型");
        }
        if (CharSequenceUtil.isNotBlank(verifyCode)) {
            return ApiResult.ok("测试模式验证码：" + verifyCode);
        }

        return ApiResult.ok();
    }

    @Override
    public ApiResult<String> verifyCodeForUpdateAccount(SysVerifyCodeParam codeParam) {
        var accountType = CharSequenceUtil.blankToDefault(codeParam.getAccountType(), VerifyCodeManager.ACCOUNT_TYPE_MOBILE);
        String verifyMsg = null;
        switch (accountType) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                verifyMsg = verifyCodeManager.verify(VerifyCodeManager.BUSINESS_TYPE_UPDATE_MOBILE, codeParam.getAccount(), codeParam.getVerifyCode());
                break;
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                verifyMsg = verifyCodeManager.verify(VerifyCodeManager.BUSINESS_TYPE_UPDATE_EMAIL, codeParam.getAccount(), codeParam.getVerifyCode());
                break;
            default:
                return ApiResult.fail("不支持的账号类型");
        }
        if (verifyMsg != null) {
            return ApiResult.fail(verifyMsg);
        }

        var verifyId = UUID.fastUUID().toString();
        redisUtils.set(verifyId, verifyId, 30, TimeUnit.MINUTES);
        return ApiResult.ok(verifyId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateAccountByVerifyCode(SysAccountUpdateParam updateParam) {
        // 校验信息
        var accountType = CharSequenceUtil.blankToDefault(updateParam.getAccountType(), VerifyCodeManager.ACCOUNT_TYPE_MOBILE);
        String verifyMsg = null;
        switch (accountType) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                verifyMsg = verifyCodeManager.verify(VerifyCodeManager.BUSINESS_TYPE_UPDATE_MOBILE, updateParam.getAccount(), updateParam.getVerifyCode());
                break;
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                verifyMsg = verifyCodeManager.verify(VerifyCodeManager.BUSINESS_TYPE_UPDATE_EMAIL, updateParam.getAccount(), updateParam.getVerifyCode());
                break;
            default:
                return ApiResult.fail("不支持的账号类型");
        }
        if (verifyMsg != null) {
            return ApiResult.fail(verifyMsg);
        }

        var verifyId = (String) redisUtils.get(updateParam.getVerifyId());
        if (!CharSequenceUtil.equals(updateParam.getVerifyId(), verifyId)) {
            return ApiResult.fail("校验超时，请重新验证之前的手机号");
        }

        var user = SecurityContextUtil.currentUser();
        Long userId = user == null ? null : user.getUser().getId();
        if (userId == null) {
            return ApiResult.fail("请重新登录");
        }

        // 校验手机号是否存在
        if (CharSequenceUtil.isBlank(updateParam.getAccountType())) {
            return ApiResult.fail("账号类型为空");
        }
        List<Long> existsUserIdList = null;
        switch (updateParam.getAccountType()) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE: {
                existsUserIdList = userRepoProc.getIdByMobile(updateParam.getAccount());
                break;
            }
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL: {
                existsUserIdList = userRepoProc.getIdByEmail(updateParam.getAccount());
                break;
            }
            default:
                return ApiResult.fail("账号类型不支持");
        }

        if (!existsUserIdList.isEmpty()) {
            if (existsUserIdList.size() == 1) {
                if (existsUserIdList.get(0).longValue() == userId) {
                    // 无需修改
                    return ApiResult.ok();
                }
                return ApiResult.fail("该" + convertAccountType(updateParam.getAccountType()) + "已被其它用户绑定");
            }
            return ApiResult.fail("该" + convertAccountType(updateParam.getAccountType()) + "已被其它用户绑定");
        }

        // 修改账号
        switch (updateParam.getAccountType()) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE: {
                userRepoProc.updateMobile(userId, updateParam.getAccount());
                break;
            }
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL: {
                userRepoProc.updateEmail(userId, updateParam.getAccount());
                break;
            }
            default:
                return ApiResult.fail("账号类型不支持");
        }

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateAvatar(SysUserAvatarUpdateParam updateParam) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (fileService == null) {
            return ApiResult.fail("未启用文件服务");
        }

        if (CharSequenceUtil.isNotBlank(updateParam.getAvatarCode())) {
            if (!fileService.existsAll(updateParam.getAvatarCode()).isSuccess()) {
                return ApiResult.fail("未找到头像文件，请重新上传");
            }
        }

        var userId = currentUser.getUser().getId();
        userRepoProc.updateAvatar(userId, updateParam.getAvatarUrl(), updateParam.getAvatarCode());

        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> clearCasUserId(List<Long> ids) {
        userRepoProc.clearCasUserId(ids);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveUserType(Long id, String userType, String identityId) {
        if (id == null) {
            return ApiResult.fail("账号ID为空");
        }
        if (CharSequenceUtil.isBlank(userType)) {
            return ApiResult.fail("用户类型为空");
        }

        userManager.updateUserType(id, userType, identityId);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteUserType(Long id, String userType, String identityId) {
        if (id == null) {
            return ApiResult.fail("账号ID为空");
        }
        if (CharSequenceUtil.isBlank(userType)) {
            return ApiResult.fail("用户类型为空");
        }

        userManager.deleteUserType(id, userType, identityId);
        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        userManager.delete(id);

        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<UserDetailRespVO> get(Long id) {
        var userDO = userRepoProc.get(id);
        if (userDO == null) {
            return ApiResult.noData();
        }
        var respVO = CONVERT.doToDetailRespVo(userDO);
        respVO.setFullName(userDO.getFullName());
        respVO.setGenderName(userDO.getGenderName());

        // 启用状态
        var tenant = super.currentTenant();
        if (tenant != null) {
            var enabled = tenantUserRepoProc.getEnabled(tenant.getId(), id);
            if (enabled != null) {
                respVO.setEnabled(enabled);
            }
        }

        //所在地
        var areaCodes = Stream.of(userDO.getProvinceCode(), userDO.getCityCode(), userDO.getCountyCode()).filter(StringUtils::hasText).collect(Collectors.toSet());
        if (!areaCodes.isEmpty()) {
            var areaNameMap = areaQueryService.queryNamesByAreaCode(areaCodes).getData();
            if (areaNameMap != null && !areaNameMap.isEmpty()) {
                respVO.setAreaVO(this.convertAreaVO(userDO, areaNameMap));
            }
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<UserPageRespVO>> page(UserPageQueryVO queryVO) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (!hasPermission(currentUser)) {
            return ApiResult.fail("无权限");
        }

        // 查询用户数据
        var userPage = userRepoProc.pageMng(currentUser, queryVO);
        if (userPage.isEmpty()) {
            // 未查询到数据，直接返回
            return ApiResult.ok(PagingVO.empty());
        }
        var userIds = userPage.getRecords().stream().map(SysUserDO::getId).collect(Collectors.toSet());
        // 获取用户类型
        var userTypeMap = convertUserType(currentUser, userIds);

        // do转vo
        var userVoPage = userPage.map(t -> {
            var vo = CONVERT.doToPageRespVo(t);
            vo.setFullName(t.getFullName());
            vo.setGenderName(new Gender(t.getGender()).getDescription());
            vo.setSourceName(new UserSourceType(t.getSourceType()).getDescription());

            var userTypes = userTypeMap.get(t.getId());
            if (CollUtil.isNotEmpty(userTypes)) {
                vo.setUserTypes(userTypes.stream().map(CodeNameParam::getCode).collect(Collectors.toList()));
                vo.setUserTypeNames(userTypes.stream().map(CodeNameParam::getName).collect(Collectors.toList()));
            }
            return vo;
        });

        if (enabledTenant() && !currentUser.isOperation()) {
            // 启用租户时返回租户相关信息
            var tenantUserMap = tenantUserRepoProc.listByUserIds(currentUser.getTenantId(), userIds).stream().collect(Collectors.toMap(SysTenantUserDO::getSysUserId, t -> t, (t1, t2) -> t1));
            userVoPage.each(t -> {
                var tenantUser = tenantUserMap.get(t.getId());
                if (tenantUser == null) {
                    return;
                }
                t.setLastLoginTime(ObjectUtil.defaultIfNull(tenantUser.getLastLoginTime(), t.getLastLoginTime()));
                t.setExpiredTime(ObjectUtil.defaultIfNull(tenantUser.getExpiredTime(), t.getExpiredTime()));
                t.setEnabled(Boolean.TRUE.equals(tenantUser.getEnabled()));
                t.setCreateTime(tenantUser.getBindTime());
            });
        }

        return ApiResult.ok(userVoPage);
    }

    private boolean hasPermission(GeneralUserDetails currentUser) {
        if (currentUser == null) {
            return false;
        }
        return currentUser.isOperation() || currentUser.isTenantAdmin() || currentUser.isTenantOrgAdmin();
    }

    private String convertAccountType(String accountType) {
        switch (accountType) {
            case VerifyCodeManager.ACCOUNT_TYPE_MOBILE:
                return "手机号";
            case VerifyCodeManager.ACCOUNT_TYPE_EMAIL:
                return "邮箱";
            default:
                throw new IllegalArgumentException("暂不支持的账号类型");
        }
    }
}
