package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.convert.UserConvert;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.SysUserTypeDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserListQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserQueryRespVO;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/28
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT, supportOperation = true)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class UserQueryServiceImpl extends UserBaseServiceImpl implements UserQueryService {
    private static final UserConvert CONVERT = UserConvert.INSTANCE;

    @Override
    public ApiResult<Long> getIdByUsername(String username) {
        if (StrUtil.isBlank(username)) {
            return ApiResult.fail("账号为空");
        }

        Long id = userRepoProc.getIdByUsername(username);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<Boolean> existsUsername(String username) {
        if (StrUtil.isBlank(username)) {
            return ApiResult.fail("账号为空");
        }

        boolean exists = userRepoProc.existsUsername(username);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<Boolean> existsMobile(String mobile) {
        if (StrUtil.isBlank(mobile)) {
            return ApiResult.fail("手机号为空");
        }

        boolean exists = userRepoProc.existsMobile(mobile);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<Boolean> existsEmail(String email) {
        if (StrUtil.isBlank(email)) {
            return ApiResult.fail("邮箱为空");
        }

        boolean exists = userRepoProc.existsEmail(email);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<SysUserBasicDTO> getBasicById(Long id) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }

        var userDO = userRepoProc.get(id);
        if (userDO == null) {
            return ApiResult.ok();
        }
        var basic = CONVERT.do2Basic(userDO);
        super.fillTenantUser(basic);

        return ApiResult.ok(basic);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> getBasicById(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var userList = userRepoProc.get(ids).stream()
                .map(CONVERT::do2Basic)
                .collect(Collectors.toList());
        super.fillTenantUser(userList);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> queryUser(UserQueryDTO queryParam) {
        var userList = userRepoProc.queryBasicDto(queryParam);
        super.fillTenantUser(userList);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<PagingVO<UserQueryRespVO>> pageQuery(UserQueryVO queryVO) {
        GeneralUserDetails currentUser = super.currentUser(true);
        var tenantId = super.currentTenantId();
        // 先查询基本信息
        UserPageQueryVO pageQueryVO = convertPageQueryVO(queryVO);
        var userPage = userRepoProc.pageQuery(currentUser, pageQueryVO);
        if (userPage.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }
        var userIds = userPage.getRecords().stream().map(SysUserDO::getId).collect(Collectors.toSet());
        // 设置特殊字段
        var userTypeMap = convertUserType(currentUser, userIds);
        var userVoPage = userPage.map(t -> {
            var vo = CONVERT.doToQueryRespVo(t);
            vo.setFullName(t.getFullName());
            vo.setGenderName(new Gender(t.getGender()).getDescription());
            vo.setSourceName(new UserSourceType(t.getSourceType()).getDescription());

            var userTypes = userTypeMap.get(t.getId());
            if (CollUtil.isNotEmpty(userTypes)) {
                vo.setUserTypes(userTypes.stream().map(CodeNameParam::getCode).collect(Collectors.toList()));
                vo.setUserTypeNames(userTypes.stream().map(CodeNameParam::getName).collect(Collectors.toList()));
            }
            return vo;
        });

        if (!ObjectUtil.defaultIfNull(queryVO.getAllTenant(), false) && enabledTenant()) {
            // 启用租户时返回租户相关信息
            var tenantUserMap = tenantUserRepoProc.listByUserIds(tenantId, userIds).stream().collect(Collectors.toMap(SysTenantUserDO::getSysUserId, t -> t, (t1, t2) -> t1));
            userVoPage.each(t -> {
                var tenantUser = tenantUserMap.get(t.getId());
                if (tenantUser == null) {
                    return;
                }
                t.setLastLoginTime(ObjectUtil.defaultIfNull(tenantUser.getLastLoginTime(), t.getLastLoginTime()));
                t.setExpiredTime(ObjectUtil.defaultIfNull(tenantUser.getExpiredTime(), t.getExpiredTime()));
                t.setEnabled(Boolean.TRUE.equals(t.getEnabled()) && Boolean.TRUE.equals(tenantUser.getEnabled()));
                t.setCreateTime(tenantUser.getBindTime());
            });
        }

        return ApiResult.ok(userVoPage);
    }

    @Override
    public ApiResult<List<UserListRespVO>> listUser(UserListQueryVO queryVO) {
        var userList = userRepoProc.queryUser(queryVO);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<IdNameParam>> queryUserName(Set<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("ID为空");
        }

        var userNames = userRepoProc.queryUserName(ids);
        return ApiResult.ok(userNames);
    }

    @Override
    public ApiResult<List<SysUserTypeDTO>> getUserTypes(Long id) {
        if (id == null) {
            return ApiResult.fail("账号ID为空");
        }

        var tenantId = super.currentTenantId();
        var userTypeList = userTypeRepoProc.getUserTypeList(id, tenantId)
                .stream()
                .map(t -> {
                    SysUserTypeDTO dto = new SysUserTypeDTO();
                    dto.setUserType(t.getType());
                    dto.setIdentityId(t.getIdentityId());

                    return dto;
                }).collect(Collectors.toList());

        return ApiResult.ok(userTypeList);
    }

    private UserPageQueryVO convertPageQueryVO(UserQueryVO queryVO) {
        UserPageQueryVO pageQueryVO = new UserPageQueryVO();
        pageQueryVO.setAllTenant(queryVO.getAllTenant());
        pageQueryVO.setUsername(queryVO.getUsername());
        pageQueryVO.setFullName(queryVO.getFullName());
        pageQueryVO.setMobile(queryVO.getMobile());
        pageQueryVO.setEmail(queryVO.getEmail());
        pageQueryVO.setIncludeSelf(queryVO.getIncludeSelf());
        pageQueryVO.setUserType(queryVO.getUserType());
        pageQueryVO.setUserTypeList(queryVO.getUserTypeList());
        pageQueryVO.setOrders(queryVO.getOrders());
        pageQueryVO.setKeyword(queryVO.getKeyword());
        pageQueryVO.setEnabled(queryVO.getEnabled());
        pageQueryVO.setCurrent(queryVO.getCurrent() + 1);
        pageQueryVO.setSize(queryVO.getSize());

        return pageQueryVO;
    }
}
