package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.convert.EmployeeConvert;
import com.elitescloud.cloudt.system.service.callback.EmployeeChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysEmployeeOrgSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysEmployeeSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysUserSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeOrgDO;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/9
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class EmployeeMngManager {
    private static final EmployeeConvert CONVERT = EmployeeConvert.INSTANCE;

    @Autowired
    private EmployeeRepo employeeRepo;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private EmployeeOrgRepo employeeOrgRepo;
    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private UserTypeRepoProc userTypeRepoProc;

    @Autowired
    private UserMngManager userMngManager;
    @Autowired
    private EmployeeOrgManager employeeOrgManager;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Autowired
    private ObjectProvider<EmployeeChangedCallback> employeeChangedCallbacks;

    /**
     * 新增修改员工
     *
     * @param saveBO 员工信息
     * @return 员工
     */
    public SysEmployeeDO upsert(@NotNull SysEmployeeSaveBO saveBO) {
        // 判断账号是否已与其它员工绑定
        if (isBoundUser(saveBO.getId(), saveBO.getUserId())) {
            throw new IllegalArgumentException("该账号已与其他员工绑定");
        }
        // 保存账号信息
        var userSaveBO = saveBO.getUserSaveBO();
        if (userSaveBO != null) {
            userSaveBO.setMobile(CharSequenceUtil.blankToDefault(userSaveBO.getMobile(), saveBO.getPhone()));
            userSaveBO.setEmail(CharSequenceUtil.blankToDefault(userSaveBO.getEmail(), saveBO.getEmail()));
            var userDO = userMngManager.upsert(userSaveBO);
            saveBO.setUserId(userDO.getId());
            saveBO.setPhone(CharSequenceUtil.blankToDefault(saveBO.getPhone(), userSaveBO.getMobile()));
            saveBO.setEmail(CharSequenceUtil.blankToDefault(saveBO.getEmail(), userSaveBO.getEmail()));
        }

        // 检查员工信息
        boolean isAdd = saveBO.getId() == null;
        SysEmployeeDO employeeOld = isAdd ? null : employeeRepoProc.get(saveBO.getId());
        SysEmployeeDO employeeDO = isAdd ? checkForInsert(saveBO) : checkForUpdate(saveBO, employeeOld);
        this.checkForEmployeeOrg(saveBO, employeeDO);

        if (isAdd) {
            // 判断是否已存在员工信息
            Assert.notNull(saveBO.getUserId(), "未知员工账号信息");
            Long employeeId = employeeRepoProc.getIdByUserId(saveBO.getUserId());
            if (employeeId != null) {
                isAdd = false;
                saveBO.setId(employeeId);
                employeeDO.setId(employeeId);
            }
        }

        // 保存员工信息
        employeeRepo.save(employeeDO);
        userMngManager.updateUserType(employeeDO.getUserId(), UserType.EMPLOYEE.getValue(), employeeDO.getId().toString());

        // 保存员工组织
        saveEmployeeOrg(employeeDO.getId(), saveBO.getOrgInfoList());

        // 保存后的回调
        boolean finalIsAdd = isAdd;
        employeeChangedCallbacks.forEach(t -> t.onUpsert(finalIsAdd, saveBO, employeeDO));

        return employeeDO;
    }
    private boolean isBoundUser(Long userId, Long employeeId) {
        if (userId == null) {
            return false;
        }
        var existsEmployeeId = employeeRepoProc.getIdByUserId(userId);
        if (existsEmployeeId == null) {
            return false;
        }

        // 账号绑定的员工已存在时，必须是当前员工
        return employeeId == null || employeeId.longValue() != existsEmployeeId;
    }

    /**
     * 检查员工信息
     *
     * @param saveBO 员工信息
     */
    public void check(@NotNull SysEmployeeSaveBO saveBO) {
        // 保存账号信息
        if (saveBO.getUserSaveBO() != null) {
            // 判断用户所属租户
            userMngManager.check(saveBO.getUserSaveBO());
        }

        // 检查员工信息
        boolean isAdd = saveBO.getId() == null;
        SysEmployeeDO employeeOld = isAdd ? null : employeeRepoProc.get(saveBO.getId());
        SysEmployeeDO employeeDO = isAdd ? checkForInsert(saveBO) : checkForUpdate(saveBO, employeeOld);
        this.checkForEmployeeOrg(saveBO, employeeDO);

        if (isAdd) {
            // 判断是否已存在员工信息
            Assert.notNull(saveBO.getUserId(), "未知员工账号信息");
        }
    }

    /**
     * 更新员工账号信息
     *
     * @param userSaveBO 账号信息
     * @param orgId      组织ID
     * @return 员工信息
     */
    public SysEmployeeDO upsert(@NotNull SysUserSaveBO userSaveBO, Long orgId) {
//        //是否有权限
//        if (orgId != null) {
//            boolean hasPermission = employeeOrgManager.hasPermissionOfMng(null, Set.of(orgId));
//            Assert.isTrue(hasPermission, "仅可保存在当前所在组织或其下级组织中");
//        }

        // 先保存账号信息
        var userDO = userMngManager.upsert(userSaveBO);

        Long rootOrgId = null;
        // 判断是否已保存员工信息
        var employeeDO = employeeRepoProc.getByUserId(userDO.getId()).orElse(null);
        if (employeeDO == null) {
            // 员工不存在，初始化员工信息
            Assert.notNull(orgId, "组织ID为空");
            rootOrgId = orgRepoProc.getRootId(orgId);
            Assert.notNull(rootOrgId, "未查询到根组织");

            employeeDO = new SysEmployeeDO();
            employeeDO.setUserId(userDO.getId());
            employeeDO.setRootOrgId(rootOrgId);
            employeeDO.setUsername(userDO.getUsername());
            employeeDO.setGender(userDO.getGender());
            employeeDO.setFirstName(userDO.getFirstName());
            employeeDO.setLastName(userDO.getLastName());
            employeeDO.setEmail(CharSequenceUtil.blankToDefault(userSaveBO.getEmail(), ""));
            employeeDO.setPhone(CharSequenceUtil.blankToDefault(userSaveBO.getMobile(), ""));
            employeeDO.setJoinTime(LocalDateTime.now());
            employeeDO.setEnabled(false);
            employeeDO.setSortNo(1);
            employeeRepo.save(employeeDO);

            // 保存后的回调
            SysEmployeeDO finalEmployeeDO = employeeDO;
            employeeChangedCallbacks.forEach(t -> t.onUpsert(true, null, finalEmployeeDO));
        } else {
            employeeDO.setUsername(userDO.getUsername());
            employeeDO.setGender(userDO.getGender());
            employeeDO.setFirstName(userDO.getFirstName());
            employeeDO.setLastName(userDO.getLastName());
            employeeDO.setEmail(CharSequenceUtil.blankToDefault(userSaveBO.getEmail(), ""));
            employeeDO.setPhone(CharSequenceUtil.blankToDefault(userSaveBO.getMobile(), ""));
            employeeRepo.save(employeeDO);
        }

        // 保存员工与组织关联
        if (orgId != null) {
            addEmployeeOrg(employeeDO.getId(), orgId, rootOrgId);
        }

        return employeeDO;
    }

    /**
     * 保存员工所属组织
     *
     * @param employeeId  员工ID
     * @param orgInfoList 所属组织
     */
    public void saveEmployeeOrg(@NotNull Long employeeId, List<SysEmployeeOrgSaveBO> orgInfoList) {
        // 是否有权限
        if (orgInfoList == null) {
            orgInfoList = Collections.emptyList();
        }

        // 组织ID
        var orgIds = orgInfoList.stream().map(SysEmployeeOrgSaveBO::getOrgId).filter(Objects::nonNull).collect(Collectors.toSet());
        // 获取已有组织
        var orgOldMap = employeeOrgRepoProc.getEmployeeOrgByEmployeeId(employeeId).stream().collect(Collectors.toMap(SysEmployeeOrgDO::getOrgId, t -> t, (t1, t2) -> t1));
        var orgOldIds = orgOldMap.keySet();

//        var orgIdsChanged = this.obtainOrgIdsOfChanged(orgIds, orgOldIds);
//        if (CollUtil.isNotEmpty(orgIdsChanged) && SecurityContextUtil.currentUser() != null) {
//            boolean hasPermission = employeeOrgManager.hasPermissionOfMng(null, orgIdsChanged);
//            Assert.isTrue(hasPermission, "仅可保存在当前所在组织或其下级组织中");
//        }

        // 获取员工的根组织
        var rootOrgId = employeeRepoProc.getRootOrgId(employeeId);
        Assert.notNull(rootOrgId, "未知员工的根组织");

        // 需要清空员工的组织
        if (orgInfoList.isEmpty()) {
            var orgIdsToDel = orgOldIds.stream().filter(t -> t.longValue() != rootOrgId).collect(Collectors.toSet());
            employeeOrgRepoProc.delete(employeeId, orgIdsToDel);
            return;
        }

        // 需要添加和修改的组织
        Long userId = employeeRepoProc.getUserId(employeeId);
        LocalDateTime nowTime = LocalDateTime.now();
        List<SysEmployeeOrgDO> empOrgListToSave = new ArrayList<>();
        Set<Long> existsOrgIds = new HashSet<>();
        for (SysEmployeeOrgSaveBO orgInfo : orgInfoList) {
            Assert.isTrue(!existsOrgIds.contains(orgInfo.getOrgId()), "存在重复的员工组织");
            existsOrgIds.add(orgInfo.getOrgId());
            // 获取老的
            SysEmployeeOrgDO employeeOrgDO = orgOldMap.get(orgInfo.getOrgId());
            if (employeeOrgDO != null) {
                if (Objects.equals(employeeOrgDO.getLeaderUserId(), orgInfo.getLeaderUserId())) {
                    // 已存在，无需变动
                    continue;
                }
                employeeOrgDO.setUserId(userId);
                // 领导变动
                employeeOrgDO.setLeaderUserId(orgInfo.getLeaderUserId());
                if (orgInfo.getLeaderUserId() != null) {
                    employeeOrgDO.setLeaderEmployeeId(employeeRepoProc.getIdByUserId(orgInfo.getLeaderUserId()));
                } else {
                    employeeOrgDO.setLeaderEmployeeId(null);
                }
                empOrgListToSave.add(employeeOrgDO);
                continue;
            }

            // 需要新增
            employeeOrgDO = new SysEmployeeOrgDO();
            employeeOrgDO.setUserId(userId);
            employeeOrgDO.setEmployeeId(employeeId);
            employeeOrgDO.setRootOrgId(rootOrgId);
            employeeOrgDO.setOrgId(orgInfo.getOrgId());
            employeeOrgDO.setLeaderUserId(orgInfo.getLeaderUserId());
            if (orgInfo.getLeaderUserId() != null) {
                employeeOrgDO.setLeaderEmployeeId(employeeRepoProc.getIdByUserId(orgInfo.getLeaderUserId()));
            }
            employeeOrgDO.setJoinTime(nowTime);
            empOrgListToSave.add(employeeOrgDO);
        }
        if (!empOrgListToSave.isEmpty()) {
            employeeOrgRepo.saveAll(empOrgListToSave);
        }

        // 需要删除的组织
        Set<Long> orgIdsToDel = orgOldIds.stream()
                .filter(t -> !orgIds.contains(t))
                .collect(Collectors.toSet());
        if (!orgIdsToDel.isEmpty()) {
            employeeOrgRepoProc.delete(employeeId, orgIdsToDel);
        }
    }

    /**
     * 新增员工所属组织
     *
     * @param employeeId 员工ID
     * @param orgId      组织ID
     */
    public void addEmployeeOrg(@NotNull Long employeeId, @NotNull Long orgId) {
//        if (orgId != null) {
//            boolean hasPermission = employeeOrgManager.hasPermissionOfMng(null, Set.of(orgId));
//            Assert.isTrue(hasPermission, "仅可保存在当前所在组织或其下级组织中");
//        }

        addEmployeeOrg(employeeId, orgId, null);
    }

    /**
     * 移除员工的指定组织
     *
     * @param employeeId 员工ID
     * @param orgId      组织ID
     */
    public void removeEmployeeOrg(@NotNull Long employeeId, @NotNull Long orgId) {
        Long rootOrgId = employeeRepoProc.getRootOrgId(employeeId);
        if (rootOrgId == null) {
            return;
        }
        if (rootOrgId.longValue() == orgId) {
            // 根组织，不能直接移除
            return;
        }

//        boolean hasPermission = employeeOrgManager.hasPermissionOfMng(null, Set.of(orgId));
//        Assert.isTrue(hasPermission, "仅可保存在当前所在组织或其下级组织中");

        employeeOrgRepoProc.delete(employeeId, orgId);
    }

    /**
     * 更新员工的启用状态
     *
     * @param employeeId 员工ID
     * @param enabled    启用状态
     */
    public void updateEnabled(Long employeeId, Boolean enabled) {
        if (enabled == null) {
            enabled = true;
        }
        var userId = employeeRepoProc.getUserId(employeeId);
        if (userId == null) {
            return;
        }
        employeeRepoProc.updateEnabled(employeeId, enabled);

        // 更新用户状态
        userMngManager.updateEnabled(userId, enabled);

        // 回调任务
        Boolean finalEnabled = enabled;
        employeeChangedCallbacks.forEach(t -> t.onEnabled(employeeId, finalEnabled));
    }

    /**
     * 更新手机号
     *
     * @param id         员工ID
     * @param mobile     手机号
     * @param updateUser 是否更新账号的
     */
    public void updateMobile(Long id, String mobile, Boolean updateUser) {
        var userId = employeeRepoProc.getUserId(id);
        Assert.notNull(userId, "账号不存在");

        employeeRepoProc.updateMobile(id, mobile);
        if (Boolean.TRUE.equals(updateUser)) {
            userMngManager.updateMobile(userId, mobile);
        }
    }

    /**
     * 更新邮箱
     *
     * @param id         员工ID
     * @param email      邮箱
     * @param updateUser 是否更新账号的
     */
    public void updateEmail(Long id, String email, Boolean updateUser) {
        var userId = employeeRepoProc.getUserId(id);
        Assert.notNull(userId, "账号不存在");

        employeeRepoProc.updateEmail(id, email);
        if (Boolean.TRUE.equals(updateUser)) {
            userMngManager.updateEmail(userId, email);
        }
    }

    /**
     * 删除员工
     *
     * @param employeeId 员工ID
     */
    public void deleteEmployee(Long employeeId) {
        if (!employeeRepoProc.exists(employeeId)) {
            // 数据不存在
            return;
        }
        var userId = employeeRepoProc.getUserId(employeeId);

        // 删除员工信息
        employeeRepoProc.delete(employeeId);
        // 删除员工与组织的关系
        employeeOrgRepoProc.deleteByEmployeeId(employeeId);
        employeeOrgRepoProc.deleteByLeaderEmployeeId(employeeId);
        // 删除员工类型
        var tenant = tenantClientProvider.getSessionTenant();
        var tenantId = tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            userTypeRepoProc.delete(userId, Set.of(UserType.EMPLOYEE.getValue()), tenantId);
            return null;
        });

        // 删除租户用户关联，避免删除员工后依然可登录
        userMngManager.delete(userId);

        // 回调任务
        employeeChangedCallbacks.forEach(t -> t.onDelete(employeeId));
    }

    /**
     * 根据账号信息删除员工
     *
     * @param userId 账号ID
     */
    public void deleteEmployeeByUser(Long userId) {
        var employeeId = employeeRepoProc.getIdByUserId(userId);
        if (employeeId == null) {
            return;
        }

        // 删除员工信息
        employeeRepoProc.delete(employeeId);
        // 删除员工与组织的关系
        employeeOrgRepoProc.deleteByEmployeeId(employeeId);
        employeeOrgRepoProc.deleteByLeaderEmployeeId(employeeId);
    }

    private SysEmployeeDO checkForInsert(SysEmployeeSaveBO saveBO) {
        var employeeDO = CONVERT.saveBO2Do(saveBO);

        // 判断用户
        var userInfo = userRepoCall(repo -> repo.getBasicDto(saveBO.getUserId()));
        Assert.notNull(userInfo, "用户不存在");
        employeeDO.setUsername(userInfo.getUsername());
        employeeDO.setGender(userInfo.getGender());
        employeeDO.setFirstName(userInfo.getFirstName());
        employeeDO.setLastName(userInfo.getLastName());

        // 判断账号是否已绑定员工
        String code = employeeRepoProc.getCodeByUserId(userInfo.getId());
        Assert.isTrue(CharSequenceUtil.isBlank(code), "该账号已绑定员工：" + code);

        // 员工编号
        Assert.hasText(saveBO.getCode(), "员工编号为空");
        boolean exists = employeeRepoProc.existsCode(saveBO.getCode());
        Assert.isTrue(!exists, "员工编号已存在");

        // 工作邮箱
        if (!StringUtils.hasText(saveBO.getEmail())) {
            employeeDO.setEmail(CharSequenceUtil.blankToDefault(saveBO.getUserSaveBO().getEmail(), ""));
        }
        if (!StringUtils.hasText(saveBO.getPhone())) {
            employeeDO.setPhone(CharSequenceUtil.blankToDefault(saveBO.getUserSaveBO().getMobile(), ""));
        }

        employeeDO.setJoinTime(ObjectUtil.defaultIfNull(saveBO.getJoinTime(), LocalDateTime.now()));
        if (employeeDO.getEnabled() == null) {
            employeeDO.setEnabled(true);
        }
        if (employeeDO.getServed() == null) {
            employeeDO.setServed(true);
        }
        if (employeeDO.getSortNo() == null) {
            employeeDO.setSortNo(1);
        }

        return employeeDO;
    }

    private SysEmployeeDO checkForUpdate(SysEmployeeSaveBO saveBO, SysEmployeeDO employeeOld) {
        Assert.notNull(employeeOld, "员工信息不存在");

        var employeeDO = employeeRepoProc.get(saveBO.getId());
        CONVERT.saveBO2DO(saveBO, employeeDO);

        // 判断用户账号
        if (employeeDO.getUserId() == null) {
            employeeDO.setUserId(employeeOld.getUserId());
        } else if (employeeDO.getUserId().longValue() != employeeOld.getUserId()) {
            throw new BusinessException("员工账号信息不一致");
        }
        if (saveBO.getUserSaveBO() != null) {
            employeeDO.setUsername(saveBO.getUserSaveBO().getUsername());
            employeeDO.setGender(saveBO.getUserSaveBO().getGender());
            employeeDO.setLastName(saveBO.getUserSaveBO().getFullName());
        }

        boolean exists = false;
        // 员工编号
        Assert.hasText(employeeDO.getCode(), "员工编号为空");
        if (!employeeDO.getCode().equals(employeeOld.getCode())) {
            exists = employeeRepoProc.existsCode(employeeDO.getCode());
            Assert.isTrue(!exists, "员工编号已存在");
        }

        // 工作邮箱
        if (!StringUtils.hasText(employeeDO.getEmail())) {
            employeeDO.setEmail(CharSequenceUtil.blankToDefault(saveBO.getUserSaveBO().getEmail(), ""));
        }
        if (!StringUtils.hasText(saveBO.getPhone())) {
            employeeDO.setPhone(CharSequenceUtil.blankToDefault(saveBO.getUserSaveBO().getMobile(), ""));
        }

        if (employeeDO.getJoinTime() == null) {
            employeeDO.setJoinTime(employeeOld.getJoinTime());
        }
        if (employeeDO.getEnabled() == null) {
            employeeDO.setEnabled(true);
        }
        if (employeeDO.getServed() == null) {
            employeeDO.setServed(true);
        }
        if (employeeDO.getSortNo() == null) {
            employeeDO.setSortNo(0);
        }

        return employeeDO;
    }

    private void checkForEmployeeOrg(SysEmployeeSaveBO saveBO, SysEmployeeDO employeeDO) {
        Assert.notEmpty(saveBO.getOrgInfoList(), "员工的组织不能为空");

        var orgId = saveBO.getOrgInfoList().stream().map(SysEmployeeOrgSaveBO::getOrgId).filter(Objects::nonNull).findAny().orElse(null);
        Assert.notNull(orgId, "请选择员工的组织");
        var rootOrgId = orgRepoProc.getRootId(orgId);
        Assert.notNull(rootOrgId, "未知组织所属根组织");
        employeeDO.setRootOrgId(rootOrgId);
    }

    private void addEmployeeOrg(Long employeeId, Long orgId, Long rootOrgId) {
        // 查询根组织ID
        if (rootOrgId == null) {
            rootOrgId = orgRepoProc.getRootId(orgId);
            Assert.notNull(rootOrgId, "根组织不存在");
        }

        // 查询目前员工已绑定组织
        var orgIds = employeeOrgRepoProc.getOrgIdByEmployeeId(employeeId);
        if (orgIds.contains(orgId)) {
            // 已存在，无需添加
            return;
        }

        // 获取员工的用户ID
        var userId = employeeRepoProc.getUserId(employeeId);

        // 添加员工组织
        LocalDateTime nowTime = LocalDateTime.now();
        SysEmployeeOrgDO employeeOrgDO = new SysEmployeeOrgDO();
        employeeOrgDO.setUserId(userId);
        employeeOrgDO.setEmployeeId(employeeId);
        employeeOrgDO.setRootOrgId(rootOrgId);
        employeeOrgDO.setOrgId(orgId);
        employeeOrgDO.setJoinTime(nowTime);

        employeeOrgRepo.save(employeeOrgDO);
    }

    private Set<Long> obtainOrgIdsOfChanged(Collection<Long> orgIds1, Collection<Long> orgIds2) {
        if (CollectionUtils.isEmpty(orgIds1)) {
            return new HashSet<>(ObjectUtil.defaultIfNull(orgIds2, Collections.emptySet()));
        }
        if (CollectionUtils.isEmpty(orgIds2)) {
            return new HashSet<>(ObjectUtil.defaultIfNull(orgIds1, Collections.emptySet()));
        }

        Set<Long> result = new HashSet<>(64);
        for (Long id : orgIds1) {
            if (!orgIds2.contains(id)) {
                result.add(id);
            }
        }
        for (Long id : orgIds2) {
            if (!orgIds1.contains(id)) {
                result.add(id);
            }
        }
        return result;
    }

    /**
     * 用户信息在默认schema下
     *
     * @param function
     * @param <T>
     * @return
     */
    private <T> T userRepoCall(Function<UserRepoProc, T> function) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> function.apply(userRepoProc));
    }
}
