package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.OrgConvert;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.service.callback.OrgChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysOrgSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysOrgDO;
import com.elitescloud.cloudt.system.service.repo.EmployeeOrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepo;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/29
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class OrgMngManager {
    private static final OrgConvert CONVERT = OrgConvert.INSTANCE;

    @Autowired
    private OrgRepo orgRepo;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private ObjectProvider<OrgChangedCallback> orgChangedCallbacks;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private TaskExecutor taskExecutor;

    /**
     * 新增、修改组织
     *
     * @param saveBO 组织信息
     * @return 组织信息
     */
    public SysOrgDO upsert(@NotNull SysOrgSaveBO saveBO) {
        // 检查保存参数
        boolean isAdd = saveBO.getId() == null;
        SysOrgDO orgDO = isAdd ? this.checkForInsert(saveBO) : this.checkForUpdate(saveBO);
        Boolean executiveOld = isAdd ? null : orgRepoProc.getExecutive(orgDO.getId());

        // 保存组织信息
        orgRepo.save(orgDO);

        // 保存后的回调
        orgChangedCallbacks.forEach(t -> t.onUpsert(isAdd, saveBO, orgDO));

        // 保存树节点
        saveOrgTreeNode(orgDO, executiveOld);

        return orgDO;
    }

    /**
     * 更新组织的启用状态
     *
     * @param orgId   组织ID
     * @param enabled 启用状态
     */
    public void updateEnabled(@NotNull Long orgId, Boolean enabled) {
        if (enabled == null) {
            enabled = true;
        }
        orgRepoProc.updateEnabled(orgId, enabled);

        Boolean finalEnabled = enabled;
        orgChangedCallbacks.forEach(t -> t.onEnabled(orgId, finalEnabled));
    }

    /**
     * 删除组织数据
     *
     * @param orgId 组织ID
     */
    public void delete(@NotNull Long orgId) {
        var org = orgRepoProc.get(orgId);
        if (org == null) {
            return;
        }

        if (orgRepoProc.existsChildNode(org)) {
            throw new BusinessException("请先删除子组织");
        }

        // 先删除树节点
        if (Boolean.TRUE.equals(org.getExecutive())) {
            orgRepoProc.removeTreeNode(org);
        }
        // 再删除组织数据
        orgRepoProc.delete(orgId);
        // 删除组织与员工关联
        employeeOrgRepoProc.deleteByOrgId(orgId);

        // 删除后的回调
        orgChangedCallbacks.forEach(t -> t.onDelete(orgId));
    }

    /**
     * 重建行政组织树
     *
     * @param rootOrgId 根节点ID
     */
    public void rebuildTree(Long rootOrgId) {
        var codeIdMap = orgRepoProc.queryCodeName(rootOrgId, true).stream().collect(Collectors.toMap(IdCodeNameParam::getCode,
                IdCodeNameParam::getId, (t1, t2) -> t1));
        if (codeIdMap.isEmpty()) {
            // 没有数据
            return;
        }
        // 开始重构
        orgRepoProc.rebuildTree(rootOrgId, t -> codeIdMap.get(t.getParentCode()));
    }

    private void saveOrgTreeNode(SysOrgDO orgDO, Boolean executiveOld) {
        // 需要修改为非行政组织
        if (Boolean.FALSE.equals(orgDO.getExecutive())) {
            if (executiveOld == null) {
                // 新增时，无需操作
                return;
            }

            // 修改时，如果之前是行政组织，则需要删除树节点
            if (Boolean.TRUE.equals(executiveOld)) {
                // 之前是行政组织，则需要删除节点
                orgRepoProc.removeTreeNode(orgDO);

                // 重构组织树
                var rootId = orgDO.getRootId();
                CompletableFuture.runAsync(() -> {
                    this.rebuildTree(rootId);
                }, taskExecutor);
                return;
            }
        }

        // 父ID
        long parentId = BaseTreeModel.DEFAULT_PARENT;
        if (StringUtils.hasText(orgDO.getParentCode())) {
            var parentOrg = orgRepoProc.getByCode(orgDO.getParentCode());
            Assert.notNull(parentOrg, "上级组织不存在");
            Assert.isTrue(Boolean.TRUE.equals(parentOrg.getExecutive()), "行政组织的上级必须是行政组织");
            parentId = parentOrg.getId();
        }

        // 保存树节点
        orgRepoProc.saveTreeNode(orgDO, parentId, orgDO.getSortNo());

        // 重构组织树
        var rootId = orgDO.getRootId();
        CompletableFuture.runAsync(() -> {
            this.rebuildTree(rootId);
        }, taskExecutor);
    }

    private SysOrgDO checkForInsert(SysOrgSaveBO saveBO) {
        var orgDo = CONVERT.saveBo2Do(saveBO);

        Assert.notNull(saveBO.getExecutive(), "未知组织是否为行政组织");

        // 组织编号
        Assert.hasText(saveBO.getCode(), "组织编号为空");
        boolean exists = orgRepoProc.existsCode(saveBO.getCode());
        Assert.isTrue(!exists, "组织编号已存在");

        if (saveBO.getType() == null) {
            saveBO.setType("");
        }

        orgDo.setEntity(ObjectUtil.defaultIfNull(saveBO.getEntity(), true));
        orgDo.setSortNo(ObjectUtil.defaultIfNull(saveBO.getSortNo(), 0));
        orgDo.setEnabled(ObjectUtil.defaultIfNull(saveBO.getEnabled(), true));

        // 上级组织不能是自己
        Assert.isTrue(!saveBO.getCode().equals(saveBO.getParentCode()), "上级组织不能选择组织本身及下级组织");

        // 新增组织
        if (Boolean.TRUE.equals(orgDo.getExecutive())) {
            // 行政根组织只能有一个
            if (CharSequenceUtil.isBlank(orgDo.getParentCode())) {
                // 限制只能有一个根组织
                if (Boolean.TRUE.equals(systemProperties.getLimitOrgTreeOne())) {
                    var existsRootOrgId = orgRepoProc.getRootOrgId();
                    Assert.isNull(existsRootOrgId, "行政根组织只能有一个");
                }
            }
        }

        return orgDo;
    }

    private SysOrgDO checkForUpdate(SysOrgSaveBO saveBO) {
        var orgDo = orgRepoProc.get(saveBO.getId());
        Assert.notNull(orgDo, "组织信息不存在");

        Assert.notNull(saveBO.getExecutive(), "未知组织是否为行政组织");

        boolean exists = false;
        // 组织编号
        Assert.hasText(saveBO.getCode(), "组织编号为空");
        if (systemProperties.getOrgCodeEditable()) {
            // 组织编号可编辑
            if (!saveBO.getCode().equals(orgDo.getCode())) {
                exists = orgRepoProc.existsCode(saveBO.getCode());
                Assert.isTrue(!exists, "组织编号已存在");
            }
        } else {
            Assert.isTrue(saveBO.getCode().equals(orgDo.getCode()), "组织编号不可修改");
        }

        if (saveBO.getType() == null) {
            saveBO.setType("");
        }

        // 上级组织不能是自己
        Assert.isTrue(!saveBO.getCode().equals(saveBO.getParentCode()), "上级组织不能选择组织本身及下级组织");
        // 限制上级节点不能是自己的子节点
        if(!Objects.equals(orgDo.getParentCode(), saveBO.getParentCode())) {
            // 行政根组织只能有一个
            if (CharSequenceUtil.isBlank(saveBO.getParentCode())) {
                // 限制只能有一个根组织
                if (Boolean.TRUE.equals(systemProperties.getLimitOrgTreeOne())) {
                    var existsRootOrgId = orgRepoProc.getRootOrgId();
                    Assert.isNull(existsRootOrgId, "行政根组织只能有一个");
                }
            } else {
                var parentId = orgRepoProc.getIdByCode(saveBO.getParentCode());
                Assert.notNull(parentId, "上级组织不存在");
                var isChild = orgRepoProc.isChildNode(orgDo.getId(), parentId);
                Assert.isTrue(!isChild, "上级组织不能选择组织本身及下级组织");
            }
        }

        saveBO.setEntity(ObjectUtil.defaultIfNull(saveBO.getEntity(), true));
        saveBO.setSortNo(ObjectUtil.defaultIfNull(saveBO.getSortNo(), 0));
        saveBO.setEnabled(ObjectUtil.defaultIfNull(saveBO.getEnabled(), true));

        CONVERT.copySaveBo2Do(saveBO, orgDo);

        return orgDo;
    }
}
