package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.convert.RoleConvert;
import com.elitescloud.cloudt.system.convert.SysDpcRoleApiFieldsConvert;
import com.elitescloud.cloudt.system.convert.SysDprRoleApiRuleConvert;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.bo.MenuSimpleBO;
import com.elitescloud.cloudt.system.model.bo.RoleApiRuleBO;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.permission.DataPermissionTreeNodeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RolePermissionRespVO;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.bo.*;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.elitescloud.cloudt.system.service.repo.*;
import lombok.extern.log4j.Log4j2;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 权限管理.
 *
 * @author Kaiser（wang shao）
 * 2022/10/16
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class PermissionMngManager {

    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private RolePermissionRepo rolePermissionRepo;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private TenantMenuTreeRepoProc tenantMenuTreeRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private MenuRepoProc menuRepoProc;
    @Autowired
    private ApiRepoProc apiRepoProc;
    @Autowired
    private MenuApiRepoProc menuApiRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc roleApiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;

    @Autowired
    private RoleMngManager roleManager;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Autowired
    private RedissonClient redissonClient;
    @Autowired
    private TaskExecutor taskExecutor;

    /**
     * 保存用户角色
     *
     * @param userId  用户ID
     * @param roleIds 角色ID
     */
    public void saveUserRole(@NotNull Long userId, List<Long> roleIds) {
        var userRoles = roleManager.callByType((roleType, typeId) -> userRoleRepoProc.getByUser(userId, roleType.getValue(), typeId));
        if (CollectionUtils.isEmpty(roleIds)) {
            // 没有指定角色，则删除所有
            var ids = userRoles.stream().map(SysUserRoleDO::getId).collect(Collectors.toList());
            if (!ids.isEmpty()) {
                userRoleRepoProc.delete(ids);
            }
            return;
        }

        // 过滤存在的角色ID
        roleIds = roleRepoProc.exists(roleIds);
        if (roleIds.isEmpty()) {
            throw new BusinessException("部分角色不存在");
        }

        var userRoleMap = userRoles.stream().collect(Collectors.toMap(SysUserRoleDO::getRoleId, t -> t, (t1, t2) -> t1));
        // 新增的角色
        var userRoleToAdd = roleIds.stream()
                .filter(t -> !userRoleMap.containsKey(t))
                .map(t -> {
                    SysUserRoleDO userRoleDO = new SysUserRoleDO();
                    userRoleDO.setRoleId(t);
                    userRoleDO.setUserId(userId);
                    return userRoleDO;
                }).collect(Collectors.toList());
        if (!userRoleToAdd.isEmpty()) {
            userRoleRepoProc.save(userRoleToAdd);
        }

        // 需要删除的
        List<Long> finalRoleIds = roleIds;
        var idsToDel = userRoles.stream()
                .filter(t -> !finalRoleIds.contains(t.getRoleId()))
                .map(SysUserRoleDO::getId)
                .collect(Collectors.toList());
        if (!idsToDel.isEmpty()) {
            userRoleRepoProc.delete(idsToDel);
        }
    }

    /**
     * 获取用户角色
     *
     * @param userId  用户ID
     * @param onlyOwn 是否仅仅返回用户所拥有的
     * @return 角色列表
     */
    public List<IdCodeNameCheckParam> getUserRole(@NotNull Long userId, boolean onlyOwn) {
        return roleManager.callByType((roleType, typeId) -> userRoleRepoProc.getByUser(userId, roleType.getValue(), typeId, onlyOwn));
    }

    /**
     * 保存角色与菜单的关联
     * <p>
     * 包含按钮等
     *
     * @param roleId     角色ID
     * @param saveVOList 权限信息
     */
    public void saveRoleMenu(@NotNull Long roleId, List<SysAppPermissionSaveBO> saveVOList) {
        boolean hasCustom = hasCustomMenuTree(null);

        saveRolePermission(roleId, saveVOList, List.of(MenuTreeNodeType.APP, MenuTreeNodeType.MENU, MenuTreeNodeType.ACTION), hasCustom);
    }

    /**
     * 根据角色获取权限菜单
     * <p>
     * 如果传递了appCode，则仅查询对应应用下的权限菜单，否则查询所有（或租户所拥有）得权限菜单
     *
     * @param roleId        角色ID
     * @param appCode       应用编码，如果为空则是租户的所有应用
     * @param includeApp    是否返回应用
     * @param includeAction 是否返回按钮
     * @param tree          是否返回树形数据
     * @return 权限菜单列表
     */
    public List<RolePermissionRespVO> getPermissionMenuByRole(@NotNull Long roleId, String appCode, boolean includeApp,
                                                              boolean includeAction, boolean tree) {
        // 获取有效应用
        var currentTenant = tenantClientProvider.getCurrentTenant();
        Set<String> appCodes = null;
        if (currentTenant != null) {
            if (CollectionUtils.isEmpty(currentTenant.getAppCodes())) {
                // 租户尚未分配应用
                return Collections.emptyList();
            }
            if (StringUtils.hasText(appCode) && !currentTenant.getAppCodes().contains(appCode)) {
                // 没有分配对应应用
                return Collections.emptyList();
            }
            appCodes = StringUtils.hasText(appCode) ? Set.of(appCode) : currentTenant.getAppCodes();
        } else {
            appCodes = StringUtils.hasText(appCode) ? Set.of(appCode) : queryAppAll(true).stream()
                    .map(IdCodeNameParam::getCode).collect(Collectors.toSet());
        }

        var tenantId = currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
        // 菜单
        var custom = this.hasCustomMenuTree(tenantId);
        List<RolePermissionRespVO> respVOList = this.queryAppAndMenu(tenantId, appCodes, custom, includeApp);
        if (respVOList.isEmpty()) {
            return Collections.emptyList();
        }
        // 挂载的接口
        this.fillApiNumOfMenu(appCodes, respVOList);

        // 按钮
        if (includeAction) {
            var menuCodes = respVOList.stream().filter(t -> MenuTreeNodeType.MENU.getValue().equals(t.getNodeType())).map(RolePermissionRespVO::getCode).collect(Collectors.toSet());
            if (!menuCodes.isEmpty()) {
                var actionList = this.queryActionList(appCodes).stream().filter(t -> StringUtils.hasText(t.getParentCode()) && menuCodes.contains(t.getParentCode())).collect(Collectors.toList());
                respVOList.addAll(actionList);
            }
        }
        // 查询已分配的权限
        var checkCodes = rolePermissionRepoProc.getCodeByRole(roleId, Set.of(MenuTreeNodeType.MENU.getValue(), MenuTreeNodeType.ACTION.getValue()), custom);
        return convertPermissionTreeRespVO(respVOList, checkCodes, tree);
    }

    /**
     * 根据角色获取数据权限
     *
     * @param roleId     角色ID
     * @param appCode    应用编码
     * @param includeApp 是否返回应用
     * @param tree       是否返回树状数据
     * @return 权限列表
     */
    public List<DataPermissionTreeNodeRespVO> getDataPermissionByRole(Long roleId, String appCode, boolean includeApp,
                                                                      boolean tree) {
        // 获取有效应用
        var currentTenant = tenantClientProvider.getCurrentTenant();
        Set<String> appCodes;
        if (currentTenant != null) {
            if (CollectionUtils.isEmpty(currentTenant.getAppCodes())) {
                // 租户尚未分配应用
                return Collections.emptyList();
            }
            if (StringUtils.hasText(appCode) && !currentTenant.getAppCodes().contains(appCode)) {
                // 没有分配对应应用
                return Collections.emptyList();
            }
            appCodes = StringUtils.hasText(appCode) ? Set.of(appCode) : currentTenant.getAppCodes();
        } else {
            appCodes = StringUtils.hasText(appCode) ? Set.of(appCode) : queryAppAll(true).stream()
                    .map(IdCodeNameParam::getCode).collect(Collectors.toSet());
        }

        // 查询接口
        var menuApiMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuApiRepoProc.queryApiDetailOfMenu(appCodes));
        if (menuApiMap.isEmpty()) {
            return Collections.emptyList();
        }

        var tenantId = currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
        // 菜单
        var custom = this.hasCustomMenuTree(tenantId);
        List<RolePermissionRespVO> respVOList = this.queryAppAndMenu(tenantId, appCodes, custom, includeApp);
        if (respVOList.isEmpty()) {
            return Collections.emptyList();
        }
        // 分配的权限规则
        List<RoleApiRuleBO> roleApiRuleList = roleId == null ? Collections.emptyList() : roleApiRowRuleRepoProc.queryByRole(roleId);
        var treeNodeList = respVOList.stream().map(this::convertDataPermissionTreeNode).collect(Collectors.toList());
        treeNodeList.addAll(this.convertDataPermissionTreeNode(menuApiMap, roleApiRuleList));

        return this.convertDataPermissionTreeRespVO(treeNodeList, tree);
    }

    /**
     * 保存数据权限
     *
     * @param saveBO 数据权限信息
     */
    public void saveDataPermission(SysDprSaveBO saveBO) {
        AtomicLong appId = new AtomicLong();
        AtomicReference<List<IdCodeNameParam>> apiList = new AtomicReference<>();
        tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var tempAppId = menuRepoProc.getAppIdByMenuCode(saveBO.getMenuCode());
            Assert.notNull(tempAppId, "菜单信息异常，请稍后再试");
            appId.set(tempAppId);
            var apis = menuApiRepoProc.queryApiOfMenu(saveBO.getMenuCode()).stream()
                    .map(t -> {
                        IdCodeNameParam param = new IdCodeNameParam();
                        param.setId(t.getId());
                        param.setCode(t.getApiCode());
                        param.setName(t.getApiName());

                        return param;
                    }).collect(Collectors.toList());
            apiList.set(apis);
            return null;
        });
        var apiCodesAll = apiList.get().stream().map(IdCodeNameParam::getCode).collect(Collectors.toSet());

        // 保存自定义规则
        this.saveDataPermissionCustomRule(appId.get(), saveBO, DprRuleRelationEnum.DPR_RULE_RELATION_AND, apiCodesAll);

        // 保存字段权限
        this.saveDataPermissionField(appId.get(), saveBO);
    }

    /**
     * 保存批量数据权限
     *
     * @param saveBO 数据权限信息
     */
    public void saveDataPermissionBatch(SysDprBatchSaveBO saveBO) {
        // 查询菜单信息
        var menuCodes = saveBO.getMenuApiList().stream().map(SysDprBatchSaveBO.MenuApiSaveBO::getMenuCode).collect(Collectors.toList());
        Map<String, MenuSimpleBO> menuMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.listSimpleByMenuCode(menuCodes))
                .stream().collect(Collectors.toMap(MenuSimpleBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 保存
        Map<String, Long> existsData = null;
        if (SysDprBatchSaveBO.Strategy.APPEND == saveBO.getStrategy()) {
            // 直接追加的，无需做什么
        } else if (SysDprBatchSaveBO.Strategy.REPLACE == saveBO.getStrategy()) {
            // 替换的，先查询已有数据
            existsData = roleApiRowRuleRepoProc.listByRole(saveBO.getRoleId()).stream()
                    .collect(Collectors.toMap(t -> t.getMenuCode() + ":" + t.getApiCode() + ":" + t.getDprRuleField(),
                            SysDprRoleApiRowRuleDO::getId, (t1, t2) -> t1));
        } else if (SysDprBatchSaveBO.Strategy.CLEAR == saveBO.getStrategy()) {
            // 清空已有数据
            for (SysDprBatchSaveBO.MenuApiSaveBO menuApiSaveBO : saveBO.getMenuApiList()) {
                roleApiRowRuleRepoProc.deleteByApi(saveBO.getRoleId(), menuApiSaveBO.getMenuCode(), menuApiSaveBO.getApiCode());
            }
        }

        List<SysDprRoleApiRowRuleDO> ruleList = new ArrayList<>(16);
        SysDprRoleApiRowRuleDO ruleDO = null;
        for (SysDprBatchSaveBO.MenuApiSaveBO menuApiSaveBO : saveBO.getMenuApiList()) {
            SysDprSaveBO dprSaveBO = new SysDprSaveBO();
            dprSaveBO.setRoleId(saveBO.getRoleId());
            dprSaveBO.setMenuCode(menuApiSaveBO.getMenuCode());
            dprSaveBO.setApiCode(menuApiSaveBO.getApiCode());
            dprSaveBO.setCustomRuleList(saveBO.getCustomRuleList());

            var menu = menuMap.get(menuApiSaveBO.getMenuCode());
            Assert.notNull(menu, "菜单不存在");
            int order = 0;
            for (SysDprApiCustomRuleSaveBO customRuleSaveBO : saveBO.getCustomRuleList()) {
                ruleDO = this.convertDprRoleApiRowRuleDO(menu.getMenusAppCode(), dprSaveBO, customRuleSaveBO, order++, DprRuleRelationEnum.DPR_RULE_RELATION_AND);

                if (existsData != null) {
                    var id = existsData.get(menuApiSaveBO.getMenuCode() + ":" + menuApiSaveBO.getApiCode() + ":" + customRuleSaveBO.getDprRuleField());
                    ruleDO.setId(id);
                }

                ruleList.add(ruleDO);
            }
        }

        if (!ruleList.isEmpty()) {
            roleApiRowRuleRepoProc.save(ruleList);
        }
    }

    /**
     * 复制角色权限
     *
     * @param originalId 原始角色ID
     * @param roleId     新角色ID
     */
    public void cloneRolePermission(@NotNull Long originalId, @NotNull Long roleId) {
        // 先复制角色权限信息
        this.cloneRoleMenu(originalId, roleId);
        // 保存数据权限
        this.cloneRoleDataPermission(originalId, roleId);
    }

    private List<DataPermissionTreeNodeRespVO> convertDataPermissionTreeNode(Map<String, List<SysMenuApiRespVO>> menuApiMap,
                                                                             List<RoleApiRuleBO> roleApiRuleList) {
        var roleApiMap = roleApiRuleList.stream().collect(Collectors.groupingBy(t -> t.getMenuCode() + ":" + t.getApiCode(), Collectors.counting()));
        List<DataPermissionTreeNodeRespVO> respVoList = new ArrayList<>();
        DataPermissionTreeNodeRespVO respVO = null;
        for (var entry : menuApiMap.entrySet()) {
            int i = 0;
            var menuCode = entry.getKey();
            for (var param : entry.getValue()) {
                respVO = new DataPermissionTreeNodeRespVO();
                respVO.setNodeType(MenuTreeNodeType.API.getValue());
                respVO.setRuleNum(roleApiMap.getOrDefault(menuCode + ":" + param.getCode(), 0L));
                respVO.setCode(param.getCode());
                respVO.setName(param.getName());
                respVO.setSortNo(i++);
                respVO.setParentCode(menuCode);
                respVO.setHasChildren(false);
                respVO.setChildren(Collections.emptyList());

                respVO.setApiDetail(param);

                respVoList.add(respVO);
            }
        }
        return respVoList;
    }

    private DataPermissionTreeNodeRespVO convertDataPermissionTreeNode(RolePermissionRespVO rolePermissionRespVO) {
        DataPermissionTreeNodeRespVO treeNodeRespVO = new DataPermissionTreeNodeRespVO();
        treeNodeRespVO.setRoute(rolePermissionRespVO.getRoute());
        treeNodeRespVO.setNodeType(rolePermissionRespVO.getNodeType());
        treeNodeRespVO.setRuleNum(0L);
        treeNodeRespVO.setCode(rolePermissionRespVO.getCode());
        treeNodeRespVO.setName(rolePermissionRespVO.getName());
        treeNodeRespVO.setSortNo(ObjectUtil.defaultIfNull(rolePermissionRespVO.getSortNo(), 0));
        treeNodeRespVO.setParentId(null);
        treeNodeRespVO.setParentCode(rolePermissionRespVO.getParentCode());
        treeNodeRespVO.setHasChildren(false);
        treeNodeRespVO.setChildren(new ArrayList<>(0));

        return treeNodeRespVO;
    }

    private void cloneRoleDataPermission(@NotNull Long originalId, @NotNull Long roleId) {
        // 行权限
        var roleDataPermissionList = roleApiRowRuleRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var ruleDO = SysDprRoleApiRuleConvert.INSTANCE.do2Do(t);
                    ruleDO.setId(null);
                    ruleDO.setRoleId(roleId);
                    return ruleDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(roleDataPermissionList)) {
            roleApiRowRuleRepoProc.save(roleDataPermissionList);
        }

        // 列权限
        var roleColumnPermissionList = apiFieldsRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var permissionDO = SysDpcRoleApiFieldsConvert.INSTANCE.do2Do(t);
                    permissionDO.setId(null);
                    permissionDO.setRoleId(roleId);
                    return permissionDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(roleColumnPermissionList)) {
            apiFieldsRepoProc.save(roleColumnPermissionList);
        }
    }

    private void cloneRoleMenu(Long originalId, Long roleId) {
        var rolePermissionList = rolePermissionRepoProc.listByRole(originalId).stream()
                .map(t -> {
                    var permissionDO = RoleConvert.INSTANCE.do2do(t);
                    permissionDO.setId(null);
                    permissionDO.setRoleId(roleId);
                    return permissionDO;
                }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(rolePermissionList)) {
            rolePermissionRepoProc.save(rolePermissionList);
        }
    }

    private Long currentTenantId() {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        return currentTenant == null ? TenantConstant.DEFAULT_TENANT_ID : currentTenant.getId();
    }

    private boolean hasCustomMenuTree(Long tenantId) {
        if (tenantId == null) {
            GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
            tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        }

        var custom = tenantMenuRepoProc.getEnabled(tenantId);
        return custom != null && custom;
    }

    private Set<String> getApps() {
        var currentTenant = tenantClientProvider.getCurrentTenant();
        if (currentTenant != null) {
            return currentTenant.getAppCodes();
        }
        return queryAppAll(true).stream().map(IdCodeNameParam::getCode).collect(Collectors.toSet());
    }

    private List<IdCodeNameParam> queryAppAll(Boolean enabled) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allParams(enabled));
    }

    private SysDprRoleApiRowRuleDO convertDprRoleApiRowRuleDO(String appCode, SysDprSaveBO saveBO, SysDprApiCustomRuleSaveBO ruleBO,
                                                              int ruleOrder, DprRuleRelationEnum relation) {
        SysDprRoleApiRowRuleDO ruleDO = new SysDprRoleApiRowRuleDO();
        ruleDO.setRoleId(saveBO.getRoleId());
        ruleDO.setAppCode(appCode);
        ruleDO.setMenuCode(saveBO.getMenuCode());
        ruleDO.setApiCode(saveBO.getApiCode());
        ruleDO.setRuleOrder(ruleOrder);
        ruleDO.setDprRuleRelation(relation.name());
        ruleDO.setDprRuleRelationName(relation.getValueDescription());
        ruleDO.setDprRuleName(ruleBO.getDprRuleName());
        ruleDO.setDprRuleDeclare(ruleBO.getDprRuleDeclare());
        ruleDO.setDprRuleField(ruleBO.getDprRuleField());
        if (ruleBO.getDprRuleCondition() != null) {
            ruleDO.setDprRuleCondition(ruleBO.getDprRuleCondition().name());
            ruleDO.setDprRuleConditionName(ruleBO.getDprRuleCondition().getValueDescription());
        }

        var valueType = ruleBO.getDprRuleValueType();
        Assert.notNull(valueType, "取值方式为空");
        if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM != valueType
                && DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS != valueType
                && DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS != valueType) {
            throw new IllegalArgumentException("暂不支持的取值方式：" + valueType.getValueDescription());
        }
        ruleDO.setDprRuleValueType(valueType.name());
        ruleDO.setDprRuleValueTypeName(valueType.getValueDescription());
        ruleDO.setDprRuleValue(ruleBO.getDprRuleValue());
        ruleDO.setDprRuleValueName(ruleBO.getDprRuleValueName());
        ruleDO.setRoleRuleValueType(valueType.name());
        ruleDO.setDataSet(ruleBO.getDataSet());
        ruleDO.setRoleRuleValue(ruleBO.getDprRuleValue());
        if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS == valueType) {
            ruleDO.setDprSysInternally(ruleBO.getDprRuleValue());
            ruleDO.setDprSysInternallyName(DprSysInternallyEnum.valueOf(ruleBO.getDprRuleValue()).getValueDescription());
        }
        ruleDO.setDataRange(false);
        ruleDO.setBs1(ruleBO.getBs1());
        ruleDO.setBs2(ruleBO.getBs2());
        ruleDO.setBs3(ruleBO.getBs3());
        return ruleDO;
    }

    private void saveDataPermissionCustomRule(Long appId, SysDprSaveBO saveBO, DprRuleRelationEnum relation, Set<String> apiCodesAll) {
        String appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getCode(appId));
        List<SysDprRoleApiRowRuleDO> rowRuleDOList = new ArrayList<>(saveBO.getCustomRuleList().size());
        // 数据范围
        int order = 0;
        SysDprRoleApiRowRuleDO ruleDO = null;
        if (saveBO.getRange() != null) {
            ruleDO = new SysDprRoleApiRowRuleDO();
            ruleDO.setRoleId(saveBO.getRoleId());
            ruleDO.setAppCode(appCode);
            ruleDO.setMenuCode(saveBO.getMenuCode());
            ruleDO.setApiCode(saveBO.getApiCode());
            ruleDO.setRuleOrder(order++);
            ruleDO.setDprRuleRelation(relation.name());
            ruleDO.setDprRuleRelationName(relation.getValueDescription());
            ruleDO.setDprRuleName("数据范围");
            ruleDO.setDprRuleValueType(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name());
            ruleDO.setDprRuleValueTypeName(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.getValueDescription());
            ruleDO.setDprSysInternally(saveBO.getRange().name());
            ruleDO.setDprSysInternallyName(saveBO.getRange().getValueDescription());
            ruleDO.setDataRange(true);
            ruleDO.setBs1(null);
            ruleDO.setBs2(null);
            ruleDO.setBs3(null);
            rowRuleDOList.add(ruleDO);
        }

        // 自定义规则
        for (SysDprApiCustomRuleSaveBO bo : saveBO.getCustomRuleList()) {
            ruleDO = this.convertDprRoleApiRowRuleDO(appCode, saveBO, bo, order++, relation);

            rowRuleDOList.add(ruleDO);
        }
        // 先删除
        roleApiRowRuleRepoProc.deleteByApi(saveBO.getRoleId(), saveBO.getMenuCode(), saveBO.getApiCode());
        // 删除菜单下已不存在的api
        roleApiRowRuleRepoProc.deleteForNotExistsApi(saveBO.getMenuCode(), apiCodesAll);
        // 再新增
        if (!rowRuleDOList.isEmpty()) {
            roleApiRowRuleRepoProc.save(rowRuleDOList);
        }
    }

    private void saveDataPermissionField(Long appId, SysDprSaveBO saveBO) {
        String appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getCode(appId));
        List<SysDpcrApiFieldsDO> fieldDoList = new ArrayList<>(saveBO.getFieldList().size());
        SysDpcrApiFieldsDO fieldsDO = null;
        for (SysDprApiFieldSaveBO bo : saveBO.getFieldList()) {
            fieldsDO = new SysDpcrApiFieldsDO();
            fieldsDO.setRoleId(saveBO.getRoleId());
            fieldsDO.setAppCode(appCode);
            fieldsDO.setMenuCode(saveBO.getMenuCode());
            fieldsDO.setApiCode(saveBO.getApiCode());
            fieldsDO.setFieldName(bo.getFieldName());
            fieldsDO.setFieldRemark(bo.getFieldRemark());
            fieldsDO.setFieldApiVisible(ObjectUtil.defaultIfNull(bo.isFieldApiVisible(), false));
            fieldsDO.setFieldFormVisible(ObjectUtil.defaultIfNull(bo.isFieldFormVisible(), false));
            fieldsDO.setFieldFormUpdate(ObjectUtil.defaultIfNull(bo.isFieldFormUpdate(), false));

            fieldDoList.add(fieldsDO);
        }
        // 先删除
        apiFieldsRepoProc.deleteByApi(saveBO.getRoleId(), saveBO.getMenuCode(), saveBO.getApiCode());
        if (!fieldDoList.isEmpty()) {
            // 再保存
            apiFieldsRepoProc.save(fieldDoList);
        }
    }

    private List<RolePermissionRespVO> queryAppAndMenu(Long tenantId, Set<String> appCodes, boolean custom, boolean includeApp) {
        // 查询自定义菜单树
        if (custom) {
            return queryCustomMenuTree(tenantId, appCodes);
        }

        // 查询默认的
        return queryDefaultMenuTree(includeApp, appCodes);
    }

    private <T> List<T> filterChildrenForTree(List<T> respVoList, Function<T, List<T>> childrenGetter, BiConsumer<T, List<T>> childrenSetter,
                                              Predicate<T> predicate) {
        if (CollUtil.isEmpty(respVoList)) {
            return Collections.emptyList();
        }

        List<T> dataList = new ArrayList<>();
        for (T t : respVoList) {
            var result = predicate.test(t);
            if (result) {
                dataList.add(t);
            }

            // 判断子节点是否有满足的
            var children = filterChildrenForTree(childrenGetter.apply(t), childrenGetter, childrenSetter, predicate);
            childrenSetter.accept(t, children);
            if (!result && CollUtil.isNotEmpty(children)) {
                dataList.add(t);
            }
        }
        return dataList;
    }

    private List<DataPermissionTreeNodeRespVO> convertDataPermissionTreeRespVO(List<DataPermissionTreeNodeRespVO> respVoList, boolean tree) {
        TreeDataUtil<DataPermissionTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(respVoList, DataPermissionTreeNodeRespVO::getCode,
                DataPermissionTreeNodeRespVO::getParentCode, DataPermissionTreeNodeRespVO::setChildren, Comparator.comparingInt(DataPermissionTreeNodeRespVO::getSortNo));
        respVoList = (List<DataPermissionTreeNodeRespVO>) treeDataUtil.getRoots();

        // 过滤出挂载有api的
        respVoList = filterChildrenForTree(respVoList, DataPermissionTreeNodeRespVO::getChildren, DataPermissionTreeNodeRespVO::setChildren,
                t -> MenuTreeNodeType.API.getValue().equals(t.getNodeType()));
        return tree ? respVoList : CollectionUtil.expandTree(respVoList, DataPermissionTreeNodeRespVO::getChildren);
    }

    private List<RolePermissionRespVO> convertPermissionTreeRespVO(List<RolePermissionRespVO> respVOList, List<String> checked, boolean tree) {
        // 设置已选择项
        for (RolePermissionRespVO respVO : respVOList) {
            if (checked.contains(respVO.getCode())) {
                respVO.setChecked(true);
            }

            if (respVO.getSortNo() == null) {
                respVO.setSortNo(1);
            }
        }

        if (!tree) {
            return respVOList;
        }

        TreeDataUtil<RolePermissionRespVO> treeDataUtil = new TreeDataUtil<>(respVOList, RolePermissionRespVO::getCode,
                RolePermissionRespVO::getParentCode, RolePermissionRespVO::setChildren, Comparator.comparingInt(RolePermissionRespVO::getSortNo));

        return (List<RolePermissionRespVO>) treeDataUtil.getRoots();
    }

    private List<RolePermissionRespVO> queryActionList(Set<String> appCodes) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryActionByAppCode(appCodes, true)).stream()
                .filter(t -> (t.getMenusState() == null || t.getMenusState()) && StringUtils.hasText(t.getMenusParentCode()))
                .map(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setCode(t.getMenusCode());
                    respVO.setName(t.getMenusName());
                    respVO.setNodeType(MenuTreeNodeType.ACTION.getValue());
                    respVO.setParentCode(t.getMenusParentCode());
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setRoute(t.getMenusRoute());
                    respVO.setChecked(false);
                    respVO.setSortNo(t.getMenusOrder());
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(8));

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<RolePermissionRespVO> queryApiList(Set<String> appCodes) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> apiRepoProc.queryByAppCode(appCodes)).stream()
                .map(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setCode(t.getPermissonCode());
                    respVO.setName(t.getPermissonName());
                    respVO.setNodeType(MenuTreeNodeType.API.getValue());
                    respVO.setParentCode(t.getMenusCode());
                    respVO.setChecked(false);
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(8));

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<RolePermissionRespVO> queryCustomMenuTree(Long tenantId, Set<String> appCodes) {
        // 查询菜单设置
        var menuTreeDOList = tenantMenuTreeRepoProc.queryByTenantId(tenantId);
        if (menuTreeDOList.isEmpty()) {
            // 尚未设置
            return Collections.emptyList();
        }

        // 查询默认菜单
        var menuDOMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenuByAppCode(appCodes, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }
                    return true;
                }).collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));
        var allApps = queryAppAll(true).stream().collect(Collectors.toMap(IdCodeNameParam::getCode, t -> t, (t1, t2) -> t1));

        // 组织vo
        List<RolePermissionRespVO> menuVOList = new ArrayList<>(menuTreeDOList.size());
        RolePermissionRespVO menuVO = null;
        for (SysTenantMenuTreeDO menuTreeDO : menuTreeDOList) {
            menuVO = new RolePermissionRespVO();
            menuVO.setCode(menuTreeDO.getMenuCode());
            menuVO.setName(menuTreeDO.getMenuName());

            // 设置节点原始信息
            var nodeType = MenuTreeNodeType.valueOf(menuTreeDO.getNodeType());
            if (nodeType == null) {
                log.error("未知菜单节点类型：" + menuTreeDO.getNodeType());
                continue;
            }
            if (MenuTreeNodeType.MENU == nodeType) {
                MenuBO menuBO = menuDOMap.get(menuTreeDO.getMenuCode());
                if (menuBO == null) {
                    // 是否是自定义菜单
                    if (!menuTreeDO.getCustom()) {
                        log.info("菜单不存在：" + menuTreeDO.getMenuCode());
                        continue;
                    }
                } else {
                    if (CharSequenceUtil.isBlank(menuVO.getName())) {
                        menuVO.setName(menuBO.getMenusName());
                    }

                    if (menuBO.getMenusState() != null && !menuBO.getMenusState()) {
                        // 菜单已禁用
                        continue;
                    }
                    menuVO.setRoute(menuBO.getMenusRoute());
                }
            } else if (MenuTreeNodeType.APP == nodeType) {
                var app = allApps.get(menuTreeDO.getMenuCode());
                if (app == null) {
                    log.info("应用已不存在：{}", menuTreeDO.getMenuCode());
                    continue;
                }
                if (CharSequenceUtil.isBlank(menuVO.getName())) {
                    menuVO.setName(app.getName());
                }
            } else {
                log.error("暂不支持的节点类型：{}", nodeType.getValue());
                continue;
            }
            menuVO.setNodeType(menuTreeDO.getNodeType());
            menuVO.setParentCode(menuTreeDO.getParentMenuCode());
            menuVO.setMenusIcon(menuTreeDO.getIcon());
            menuVO.setChecked(false);
            menuVO.setSortNo(menuTreeDO.getSortNo());
            menuVO.setHasChildren(false);
            menuVO.setChildren(new ArrayList<>(128));

            menuVOList.add(menuVO);
        }

        return menuVOList;
    }

    private List<RolePermissionRespVO> queryDefaultMenuTree(boolean includeApp, Set<String> appCodes) {
        List<RolePermissionRespVO> respVOList = includeApp ? convertMenuTreeRespVO(appCodes) : new ArrayList<>(128);
        if (includeApp && respVOList.isEmpty()) {
            return Collections.emptyList();
        }

        // 获取菜单
        List<RolePermissionRespVO> finalRespVOList = respVOList;
        tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenuByAppCode(appCodes, true)).stream()
                .filter(t -> {
                    // 过滤业务菜单
                    if (!PlatformAppMenusTypeEnum.MENUS_TYPE_BUS.name().equals(t.getMenusType())) {
                        return false;
                    }

                    return true;
                })
                .forEach(t -> {
                    RolePermissionRespVO respVO = new RolePermissionRespVO();
                    respVO.setCode(t.getMenusCode());
                    respVO.setName(t.getMenusName());
                    respVO.setNodeType(MenuTreeNodeType.MENU.getValue());
                    respVO.setParentCode(t.getMenusParentCode());
                    if (CharSequenceUtil.isBlank(t.getMenusParentCode())) {
                        // 顶级的设置应用编码
                        respVO.setParentCode(t.getMenusAppCode());
                    }
                    respVO.setMenusIcon(t.getMenusIcon());
                    respVO.setRoute(t.getMenusRoute());
                    respVO.setChecked(false);
                    respVO.setSortNo(t.getMenusOrder());
                    respVO.setHasChildren(false);
                    respVO.setChildren(new ArrayList<>(128));

                    finalRespVOList.add(respVO);
                });

        return respVOList;
    }

    private List<RolePermissionRespVO> convertMenuTreeRespVO(Set<String> appCodes) {
        // 先获取应用
        var allApps = queryAppAll(true);
        if (allApps.isEmpty()) {
            throw new BusinessException("未获到有效的应用列表");
        }

        List<RolePermissionRespVO> respVOList = new ArrayList<>(256);
        if (CollectionUtils.isEmpty(appCodes)) {
            // 所有应用
            RolePermissionRespVO respVO = null;
            int sortNo = 1;
            for (IdCodeNameParam app : allApps) {
                respVO = new RolePermissionRespVO();
                respVO.setCode(app.getCode());
                respVO.setName(app.getName());
                respVO.setNodeType(MenuTreeNodeType.APP.getValue());
                respVO.setChecked(false);
                respVO.setSortNo(sortNo++);
                respVO.setHasChildren(false);
                respVO.setChildren(new ArrayList<>(128));

                respVOList.add(respVO);
            }
            return respVOList;
        }

        RolePermissionRespVO respVO = null;
        int sortNo = 1;
        for (IdCodeNameParam app : allApps) {
            if (!appCodes.contains(app.getCode())) {
                // 不包含当前应用
                continue;
            }
            respVO = new RolePermissionRespVO();
            respVO.setCode(app.getCode());
            respVO.setName(app.getName());
            respVO.setNodeType(MenuTreeNodeType.APP.getValue());
            respVO.setChecked(false);
            respVO.setSortNo(sortNo++);
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(128));

            respVOList.add(respVO);
        }

        return respVOList;
    }

    private void fillApiNumOfMenu(Set<String> appCodes, List<RolePermissionRespVO> respVOList) {
        if (CollectionUtils.isEmpty(respVOList)) {
            return;
        }

        // 查询菜单下的api数量
        var apiNumMap = tenantDataIsolateProvider.byDefaultDirectly(() -> menuApiRepoProc.queryNumOfApi(appCodes));
        if (apiNumMap.isEmpty()) {
            return;
        }
        for (RolePermissionRespVO respVO : respVOList) {
            if (MenuTreeNodeType.MENU.getValue().equals(respVO.getNodeType())) {
                respVO.setApiNum(apiNumMap.get(respVO.getCode()));
            }
        }
    }

    private void saveRolePermission(Long roleId, List<SysAppPermissionSaveBO> saveVOList, List<MenuTreeNodeType> nodeTypes, boolean custom) {
        Set<String> permissionType = nodeTypes.stream().map(MenuTreeNodeType::getValue).collect(Collectors.toSet());

        CompletableFuture.runAsync(() -> {
                    saveVOList.forEach(saveBO -> {
                        RLock redLock = null;
                        try {
                            redLock = redissonClient.getLock("cloudt_sys_role_permission:" + roleId + ":" + saveBO.getAppCode());
                            if (redLock.tryLock(30, TimeUnit.SECONDS)) {
                                // 先删除应用下所有
                                rolePermissionRepoProc.deleteByRole(roleId, saveBO.getAppCode(), permissionType, custom);
                                if (CollectionUtils.isEmpty(saveBO.getPermissionList())) {
                                    return;
                                }

                                var doList = saveBO.getPermissionList().stream()
                                        .map(t -> {
                                            SysRolePermissionDO permissionDO = new SysRolePermissionDO();
                                            permissionDO.setRoleId(roleId);
                                            permissionDO.setPermissionCode(t.getCode());
                                            permissionDO.setAppCode(saveBO.getAppCode());
                                            permissionDO.setPermissionType(t.getNodeType());
                                            permissionDO.setCustom(custom);

                                            return permissionDO;
                                        }).collect(Collectors.toList());
                                rolePermissionRepoProc.save(doList);
                            }
                        } catch (Exception e) {
                            log.error("保存角色权限异常：{}，{}", roleId, saveBO.getAppCode(), e);
                            throw new BusinessException("当前角色操作人过多，请稍后再试", e);
                        } finally {
                            if (redLock != null) {
                                redLock.unlock();
                                log.info("释放保存权限锁：{}，{}", roleId, saveBO.getAppCode());
                            }
                        }
                    });
                }, taskExecutor)
                .whenComplete((r, e) -> {
                    if (e == null) {
                        log.info("角色权限保存成功：{}", roleId);
                        return;
                    }
                    log.error("角色权限保存失败：{}", roleId, e);
                });
    }
}
