package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.zhxu.bs.BeanSearcher;
import cn.zhxu.bs.FieldOps;
import cn.zhxu.bs.util.MapUtils;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprSysInternallyEnum;
import com.elitescloud.boot.provider.PlatformAppProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.security.common.InnerRole;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.model.constant.PlatformAdminTypeEnum;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.security.common.InnerUserEnum;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.convert.SysDpcRoleApiFieldsConvert;
import com.elitescloud.cloudt.system.convert.SysDprRoleApiRuleConvert;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiRowColumnRuleDTO;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.bo.PermissionBO;
import com.elitescloud.cloudt.system.model.bo.PermissionMenuBO;
import com.elitescloud.cloudt.system.model.bo.PermissionParameterBO;
import com.elitescloud.cloudt.system.model.vo.resp.index.UserFieldRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.index.UserMenuRespVO;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataColumnRuleListQueryBean;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataRuleListQueryBean;
import com.elitescloud.cloudt.system.rpc.dpr.RoleAppApiDataPermissionRpcServiceUtil;
import com.elitescloud.cloudt.system.rpc.dpr.strategy.IDprSysInternallyStrategy;
import com.elitescloud.cloudt.system.service.common.constant.MenuTreeNodeType;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantMenuTreeDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.spi.DprSysRuleSysInternallyValueSpi;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 权限查询相关接口.
 *
 * @author Kaiser（wang shao）
 * 2022/10/19
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@Log4j2
public class PermissionQueryManager {

    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private TenantMenuRepoProc tenantMenuRepoProc;
    @Autowired
    private MenuRepoProc menuRepoProc;
    @Autowired
    private AdminMenuRepoProc adminMenuRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private ApiManageRepoProc apiRepoProc;
    @Autowired
    private ApiParameterRepoProc apiParameterRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc dpcrApiFieldsRepoProc;

    @Autowired
    private PlatformAppProvider appProvider;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private AuthorizationProperties authorizationProperties;

    @Autowired
    private BeanSearcher beanSearcher;
    @Autowired
    private DprSysRuleSysInternallyValueSpi valueServiceSpi;

    /**
     * 获取用户的角色编码
     *
     * @param userDTO 用户
     * @return 角色编码
     */
    public Set<String> queryRoleByUser(SysUserDTO userDTO) {
        if (userDTO == null) {
            // 未登录
            return Set.of(InnerRole.ANONYMOUS.getValue());
        }

        Set<String> roles = new HashSet<>(16);
        // 内置角色
        // 系统管理员
        if (InnerUserEnum.ADMIN.getUsername().equals(userDTO.getUsername())) {
            roles.add(InnerRole.SYSTEM_ADMIN.getValue());
        }
        if (userDTO.getSysTenantVO() != null && Objects.equals(userDTO.getSysTenantVO().getSysUserId(), userDTO.getId())) {
            roles.add(InnerRole.TENANT_ADMIN.getValue());
        }
        if (Objects.equals(userDTO.getId(), userDTO.getTenantOrgAdminId())) {
            roles.add(InnerRole.TENANT_ORG_ADMIN.getValue());
        }
        // 分配的业务角色
        var businessRoles = userRoleRepoProc.getRoleOfUser(userDTO.getId(), userDTO.getTenantId(), userDTO.getTenantOrg() == null ? null : userDTO.getTenantOrg().getId());
        roles.addAll(businessRoles.stream().map(IdCodeNameParam::getCode).collect(Collectors.toList()));

        userDTO.setRoles(businessRoles);
        userDTO.setRoleIds(businessRoles.stream().map(IdCodeNameParam::getId).collect(Collectors.toSet()));

        return Collections.unmodifiableSet(roles);
    }

    /**
     * 查询用户的角色
     *
     * @param userIds 用户
     * @return 角色
     */
    public Map<Long, List<IdCodeNameParam>> queryUserRoles(@NotEmpty Set<Long> userIds) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        return userRoleRepoProc.getRoleOfUser(userIds, currentUser.getTenantId(), currentUser.getTenantOrgId());
    }

    /**
     * 获取角色ID
     *
     * @param roleCodes 角色编码
     * @return 角色ID
     */
    public Set<Long> getRoleId(Set<String> roleCodes) {
        if (CollUtil.isEmpty(roleCodes)) {
            return Collections.emptySet();
        }

        return roleRepoProc.getIds(roleCodes);
    }

    /**
     * 获取用户菜单
     *
     * @param includeAction 是否包含操作按钮
     * @param tree          是否返回树状结构
     * @return 菜单列表
     */
    public List<UserMenuRespVO> queryUserMenu(Boolean includeAction, Boolean tree) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        List<UserMenuRespVO> respVoList = new ArrayList<>(128);

        if (includeAction == null) {
            // 默认不包含按钮
            includeAction = false;
        }
        // 有权限的应用
        Set<String> appCodes = null;
        if (currentUser.getTenant() != null) {
            appCodes = currentUser.getTenant().getAppCodes();
            if (CollectionUtils.isEmpty(appCodes)) {
                return Collections.emptyList();
            }
        }
        // 管理员菜单
        respVoList.addAll(permissionMenuBo2Vo(queryUserMenuOfAdmin(currentUser, appCodes, includeAction)));
        // 普通用户的菜单
        respVoList.addAll(permissionMenuBo2Vo(queryUserMenuOfCommon(currentUser, appCodes, includeAction)));

        if (respVoList.isEmpty()) {
            return Collections.emptyList();
        }

        // 菜单去重
        respVoList = respVoList.stream().collect(Collectors.collectingAndThen(
                Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(UserMenuRespVO::getMenuCode))),
                ArrayList::new
        ));

        // 树形处理
        if (tree != null && !tree) {
            return filterNoParent(respVoList);
        }
        TreeDataUtil<UserMenuRespVO> treeDataUtil = new TreeDataUtil<>(respVoList,
                UserMenuRespVO::getMenuCode, UserMenuRespVO::getParentMenuCode,
                UserMenuRespVO::setChildren, Comparator.comparingInt(UserMenuRespVO::getSortNo));
        return ((List<UserMenuRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> !StringUtils.hasText(t.getParentMenuCode()))
                .collect(Collectors.toList());
    }

    /**
     * 根据菜单查询用户的按钮
     *
     * @param menuCode 菜单编码
     * @return 按钮列表
     */
    public List<UserMenuRespVO> queryUserActionByMenu(@NotBlank String menuCode) {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        // 先查询下级所有按钮
        AtomicReference<String> menuType = new AtomicReference<>();
        List<MenuBO> actionBos = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            var tempMenuType = menuRepoProc.getMenuTypeByMenuCode(menuCode);
            if (CharSequenceUtil.isBlank(tempMenuType)) {
                log.info("菜单{}不存在或配置异常", menuCode);
                return Collections.emptyList();
            }
            menuType.set(tempMenuType);
            return menuRepoProc.queryActionByMenuCode(menuCode, true);
        });
        if (actionBos.isEmpty()) {
            return Collections.emptyList();
        }
        actionBos.forEach(t -> t.setNodeType(MenuTreeNodeType.ACTION.getValue()));

        // 如果是管理菜单，则无需授权，直接返回所有按钮
        if (PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(menuType.get())) {
            return platformMenu2Bo2Vo(actionBos);
        }

        // 普通菜单，则过滤出已授权的按钮
        var userRoles = obtainUserRole(currentUser);
        if (userRoles.isEmpty()) {
            // 没有权限
            return Collections.emptyList();
        }
        var actionBoMap = actionBos.stream().collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));
        var hasCustom = hasCustomMenuTree(currentUser);
        // 查询有权限的按钮编码
        var permissionActionCodes = rolePermissionRepoProc.queryPermissionCodeByRoles(userRoles,
                Set.of(MenuTreeNodeType.ACTION.getValue()), actionBoMap.keySet(), hasCustom);
        if (permissionActionCodes.isEmpty()) {
            return Collections.emptyList();
        }
        actionBos = permissionActionCodes.stream().filter(actionBoMap::containsKey).map(actionBoMap::get).collect(Collectors.toList());
        return platformMenu2Bo2Vo(actionBos);
    }

    /**
     * 查询用户的所有按钮
     *
     * @return 按钮列表
     */
    public List<UserMenuRespVO> queryAllUserAction() {
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        // 有权限的应用
        Set<String> appCodes = null;
        if (currentUser.getTenant() != null) {
            appCodes = currentUser.getTenant().getAppCodes();
            if (CollectionUtils.isEmpty(appCodes)) {
                return Collections.emptyList();
            }
        }
        // 获取用户角色
        var userRoles = obtainUserRole(currentUser);
        if (userRoles.isEmpty()) {
            return Collections.emptyList();
        }
        // 获取用户的权限按钮
        boolean hasCustom = hasCustomMenuTree(currentUser);
        var permissionActionCodes = rolePermissionRepoProc.queryPermissionCodeByRoles(userRoles, Set.of(MenuTreeNodeType.ACTION.getValue()), hasCustom);
        if (permissionActionCodes.isEmpty()) {
            return Collections.emptyList();
        }
        // 获取原始按钮
        var originalActionList = originalActions(appCodes);
        var originalActionMap = originalActionList.stream().collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 转换返回的结果
        var actionList = permissionActionCodes.stream().filter(originalActionMap::containsKey).map(t -> {
            var actionBO = originalActionMap.get(t);
            actionBO.setNodeType(MenuTreeNodeType.ACTION.getValue());
            return actionBO;
        }).collect(Collectors.toList());
        List<UserMenuRespVO> respVoList = new ArrayList<>(platformMenu2Bo2Vo(actionList));

        // 管理员的按钮
        var adminActionList = originalActionList.stream().filter(t -> PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(t.getMenusType())).collect(Collectors.toList());
        respVoList.addAll(platformMenu2Bo2Vo(adminActionList));

        return respVoList;
    }

    /**
     * 查询用户授权的字段
     *
     * @param menuCode 菜单编码
     * @param apiCode  api接口编码
     * @return 字段列表
     */
    public List<UserFieldRespVO> queryUserField(@NotBlank String menuCode, @NotBlank String apiCode) {
        Assert.hasText(menuCode, "菜单编码为空");
        Assert.hasText(apiCode, "API接口编码为空");

        // 获取用户角色
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        AtomicReference<String> appCode = new AtomicReference<>();
        // 查询接口的出参信息
        List<CodeNameParam> apiParamList = tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 查询出参
            var params = apiParameterRepoProc.queryOutParamByApiCode(apiCode);
            if (params.isEmpty()) {
                log.warn("接口{}未配置出参", apiCode);
                return Collections.emptyList();
            }
            // 查询接口是否还在
            appCode.set(apiRepoProc.getAppCodeByCode(apiCode));
            Assert.hasText(appCode.get(), "接口不存在或数据异常");
            return params;
        });
        if (apiParamList.isEmpty()) {
            return Collections.emptyList();
        }
        if (currentUser.getTenant() != null && !CollUtil.contains(currentUser.getTenant().getAppCodes(), appCode.get())) {
            // 应用无权限
            log.warn("接口{}所属应用{}未分配给租户{}", apiCode, appCode, currentUser.getTenantId());
            return this.convertUserField(apiParamList, Collections.emptyList());
        }
        var userRoles = this.obtainUserRoleId(currentUser);
        if (userRoles.isEmpty()) {
            // 未分配权限
            return this.convertUserField(apiParamList, Collections.emptyList());
        }
        // 查询角色分配的权限
        var permissionFields = dpcrApiFieldsRepoProc.queryBoByRole(userRoles, menuCode, apiCode);
        return this.convertUserField(apiParamList, permissionFields);
    }

    /**
     * 获取角色管理的用户ID
     *
     * @param roleCode 角色编码
     * @return 用户ID
     */
    public Set<Long> queryUserIdOfRole(@NotBlank String roleCode) {
        return userRoleRepoProc.getUserIdByRole(roleCode);
    }

    /**
     * 获取用户的数据权限
     *
     * @return 数据权限
     */
    public SysDprRoleApiRowColumnRuleDTO getDataPermissionOfCurrentUser() {
        SysDprRoleApiRowColumnRuleDTO roleRuleDTO = new SysDprRoleApiRowColumnRuleDTO();
        // 用户信息
        var userInfo = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        roleRuleDTO.setUserId(userInfo.getUserId());
        roleRuleDTO.setTenantId(ObjectUtil.defaultIfNull(userInfo.getTenantId(), TenantConstant.DEFAULT_TENANT_ID));
        // 角色信息
        List<Long> roleIds = CollUtil.isEmpty(userInfo.getUser().getRoles()) ? Collections.emptyList() :
                userInfo.getUser().getRoles().stream().map(IdCodeNameParam::getId).collect(Collectors.toList());
        roleRuleDTO.setRoelIdList(roleIds);
        if (roleRuleDTO.getRoelIdList().isEmpty() ||
                (userInfo.getTenant() != null && CollUtil.isEmpty(userInfo.getTenant().getAppCodes()))) {
            // 无权限，直接返回
            roleRuleDTO.setSysDpcRoleApiFieldsDTOList(Collections.emptyList());
            roleRuleDTO.setSysDprRoleApiDataRuleListQueryDTO(Collections.emptyList());
            return roleRuleDTO;
        }

        // 查询行规则权限
        var mapWhere =
                MapUtils.builder()
                        .field(SysDprRoleApiDataRuleListQueryBean::getRoleId, roleIds).op(FieldOps.InList)
                        .build();
        var sysDprRoleApiDataRuleListQueryBeans
                = beanSearcher.searchAll(SysDprRoleApiDataRuleListQueryBean.class, mapWhere);

        var beanList = sysDprRoleApiDataRuleListQueryBeans.stream()
                .filter(t -> {
                    // 过滤掉【全部】 的
                    if (CharSequenceUtil.equals(DprSysInternallyEnum.DPR_SYS_INTERNALLY_ALL.name(), t.getDprSysInternally())) {
                        return false;
                    }
                    if (t.getApiId() == null) {
                        log.error("接口不存在：{}", t.getId());
                        return false;
                    }
                    if (CharSequenceUtil.isBlank(t.getMenusCode())) {
                        log.error("菜单不存在：{}", t.getId());
                        return false;
                    }
                    if (CharSequenceUtil.isBlank(t.getDprRuleValueType())) {
                        log.error("规则值类型不存在：{}", t.getId());
                        return false;
                    }
                    return true;
                }).collect(Collectors.toList());

        // 根据规则值类型进行分组
        Map<String, List<SysDprRoleApiDataRuleListQueryBean>> dprRuleValueTypeGroupMap =
                beanList.stream()
                        .collect(Collectors.groupingBy(SysDprRoleApiDataRuleListQueryBean::getDprRuleValueType));
        //汇总全部的规则集合
        List<SysDprRoleApiDataRuleListQueryDTO> dtoList = new ArrayList<>();
        //业务集合
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> businessBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name());
            var beanValueList=  RoleAppApiDataPermissionRpcServiceUtil
                    .setBusinessSysDprRoleApiRuleValue(userInfo, businessBean);
            dtoList.addAll(beanValueList);
        }
        //自定义
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> customBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_CUSTOM.name());
            var beanValueList=RoleAppApiDataPermissionRpcServiceUtil.
                    setCustomSysDprRoleApiRuleValue(userInfo, customBean);
            dtoList.addAll(beanValueList);
        }
        //系统内置 重点
        if (dprRuleValueTypeGroupMap.containsKey(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name())) {
            List<SysDprRoleApiDataRuleListQueryBean> sysBean =
                    dprRuleValueTypeGroupMap.get(DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name());
            var beanValueList= RoleAppApiDataPermissionRpcServiceUtil
                    .setSysSysDprRoleApiRuleValue(valueServiceSpi, userInfo, sysBean);
            dtoList.addAll(beanValueList);
        }

        roleRuleDTO.setSysDprRoleApiDataRuleListQueryDTO(dtoList);

        // 查询字段权限
        var mapColumnWhere =
                MapUtils.builder()
                        .field(SysDprRoleApiDataColumnRuleListQueryBean::getRoleId, roleIds).op(FieldOps.InList)
                        .build();
        var dprRoleApiDataColumnRuleListQueryBeans
                = beanSearcher.searchAll(SysDprRoleApiDataColumnRuleListQueryBean.class, mapColumnWhere);
        if (dprRoleApiDataColumnRuleListQueryBeans != null) {
            var doList = dprRoleApiDataColumnRuleListQueryBeans.stream()
                    .map(SysDpcRoleApiFieldsConvert.INSTANCE::beanToDto)
                    .collect(Collectors.toList());
            roleRuleDTO.setSysDpcRoleApiFieldsDTOList(doList);
        }
        return roleRuleDTO;
    }

    private List<UserFieldRespVO> convertUserField(List<CodeNameParam> apiParamList, List<PermissionParameterBO> permissionFields) {
        Map<String, List<PermissionParameterBO>> perMap = CollectionUtils.isEmpty(permissionFields) ? Collections.emptyMap() :
                permissionFields.stream().collect(Collectors.groupingBy(PermissionParameterBO::getFieldName));
        var conflictShow = systemProperties.getPermissionConflictShow() == null || systemProperties.getPermissionConflictShow();
        return apiParamList.stream()
                .map(t -> {
                    UserFieldRespVO respVO = new UserFieldRespVO();
                    respVO.setFieldName(t.getCode());
                    respVO.setFieldRemark(t.getName());

                    if (perMap.isEmpty() || !perMap.containsKey(t.getCode())) {
                        // 默认不显示
                        respVO.setFieldApiVisible(false);
                        respVO.setFieldFormVisible(false);
                        respVO.setFieldFormUpdate(false);
                        return respVO;
                    }

                    var perList = perMap.get(t.getCode());
                    respVO.setId(CollUtil.isEmpty(perList) ? null : perList.get(0).getId());
                    respVO.setFieldApiVisible(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldApiVisible));
                    respVO.setFieldFormVisible(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldFormVisible));
                    respVO.setFieldFormUpdate(this.authShow(conflictShow, perList, PermissionParameterBO::getFieldFormUpdate));
                    return respVO;
                }).collect(Collectors.toList());
    }

    private boolean authShow(boolean conflictShow, List<PermissionParameterBO> parameterBOList, Function<PermissionParameterBO, Boolean> valueFunction) {
        if (CollUtil.isEmpty(parameterBOList)) {
            return false;
        }

        var show = false;
        for (PermissionParameterBO parameterBO : parameterBOList) {
            show = Boolean.TRUE.equals(valueFunction.apply(parameterBO));
            if (conflictShow) {
                // 如果是【冲突时显示】，则只要有一个显示就显示
                if (show) {
                    return true;
                }
            } else {
                // 如果是【冲突时不显示】，则只要有一个不显示就不显示
                if (!show) {
                    return false;
                }
            }
        }
        return show;
    }

    private List<UserMenuRespVO> filterNoParent(List<UserMenuRespVO> respVos) {
        TreeDataUtil<UserMenuRespVO> treeDataUtil = new TreeDataUtil<>(respVos, UserMenuRespVO::getMenuCode, UserMenuRespVO::getParentMenuCode,
                UserMenuRespVO::setChildren);
        List<UserMenuRespVO> rootDataList = ((List<UserMenuRespVO>) treeDataUtil.getRoots()).stream()
                .filter(t -> !StringUtils.hasText(t.getParentMenuCode())).collect(Collectors.toList());
        if (rootDataList.isEmpty()) {
            return Collections.emptyList();
        }

        return CollectionUtil.expandTree(rootDataList, UserMenuRespVO::getChildren);
    }

    private List<UserMenuRespVO> permissionMenuBo2Vo(List<PermissionMenuBO> menuBos) {
        if (CollectionUtils.isEmpty(menuBos)) {
            return Collections.emptyList();
        }
        List<UserMenuRespVO> respVos = new ArrayList<>(menuBos.size());
        UserMenuRespVO respVO = null;
        for (PermissionMenuBO menuBO : menuBos) {
            respVO = new UserMenuRespVO();
            respVO.setMenuCode(menuBO.getMenusCode());
            respVO.setMenuName(menuBO.getMenusName());
            respVO.setParentMenuCode(menuBO.getMenusParentCode());
            respVO.setMenusIcon(menuBO.getMenusIcon());
            respVO.setRoute(menuBO.getMenusRoute());
            respVO.setNodeType(menuBO.getNodeType());
            respVO.setSortNo(ObjectUtil.defaultIfNull(menuBO.getMenusOrder(), 1));
            respVO.setDisplay(ObjectUtil.defaultIfNull(menuBO.getDisplay(), true));
            respVO.setOuterLink(menuBO.getOuterLink());
            respVO.setOuterLinkType(menuBO.getOuterLinkType());
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(16));

            respVos.add(respVO);
        }
        return respVos;
    }

    private List<UserMenuRespVO> platformMenu2Bo2Vo(List<MenuBO> menuBos) {
        if (CollectionUtils.isEmpty(menuBos)) {
            return Collections.emptyList();
        }
        List<UserMenuRespVO> respVos = new ArrayList<>(menuBos.size());
        UserMenuRespVO respVO = null;
        for (MenuBO menuBO : menuBos) {
            respVO = new UserMenuRespVO();
            respVO.setMenuCode(menuBO.getMenusCode());
            respVO.setMenuName(menuBO.getMenusName());
            respVO.setParentMenuCode(menuBO.getMenusParentCode());
            respVO.setMenusIcon(menuBO.getMenusIcon());
            respVO.setRoute(menuBO.getMenusRoute());
            respVO.setNodeType(menuBO.getNodeType());
            respVO.setSortNo(ObjectUtil.defaultIfNull(menuBO.getMenusOrder(), 1));
            respVO.setDisplay(ObjectUtil.defaultIfNull(menuBO.getDisplay(), true));
            respVO.setOuterLink(menuBO.getOuterLink());
            respVO.setOuterLinkType(menuBO.getOuterLinkType());
            respVO.setHasChildren(false);
            respVO.setChildren(new ArrayList<>(16));

            respVos.add(respVO);
        }
        return respVos;
    }

    private List<PermissionMenuBO> queryUserMenuOfCommon(GeneralUserDetails currentUser, Set<String> appCodes, boolean includeAction) {
        if (currentUser == null || CollectionUtils.isEmpty(currentUser.getRoleCodes())) {
            return Collections.emptyList();
        }

        // 过滤出在用状态的角色
        var roleCodes = obtainUserRole(currentUser);

        // 判断是否有自定义菜单树
        boolean hasCustom = hasCustomMenuTree(currentUser);
        if (hasCustom) {
            // 有自定义菜单树
            return queryUserMenuForCustomMenuTree(roleCodes, appCodes, includeAction);
        }
        // 使用默认的菜单树
        return queryUserMenuForDefaultMenuTree(roleCodes, appCodes, includeAction);
    }

    private List<PermissionMenuBO> queryUserMenuOfAdmin(GeneralUserDetails currentUser, Set<String> appCodes, boolean includeAction) {
        if (currentUser == null) {
            return Collections.emptyList();
        }
        List<PermissionMenuBO> menuBos = new ArrayList<>(128);

        return tenantDataIsolateProvider.byDefaultDirectly(() -> {
            // 管理员类型
            Set<PlatformAdminTypeEnum> adminTypeEnums = new HashSet<>();
            if (currentUser.isSystemAdmin()) {
                adminTypeEnums.add(PlatformAdminTypeEnum.SYS_ADMIN);
            }
            if (currentUser.isTenantAdmin() && CollUtil.isNotEmpty(appCodes)) {
                adminTypeEnums.add(PlatformAdminTypeEnum.TENANT_ADMIN);
            }
            if (currentUser.isTenantOrgAdmin()) {
                adminTypeEnums.add(PlatformAdminTypeEnum.TENANT_ORG_ADMIN);
            }
            if (!adminTypeEnums.isEmpty()) {
                var adminMenus = adminMenuRepoProc.queryMenuOfTenantAdmin(adminTypeEnums, appCodes, false)
                        .stream()
                        .peek(t -> t.setNodeType(MenuTreeNodeType.MENU.getValue()))
                        .collect(Collectors.toList());
                menuBos.addAll(adminMenus);
            }

            if (menuBos.isEmpty()) {
                if (currentUser.isSystemAdmin()) {
                    // 如果是系统管理员，可能是初始化，则获取所有管理菜单
                    var apps = menuRepoProc.queryMenuForMng(Set.of(CloudtAppHolder.getAppCode()), false, true).stream()
                            .map(t -> {
                                PermissionMenuBO actionBO = new PermissionMenuBO();
                                actionBO.setMenusName(t.getMenusName());
                                actionBO.setNodeType(MenuTreeNodeType.MENU.getValue());
                                actionBO.setMenusCode(t.getMenusCode());
                                actionBO.setMenusOrder(t.getMenusOrder());
                                actionBO.setMenusParentCode(t.getMenusParentCode());
                                actionBO.setMenusRoute(t.getMenusRoute());
                                actionBO.setMenusIcon(t.getMenusIcon());
                                actionBO.setDisplay(t.getDisplay());
                                actionBO.setMenusIcon(t.getMenusIcon());
                                actionBO.setOuterLink(t.getOuterLink());
                                actionBO.setOuterLinkType(t.getOuterLinkType());
                                return actionBO;
                            }).collect(Collectors.toList());
                    menuBos.addAll(apps);
                }
                return Collections.emptyList();
            }
            // 按钮
            if (includeAction) {
                var menuCodeSet = menuBos.stream().map(PermissionMenuBO::getMenusCode).collect(Collectors.toSet());
                var actionBoList = menuRepoProc.queryActionByAppCode(appCodes, true).stream()
                        .filter(t -> PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name().equals(t.getMenusType()) && menuCodeSet.contains(t.getMenusParentCode()))
                        .map(t -> {
                            PermissionMenuBO actionBO = new PermissionMenuBO();
                            actionBO.setMenusName(t.getMenusName());
                            actionBO.setNodeType(MenuTreeNodeType.ACTION.getValue());
                            actionBO.setMenusCode(t.getMenusCode());
                            actionBO.setMenusOrder(t.getMenusOrder());
                            actionBO.setMenusParentCode(t.getMenusParentCode());
                            actionBO.setMenusRoute(t.getMenusRoute());
                            actionBO.setMenusIcon(t.getMenusIcon());
                            actionBO.setDisplay(t.getDisplay());
                            actionBO.setOuterLink(t.getOuterLink());
                            actionBO.setOuterLinkType(t.getOuterLinkType());
                            return actionBO;
                        }).collect(Collectors.toList());
                menuBos.addAll(actionBoList);
            }
            return menuBos;
        });
    }

    private List<PermissionMenuBO> queryUserMenuForDefaultMenuTree(Set<String> roleCodes, Set<String> appCodes, boolean includeAction) {
        // 查询用户角色绑定的权限编码
        Set<String> permissionTypes = new HashSet<>(4);
        permissionTypes.add(MenuTreeNodeType.APP.getValue());
        permissionTypes.add(MenuTreeNodeType.MENU.getValue());
        if (includeAction) {
            permissionTypes.add(MenuTreeNodeType.ACTION.getValue());
        }
        var permissionBos = rolePermissionRepoProc.queryPermissionByRole(roleCodes, permissionTypes, false);
        if (permissionBos.isEmpty()) {
            return Collections.emptyList();
        }

        // 应用
        var allApp = allApp();
        Map<String, Integer> appOrderMap = new HashMap<>();
        int order = 1;
        for (CodeNameParam codeNameParam : allApp) {
            appOrderMap.put(codeNameParam.getCode(), order++);
        }
        var appsMap = allApp.stream().collect(Collectors.toMap(CodeNameParam::getCode, t -> t, (t1, t2) -> t1));
        // 原始菜单和按钮
        var menuMap = originalMenus(appCodes, includeAction).stream()
                .filter(t -> t.getMenusState() == null || t.getMenusState())
                .collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 组织返回菜单
        List<PermissionMenuBO> menuBoList = new ArrayList<>(permissionBos.size());
        PermissionMenuBO menuBO = null;
        for (PermissionBO permissionBo : permissionBos) {
            menuBO = new PermissionMenuBO();
            // 应用
            if (MenuTreeNodeType.APP.getValue().equals(permissionBo.getPermissionType())) {
                if (CollUtil.isNotEmpty(appCodes) && !appCodes.contains(permissionBo.getPermissionCode())) {
                    // 已无该应用
                    continue;
                }
                var app = appsMap.get(permissionBo.getPermissionCode());
                if (app == null) {
                    // 应用不存在
                    continue;
                }
                menuBO.setMenusName(app.getName());
                menuBO.setNodeType(MenuTreeNodeType.MENU.getValue());
                menuBO.setMenusOrder(appOrderMap.getOrDefault(app.getCode(), 1));
                menuBO.setDisplay(true);
            } else if (MenuTreeNodeType.MENU.getValue().equals(permissionBo.getPermissionType()) ||
                    MenuTreeNodeType.ACTION.getValue().equals(permissionBo.getPermissionType())) {
                // 菜单或按钮
                var menu = menuMap.get(permissionBo.getPermissionCode());
                if (menu == null) {
                    continue;
                }
                if (CollUtil.isNotEmpty(appCodes) && !appCodes.contains(menu.getMenusAppCode())) {
                    // 已无该应用
                    continue;
                }
                menuBO.setMenusName(menu.getMenusName());
                menuBO.setMenusParentCode(menu.getMenusParentCode());
                menuBO.setMenusOrder(menu.getMenusOrder());
                menuBO.setMenusRoute(menu.getMenusRoute());
                menuBO.setMenusIcon(menu.getMenusIcon());
                menuBO.setOuterLink(menu.getOuterLink());
                menuBO.setOuterLinkType(menu.getOuterLinkType());

                if (MenuTreeNodeType.MENU.getValue().equals(permissionBo.getPermissionType())) {
                    // 菜单
                    menuBO.setNodeType(MenuTreeNodeType.MENU.getValue());
                    if (!StringUtils.hasText(menu.getMenusParentCode())) {
                        // 顶级菜单，上级取app
                        menuBO.setMenusParentCode(menu.getMenusAppCode());
                    }
                } else {
                    menuBO.setNodeType(MenuTreeNodeType.ACTION.getValue());
                }
                menuBO.setDisplay(menu.getDisplay());
            } else {
                log.error("未知该权限资源类型：{}, {}", permissionBo.getPermissionType(), permissionBo.getPermissionCode());
                continue;
            }
            menuBO.setMenusCode(permissionBo.getPermissionCode());

            menuBoList.add(menuBO);
        }

        return menuBoList;
    }

    private List<PermissionMenuBO> queryUserMenuForCustomMenuTree(Set<String> roleCodes, Set<String> appCodes, boolean includeAction) {
        // 根据用户角色查询菜单
        Set<String> permissionTypes = new HashSet<>(4);
        permissionTypes.add(MenuTreeNodeType.APP.getValue());
        permissionTypes.add(MenuTreeNodeType.MENU.getValue());
        var customMenus = rolePermissionRepoProc.queryMenuForCustom(roleCodes, permissionTypes);
        if (customMenus.isEmpty()) {
            // 根据角色没有查询出来菜单
            return Collections.emptyList();
        }

        // 应用
        var appsMap = allApp().stream().collect(Collectors.toMap(CodeNameParam::getCode, t -> t, (t1, t2) -> t1));
        // 原始菜单
        var menuMap = originalMenus(appCodes, false).stream()
                .filter(t -> t.getMenusState() == null || t.getMenusState())
                .collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));

        // 组织权限菜单
        List<PermissionMenuBO> permissionMenuBos = new ArrayList<>(customMenus.size());
        PermissionMenuBO menuBO = null;
        for (SysTenantMenuTreeDO customMenu : customMenus) {
            menuBO = new PermissionMenuBO();
            menuBO.setMenusName(customMenu.getMenuName());
            menuBO.setNodeType(MenuTreeNodeType.MENU.getValue());
            menuBO.setMenusCode(customMenu.getMenuCode());
            menuBO.setMenusOrder(ObjectUtil.defaultIfNull(customMenu.getSortNo(), 1));
            menuBO.setMenusParentCode(customMenu.getParentMenuCode());
            menuBO.setMenusIcon(customMenu.getIcon());
            menuBO.setDisplay(true);
            if (MenuTreeNodeType.APP.getValue().equals(customMenu.getNodeType())) {
                // 应用
                var app = appsMap.get(customMenu.getMenuCode());
                if (app == null || (CollUtil.isNotEmpty(appCodes)) && !appCodes.contains(app.getCode())) {
                    continue;
                }
                if (!StringUtils.hasText(menuBO.getMenusName())) {
                    menuBO.setMenusName(app.getName());
                }
            } else if (MenuTreeNodeType.MENU.getValue().equals(customMenu.getNodeType())) {
                if (!customMenu.getCustom()) {
                    // 菜单
                    var menu = menuMap.get(customMenu.getMenuCode());
                    if (menu == null || (CollUtil.isNotEmpty(appCodes)) && !appCodes.contains(menu.getMenusAppCode())) {
                        continue;
                    }
                    if (!StringUtils.hasText(menuBO.getMenusName())) {
                        menuBO.setMenusName(menu.getMenusName());
                    }
                    menuBO.setMenusRoute(menu.getMenusRoute());
                    menuBO.setDisplay(menu.getDisplay() == null || menu.getDisplay());
                    menuBO.setOuterLink(menu.getOuterLink());
                    menuBO.setOuterLinkType(menu.getOuterLinkType());
                }
            } else {
                log.error("未知菜单类型：{}", customMenu.getNodeType());
                continue;
            }

            permissionMenuBos.add(menuBO);
        }
        if (!includeAction) {
            return permissionMenuBos;
        }

        // 组织权限按钮
        Set<String> actionCodes = rolePermissionRepoProc.queryPermissionCodeByRoles(roleCodes,
                Set.of(MenuTreeNodeType.ACTION.getValue()), true);
        if (actionCodes.isEmpty()) {
            return permissionMenuBos;
        }
        var actionMap = originalActions(appCodes).stream()
                .filter(t -> t.getMenusState() == null || t.getMenusState())
                .collect(Collectors.toMap(MenuBO::getMenusCode, t -> t, (t1, t2) -> t1));
        MenuBO actionBO = null;
        for (String actionCode : actionCodes) {
            actionBO = actionMap.get(actionCode);
            if (actionBO == null) {
                continue;
            }
            menuBO = new PermissionMenuBO();
            menuBO.setMenusName(actionBO.getMenusName());
            menuBO.setNodeType(MenuTreeNodeType.ACTION.getValue());
            menuBO.setMenusCode(actionBO.getMenusCode());
            menuBO.setMenusOrder(actionBO.getMenusOrder());
            menuBO.setMenusParentCode(actionBO.getMenusParentCode());
            menuBO.setMenusRoute(actionBO.getMenusRoute());
            menuBO.setMenusIcon(actionBO.getMenusIcon());
            menuBO.setOuterLink(actionBO.getOuterLink());
            menuBO.setOuterLinkType(actionBO.getOuterLinkType());
            menuBO.setDisplay(actionBO.getDisplay());

            permissionMenuBos.add(menuBO);
        }
        return permissionMenuBos;
    }

    private Set<String> obtainUserRole(GeneralUserDetails userDetails) {
        var roleCodes = this.normalizeUserRoleCodes(userDetails);
        if (CollectionUtils.isEmpty(roleCodes)) {
            return Collections.emptySet();
        }

        return roleRepoProc.filterEnabled(roleCodes);
    }

    private Set<Long> obtainUserRoleId(GeneralUserDetails userDetails) {
        var roleCodes = this.normalizeUserRoleCodes(userDetails);
        if (CollectionUtils.isEmpty(roleCodes)) {
            return Collections.emptySet();
        }

        return roleRepoProc.filterEnabledId(roleCodes);
    }

    private Set<String> normalizeUserRoleCodes(GeneralUserDetails userDetails) {
        if (CollectionUtils.isEmpty(userDetails.getRoleCodes())) {
            return Collections.emptySet();
        }

        Set<String> roleCodes = null;
        if (StringUtils.hasText(authorizationProperties.getRolePrefix())) {
            // 去掉前缀
            String prefix = authorizationProperties.getRolePrefix();
            roleCodes = userDetails.getRoleCodes().stream()
                    .map(t -> t.startsWith(prefix) ? t.substring(authorizationProperties.getRolePrefix().length()) : t)
                    .collect(Collectors.toSet());
        } else {
            roleCodes = new HashSet<>(userDetails.getRoleCodes());
        }
        return roleCodes;
    }

    private boolean hasCustomMenuTree(GeneralUserDetails currentUser) {
        var tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        var enabled = tenantMenuRepoProc.getEnabled(tenantId);

        return enabled != null && enabled;
    }

    private List<com.elitescloud.cloudt.common.base.param.CodeNameParam> allApp() {
        return appProvider.all();
    }

    private List<MenuBO> originalMenus(Set<String> appCodes, boolean includeAction) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryMenuByAppCode(appCodes, includeAction, true));
    }

    private List<MenuBO> originalActions(Set<String> appCodes) {
        return tenantDataIsolateProvider.byDefaultDirectly(() -> menuRepoProc.queryActionByAppCode(appCodes, true));
    }
}
