package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.security.common.InnerRole;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.constant.SysRoleBusiness;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.RoleType;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.convert.RoleConvert;
import com.elitescloud.cloudt.system.service.callback.RoleChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysRoleSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Objects;
import java.util.function.BiFunction;

/**
 * 角色管理.
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class RoleMngManager {
    private static final RoleConvert CONVERT = RoleConvert.INSTANCE;

    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc apiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Autowired
    private ObjectProvider<RoleChangedCallback> roleChangedCallbacks;
    @Autowired
    private AuthorizationProperties authorizationProperties;

    /**
     * 保存角色信息
     *
     * @param saveBO 角色信息
     * @return 角色信息
     */
    public SysRoleDO upsert(@NotNull SysRoleSaveBO saveBO) {
        // 检查保存参数
        boolean isAdd = saveBO.getId() == null;
        SysRoleDO roleDO = isAdd ? checkForInsert(saveBO) : checkForUpdate(saveBO);

        SysRoleDO roleOld = isAdd ? null : roleRepoProc.get(saveBO.getId());

        // 保存角色信息
        roleRepoProc.save(roleDO);

        // 保存后的回调
        roleChangedCallbacks.forEach(t -> t.onUpsert(isAdd, saveBO, roleDO));
        if (!isAdd && !Objects.equals(roleDO.getEnabled(), roleOld.getEnabled())) {
            roleChangedCallbacks.forEach(t -> t.onEnabled(saveBO.getId(), roleDO.getEnabled()));
        }

        return roleDO;
    }

    /**
     * 更新角色的启用状态
     *
     * @param roleId  角色ID
     * @param enabled 启用状态
     */
    public void updateEnabled(@NotNull Long roleId, Boolean enabled) {
        if (enabled == null) {
            enabled = true;
        }
        roleRepoProc.updateEnabled(roleId, enabled);

        Boolean finalEnabled = enabled;
        roleChangedCallbacks.forEach(t -> t.onEnabled(roleId, finalEnabled));
    }

    /**
     * 删除角色数据
     *
     * @param roleId 角色ID
     */
    public void delete(@NotNull Long roleId) {
        var role = roleRepoProc.get(roleId);
        if (role == null) {
            return;
        }

        // 删除角色
        roleRepoProc.delete(roleId);
        // 删除角色与权限的关联
        rolePermissionRepoProc.deleteByRole(roleId);
        // 删除数据权限
        apiRowRuleRepoProc.deleteByRole(roleId);
        apiFieldsRepoProc.deleteByRole(roleId);

        // 删除后的回调
        roleChangedCallbacks.forEach(t -> t.onDelete(role));
    }

    /**
     * 根据角色类型调用
     *
     * @param function 回调方法
     * @param <R>
     * @return
     */
    public <R> R callByType(@NotNull BiFunction<RoleType, String, R> function) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUser();
        RoleType roleType = null;
        String typeId = null;
        if (currentUser == null || currentUser.isOperation()) {
            var tenant = tenantClientProvider.getSessionTenant();
            if (tenant != null) {
                roleType = RoleType.TENANT;
                typeId = tenant.getId().toString();
            } else {
                roleType = RoleType.PLATFORM;
                typeId = TenantConstant.DEFAULT_TENANT_ID.toString();
            }
        } else if (currentUser.isTenantAdmin()) {
            roleType = RoleType.TENANT;
            typeId = currentUser.getTenantId().toString();
        } else if (currentUser.isTenantOrgAdmin()) {
            roleType = RoleType.TENANT_ORG;
            typeId = currentUser.getTenantOrgId().toString();
        } else {
            roleType = RoleType.CUSTOM;
            typeId = "";
        }

        return function.apply(roleType, typeId);
    }

    /**
     * 根据角色类型调用
     *
     * @param function 回调方法
     * @param <R>
     * @return
     */
    public <R> R callByType(@NotNull BiFunction<RoleType, String, R> function, SysUserDTO userDTO) {
        RoleType roleType = null;
        String typeId = null;
        if (userDTO == null || userDTO.getSysTenantVO() == null || TenantType.OPERATION == userDTO.getSysTenantVO().getType()) {
            roleType = RoleType.PLATFORM;
            typeId = "-1";
        } else if (ObjectUtil.equal(userDTO.getSysTenantVO().getSysUserId(), userDTO.getId())) {
            roleType = RoleType.TENANT;
            typeId = userDTO.getTenantId().toString();
        } else if (userDTO.getTenantOrg() != null && ObjectUtil.equal(userDTO.getTenantOrgAdminId(), userDTO.getId())) {
            roleType = RoleType.TENANT_ORG;
            typeId = userDTO.getTenantOrg().getId().toString();
        } else {
            roleType = RoleType.CUSTOM;
            typeId = "";
        }

        return function.apply(roleType, typeId);
    }

    private SysRoleDO checkForInsert(SysRoleSaveBO saveBO) {
        // 角色编码
        Assert.hasText(saveBO.getCode(), "角色编码为空");
        String codePrefix = rolePrefix();
        Assert.isTrue(!saveBO.getCode().startsWith(codePrefix), "角色编码的前缀不能是" + codePrefix);
        boolean exists = roleRepoProc.existsCode(saveBO.getCode());
        Assert.isTrue(!exists, "角色编码已存在");
        if (InnerRole.valueOf(saveBO.getCode()) != null) {
            throw new IllegalArgumentException(saveBO.getCode() + "是系统内置角色，不可使用");
        }

        if (saveBO.getGroupId() == null) {
            saveBO.setGroupId(SysRoleDO.DEFAULT_GROUP_ID);
        }

        // 是否启用
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(true);
        }

        // 角色类型
        Assert.hasText(saveBO.getType(), "未知角色类型");
        Assert.hasText(saveBO.getTypeId(), "未知角色类型");

        // 默认系统角色
        saveBO.setBusinessKey(CharSequenceUtil.blankToDefault(saveBO.getBusinessKey(), SysRoleBusiness.SYS.getValue()));

        return CONVERT.saveBo2DO(saveBO);
    }

    private SysRoleDO checkForUpdate(SysRoleSaveBO saveBO) {
        SysRoleDO roleDO = roleRepoProc.get(saveBO.getId());
        Assert.notNull(roleDO, "角色不存在");

        // 角色编码
        Assert.isTrue(roleDO.getCode().equals(saveBO.getCode()), "角色编码不可修改");
        if (InnerRole.valueOf(saveBO.getCode()) != null) {
            throw new IllegalArgumentException(saveBO.getCode() + "是系统内置角色，不可使用");
        }

        if (saveBO.getGroupId() == null) {
            saveBO.setGroupId(SysRoleDO.DEFAULT_GROUP_ID);
        }

        // 是否启用
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(roleDO.getEnabled());
        }

        // 角色类型
        if (!StringUtils.hasText(saveBO.getType())) {
            saveBO.setType(roleDO.getType());
        }
        if (!StringUtils.hasText(saveBO.getTypeId())) {
            saveBO.setTypeId(roleDO.getTypeId());
        }

        // 默认系统角色
        saveBO.setBusinessKey(CharSequenceUtil.blankToDefault(saveBO.getBusinessKey(), SysRoleBusiness.SYS.getValue()));

        CONVERT.copySaveBO(saveBO, roleDO);
        return roleDO;
    }

    private String rolePrefix() {
        return CharSequenceUtil.blankToDefault(authorizationProperties.getRolePrefix(), "ROLE_");
    }
}
