package com.elitescloud.cloudt.system.service.manager;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepoProc;
import com.elitescloud.cloudt.system.service.repo.UserRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.List;

/**
 * 用户查询公共接口.
 *
 * @author Kaiser（wang shao）
 * 2022/10/27
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class UserQueryManager {
    @Autowired
    private UserRepoProc userRepoProc;
    @Autowired
    private SysTenantUserRepoProc tenantUserRepoProc;

    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    /**
     * 根据账号获取唯一的用户ID
     *
     * @param username 登录账号
     * @return 用户ID
     */
    public Long getUniqueIdByUsername(@NotBlank String username) {
        return userRepoProc.getIdByUsername(username);
    }

    /**
     * 根据手机号获取唯一的用户ID
     *
     * @param mobile 手机号
     * @return 用户ID
     */
    public Long getUniqueIdByMobile(@NotBlank String mobile) {
        if (!systemProperties.getMobileUnique()) {
            throw new BusinessException("系统未限制手机号唯一");
        }

        var ids = userRepoProc.getIdByMobile(mobile);
        if (ids.isEmpty()) {
            return null;
        }
        if (ids.size() == 1) {
            return ids.get(0);
        }
        throw new BusinessException("存在重复的手机号");
    }

    /**
     * 根据邮箱获取唯一的用户ID
     *
     * @param email 邮箱
     * @return 用户ID
     */
    public Long getUniqueIdByEmail(@NotBlank String email) {
        if (!systemProperties.getEmailUnique()) {
            throw new BusinessException("系统未限制邮箱唯一");
        }

        var ids = userRepoProc.getIdByEmail(email);
        if (ids.isEmpty()) {
            return null;
        }
        if (ids.size() == 1) {
            return ids.get(0);
        }
        throw new BusinessException("存在重复的邮箱");
    }

    /**
     * 根据身份证获取唯一的用户ID
     *
     * @param idCard 身份证号
     * @return 用户ID
     */
    public Long getUniqueIdByIdCard(@NotBlank String idCard) {
        if (!systemProperties.getIdCardUnique()) {
            throw new BusinessException("系统未限制身份证唯一");
        }

        var ids = userRepoProc.getIdByIdCard(idCard);
        if (ids.isEmpty()) {
            return null;
        }
        if (ids.size() == 1) {
            return ids.get(0);
        }
        throw new BusinessException("存在重复的身份证号");
    }

    /**
     * 判断用户是否属于指定租户
     *
     * @param userId   用户ID
     * @param tenantId 租户ID
     * @return 是否属于
     */
    public boolean belongToTenant(@NotNull Long userId, @NotNull Long tenantId) {
        if (!tenantClientProvider.enabledTenant()) {
            return true;
        }
        var tenantIds = tenantUserRepoProc.getTenantIdOfUser(userId);
        return tenantIds.contains(tenantId);
    }

    /**
     * 获取用户基本信息
     *
     * @param userId 用户ID
     * @return 基本信息
     */
    public SysUserBasicDTO getBasic(@NotNull Long userId) {
        return userRepoProc.getBasicDto(userId);
    }

    /**
     * 获取用户基本信息
     *
     * @param userIds 用户ID
     * @return 基本信息
     */
    public List<SysUserBasicDTO> getBasic(@NotNull Collection<Long> userIds) {
        return userRepoProc.getBasicDto(userIds);
    }
}
