package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.dto.req.SysCurrencyRateQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRatePageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysCurrencyRateDO;
import com.elitescloud.cloudt.system.service.model.entity.SysCurrencyRateDO;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Repository
public class CurrencyRateRepoProc extends BaseRepoProc<SysCurrencyRateDO> {
    private static final QSysCurrencyRateDO QDO = QSysCurrencyRateDO.sysCurrencyRateDO;

    public CurrencyRateRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id      ID
     * @param enabled 启用状态
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 根据自货币和至货币查找已存在的记录ID
     *
     * @param fromCurrCode 自货币码
     * @param toCurrCode   至货币码
     * @return ID
     */
    public Long getIdByFromAndTo(@NotBlank String fromCurrCode, @NotBlank String toCurrCode) {
        return super.jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.fromCurr.eq(fromCurrCode).and(QDO.toCurr.eq(toCurrCode)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取启用状态
     *
     * @param id id
     * @return 启用状态
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 分页查询
     * <p>
     * 用于管理
     *
     * @param queryVO 查询参数
     * @return 汇率列表
     */
    public PagingVO<SysCurrencyRateDO> pageMng(CurrencyRatePageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(queryVO.getFromCurr()), QDO.fromCurr, queryVO.getFromCurr())
                .andEq(StringUtils.hasText(queryVO.getToCurr()), QDO.toCurr, queryVO.getToCurr())
                .andEq(queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .andAfter(queryVO.getValidFromS() != null, QDO.validFrom, queryVO.getValidFromS())
                .andBefore(queryVO.getValidFromE() != null, QDO.validFrom, queryVO.getValidFromE())
                .andAfter(queryVO.getValidToS() != null, QDO.validTo, queryVO.getValidToS())
                .andBefore(queryVO.getValidToE() != null, QDO.validTo, queryVO.getValidToE())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 获取汇率
     *
     * @param queryDTO 查询参数
     * @return 值
     */
    public SysCurrencyRateDO queryOne(SysCurrencyRateQueryDTO queryDTO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.fromCurr, queryDTO.getFromCurr())
                .andEq(QDO.toCurr, queryDTO.getToCurr())
                .andEq(QDO.enabled, true)
                .andBefore(queryDTO.getValidDate() != null, QDO.validFrom, queryDTO.getValidDate())
                .andAfter(queryDTO.getValidDate() != null, QDO.validTo, queryDTO.getValidDate())
                .andAfter(queryDTO.getValidFromS() != null, QDO.validFrom, queryDTO.getValidFromS())
                .andBefore(queryDTO.getValidFromE() != null, QDO.validFrom, queryDTO.getValidFromE())
                .andAfter(queryDTO.getValidToS() != null, QDO.validTo, queryDTO.getValidToS())
                .andBefore(queryDTO.getValidToE() != null, QDO.validTo, queryDTO.getValidToE())
                .build();
        return super.jpaQueryFactory.selectFrom(QDO)
                .where(predicate)
                .orderBy(QDO.validFrom.asc())
                .limit(1)
                .fetchOne();
    }
}
