package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.bo.EmployeeOrgBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmployeeOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeOrgDO;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/8
 */
@Repository
public class EmployeeOrgRepoProc extends BaseRepoProc<SysEmployeeOrgDO> {
    private static final QSysEmployeeOrgDO QDO = QSysEmployeeOrgDO.sysEmployeeOrgDO;
    private static final QSysOrgDO QDO_ORG = QSysOrgDO.sysOrgDO;
    private static final QSysEmployeeDO QDO_EMP = QSysEmployeeDO.sysEmployeeDO;

    public EmployeeOrgRepoProc() {
        super(QDO);
    }

    /**
     * 修改员工上级领导
     *
     * @param employeeId       员工ID
     * @param orgId            组织ID
     * @param leaderEmployeeId 领导的员工ID
     * @param leaderUserId     领导的账号ID
     */
    public void updateLeader(long employeeId, long orgId, Long leaderEmployeeId, Long leaderUserId) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.leaderEmployeeId, leaderEmployeeId)
                .set(QDO.leaderUserId, leaderUserId)
                .where(QDO.employeeId.eq(employeeId).and(QDO.orgId.eq(orgId)))
                .execute();

    }

    /**
     * 根据员工ID删除
     *
     * @param employeeId
     */
    public void deleteByEmployeeId(long employeeId) {
        super.deleteByValue(QDO.employeeId, employeeId);
    }

    /**
     * 根据上级员工ID删除
     *
     * @param employeeId
     */
    public void deleteByLeaderEmployeeId(long employeeId) {
        super.deleteByValue(QDO.leaderEmployeeId, employeeId);
    }

    /**
     * 根据组织ID删除
     *
     * @param orgId
     */
    public void deleteByOrgId(long orgId) {
        super.deleteByValue(QDO.orgId, orgId);
    }

    /**
     * 删除员工组织关联
     *
     * @param employeeId
     * @param orgIds
     */
    public void delete(long employeeId, @NotEmpty Collection<Long> orgIds) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.employeeId.eq(employeeId).and(QDO.orgId.in(orgIds)))
                .execute();
    }

    /**
     * 删除员工组织关联
     *
     * @param employeeId
     * @param orgId
     */
    public void delete(long employeeId, long orgId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.employeeId.eq(employeeId).and(QDO.orgId.eq(orgId)))
                .execute();
    }

    /**
     * 获取员工的组织ID
     *
     * @param employeeId
     * @return
     */
    public List<Long> getOrgIdByEmployeeId(long employeeId) {
        return jpaQueryFactory.select(QDO.orgId)
                .from(QDO)
                .where(QDO.employeeId.eq(employeeId))
                .fetch();
    }

    /**
     * 获取用户的组织ID
     *
     * @param userId
     * @return
     */
    public List<Long> getOrgIdByUserId(long userId) {
        return jpaQueryFactory.select(QDO.orgId)
                .from(QDO)
                .where(QDO.userId.eq(userId))
                .fetch();
    }

    /**
     * 获取员工的员工组织配置
     *
     * @param employeeId 员工ID
     * @return 组织列表
     */
    public List<SysEmployeeOrgDO> getEmployeeOrgByEmployeeId(long employeeId) {
        return super.getListByValue(QDO.employeeId, employeeId);
    }

    /**
     * 获取员工的组织及其领导
     *
     * @param employeeId 员工ID
     * @return 部门ID与领导ID
     */
    public Map<Long, Long> getEmployeeOrgIdAndLeaderId(long employeeId) {
        return super.jpaQueryFactory.select(QDO.orgId, QDO.leaderUserId)
                .from(QDO)
                .where(QDO.employeeId.eq(employeeId).and(QDO.leaderUserId.isNotNull()))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.orgId), t -> t.get(QDO.leaderUserId), (t1, t2) -> t1));
    }

    /**
     * 获取员工的组织
     *
     * @param employeeIds 员工ID
     * @param enabled     是否启用的组织
     * @return 员工ID与组织
     */
    public Map<Long, List<IdCodeNameParam>> getEmployeeOrgs(@NotEmpty Collection<Long> employeeIds, Boolean enabled) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.employeeId, employeeIds)
                .andEq(enabled != null, QDO_ORG.enabled, enabled)
                .build();
        return jpaQueryFactory.select(QDO.employeeId, QDO_ORG.id, QDO_ORG.code, QDO_ORG.name, QDO.createTime)
                .from(QDO)
                .leftJoin(QDO_ORG).on(QDO.orgId.eq(QDO_ORG.id))
                .where(predicate)
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.employeeId),
                        Collectors.collectingAndThen(Collectors.toList(),
                                t -> t.stream()
                                        .sorted(Comparator.comparing(tt -> tt.get(QDO.createTime), Comparator.nullsFirst(LocalDateTime::compareTo)))
                                        .map(tt -> new IdCodeNameParam(tt.get(QDO_ORG.id), tt.get(QDO_ORG.code), tt.get(QDO_ORG.name)))
                                        .collect(Collectors.toList())))
                );
    }

    /**
     * 获取下属员工的用户ID
     *
     * @param leaderUserId 领导用户ID
     * @return 下属用户ID
     */
    public List<Long> getUserIdByLeaderUserId(long leaderUserId) {
        return super.jpaQueryFactory.select(QDO.userId)
                .from(QDO)
                .where(QDO.leaderUserId.eq(leaderUserId))
                .fetch();
    }

    /**
     * 获取领导
     *
     * @param employeeIds 当前员工
     * @return 领导信息
     */
    public List<EmployeeOrgBO> getLeaders(@NotEmpty Collection<Long> employeeIds) {
        var qBean = Projections.bean(EmployeeOrgBO.class,
                QDO.employeeId.as("ownerId"), QDO_EMP.type.as("employeeType"),
                QDO.leaderUserId.as("userId"), QDO.leaderEmployeeId.as("employeeId"),
                QDO_EMP.code.as("employeeCode"), QDO_EMP.username, QDO_EMP.firstName, QDO_EMP.lastName,
                QDO.orgId, QDO_ORG.code.as("orgCode"), QDO_ORG.name.as("orgName"));
        return super.jpaQueryFactory.select(qBean)
                .from(QDO)
                .leftJoin(QDO_EMP).on(QDO_EMP.id.eq(QDO.leaderEmployeeId))
                .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO.orgId))
                .where(QDO.employeeId.in(employeeIds))
                .fetch();
    }

    /**
     * 获取下属
     *
     * @param employeeIds 当前员工ID
     * @param enabled     启用状态
     * @return 领导信息
     */
    public List<EmployeeOrgBO> getUnderlings(@NotEmpty Collection<Long> employeeIds, Boolean enabled) {
        var qBean = Projections.bean(EmployeeOrgBO.class, QDO.leaderEmployeeId.as("ownerId"), QDO.userId, QDO.employeeId,
                QDO_EMP.code.as("employeeCode"), QDO_EMP.username, QDO_EMP.firstName, QDO_EMP.lastName,
                QDO.orgId, QDO_ORG.code.as("orgCode"), QDO_ORG.name.as("orgName"));
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.leaderEmployeeId, employeeIds)
                .and(enabled != null, () -> QDO_ORG.enabled.eq(enabled).and(QDO_EMP.enabled.eq(enabled)))
                .build();
        return super.jpaQueryFactory.select(qBean)
                .from(QDO)
                .leftJoin(QDO_EMP).on(QDO_EMP.id.eq(QDO.employeeId))
                .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO.orgId))
                .where(predicate)
                .fetch();
    }
}
