package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.platform.model.entity.*;
import com.elitescloud.cloudt.system.model.vo.resp.api.SysMenuApiRespVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 3/28/2023
 */
@Repository
public class MenuApiRepoProc extends BaseRepoProc<SysPlatformMenusApiDO> {
    private static final QSysPlatformMenusApiDO QDO = QSysPlatformMenusApiDO.sysPlatformMenusApiDO;
    private static final QSysPlatformApiManageDO QDO_API = QSysPlatformApiManageDO.sysPlatformApiManageDO;
    private static final QSysPlatformMenusDO QDO_MENU = QSysPlatformMenusDO.sysPlatformMenusDO;
    private static final QSysPlatformAppDO QDO_APP = QSysPlatformAppDO.sysPlatformAppDO;

    public MenuApiRepoProc() {
        super(QDO);
    }

    /**
     * 查询菜单下挂载的API数量
     *
     * @param appCodes 应用编码
     * @return 菜单与API数量关联
     */
    public Map<String, Long> queryNumOfApi(Collection<String> appCodes) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.appCode, appCodes)
                .build();
        return super.jpaQueryFactory.select(QDO.menusCode, QDO.apiId)
                .from(QDO)
                .where(predicate)
                .fetch().stream()
                .collect(Collectors.groupingBy(t -> ObjectUtil.defaultIfNull(t.get(QDO.menusCode), "unknown"), Collectors.counting()));
    }

    /**
     * 获取菜单下绑定的API
     *
     * @param menuCode 菜单编码
     * @return API列表
     */
    public List<SysPlatformApiManageDO> queryApiOfMenu(@NotBlank String menuCode) {
        return super.jpaQueryFactory.selectFrom(QDO_API)
                .where(QDO_API.id.in(
                        JPAExpressions.select(QDO.apiId).from(QDO).where(QDO.menusCode.eq(menuCode))
                )).fetch();
    }

    /**
     * 查询菜单下的APi
     *
     * @param appCodes
     * @return
     */
    public Map<String, List<IdCodeNameParam>> queryApiOfMenu(Collection<String> appCodes) {
        return super.jpaQueryFactory.select(QDO.menusCode, QDO_API.id, QDO_API.apiCode, QDO_API.apiName)
                .from(QDO)
                .leftJoin(QDO_API).on(QDO.apiId.eq(QDO_API.id))
                .where(QDO.appCode.in(appCodes))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.menusCode),
                        Collectors.mapping(t -> new IdCodeNameParam(t.get(QDO_API.id), t.get(QDO_API.apiCode), t.get(QDO_API.apiName)), Collectors.toList())));
    }

    /**
     * 查询菜单下的APi
     *
     * @param appCodes
     * @return
     */
    public Map<String, List<SysMenuApiRespVO>> queryApiDetailOfMenu(Collection<String> appCodes) {
        QBean<SysMenuApiRespVO> qBean = Projections.bean(SysMenuApiRespVO.class, QDO_API.id, QDO_API.appCode, QDO_API.apiName.as("name"),
                QDO_API.apiCode.as("code"), QDO_API.requestType, QDO_API.apiPath, QDO_API.apiDescribe,
                QDO_APP.appName,
                QDO.menusCode);

        return super.jpaQueryFactory.select(qBean)
                .from(QDO)
                .leftJoin(QDO_API).on(QDO_API.id.eq(QDO.apiId))
                .leftJoin(QDO_APP).on(QDO_API.appCode.eq(QDO_APP.appCode))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(SysMenuApiRespVO::getMenusCode));
    }

    public List<Long> getMenuIdByApiId(long apiId, String appCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.apiId, apiId)
                .andEq(QDO.appCode, appCode)
                .build();
        return super.jpaQueryFactory.select(QDO.menusId)
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 查询接口挂载的菜单
     *
     * @param apiId
     * @param appCode
     * @return
     */
    public List<SysPlatformMenusDO> getMenuDetailByApiId(long apiId, String appCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.apiId, apiId)
                .andEq(QDO.appCode, appCode)
                .build();

        return super.jpaQueryFactory.selectFrom(QDO_MENU)
                .where(QDO_MENU.id.in(
                        JPAExpressions.select(QDO.menusId)
                                .from(QDO)
                                .where(predicate)
                )).fetch();
    }
}
