package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.platform.model.constant.PlatformAppMenusTypeEnum;
import com.elitescloud.cloudt.platform.model.constant.PlatformMenusNodeEnum;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformMenusDO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformMenusDO;
import com.elitescloud.cloudt.system.model.bo.MenuBO;
import com.elitescloud.cloudt.system.model.bo.MenuSimpleBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/14
 */
@Repository
public class MenuRepoProc extends BaseRepoProc<SysPlatformMenusDO> {
    private static final QSysPlatformMenusDO QDO = QSysPlatformMenusDO.sysPlatformMenusDO;
    private static final QSysPlatformAppDO QDO_APP = QSysPlatformAppDO.sysPlatformAppDO;

    public MenuRepoProc() {
        super(QDO);
    }

    /**
     * 根据菜单名称模糊查询菜单编码
     *
     * @param menuNameLike
     * @return
     */
    public List<String> queryMenuCodeByMenuNameLike(@NotBlank String menuNameLike) {
        return jpaQueryFactory.select(QDO.menusCode)
                .from(QDO)
                .where(QDO.menusName.like("%" + menuNameLike + "%"))
                .fetch();
    }

    /**
     * 根据菜单编码获取菜单名称
     *
     * @param menuCode
     * @return
     */
    public String getMenuNameByMenuCode(@NotBlank String menuCode) {
        return super.getValueByValue(QDO.menusName, QDO.menusCode, menuCode);
    }

    /**
     * 根据菜单编码获取应用编码
     *
     * @param menuCode
     * @return
     */
    public String getAppCodeByMenuCode(@NotBlank String menuCode) {
        return super.getValueByValue(QDO.menusAppCode, QDO.menusCode, menuCode);
    }

    /**
     * 根据菜单编码获取应用ID
     *
     * @param menuCode
     * @return
     */
    public Long getAppIdByMenuCode(@NotBlank String menuCode) {
        return super.jpaQueryFactory.select(QDO_APP.id)
                .from(QDO)
                .leftJoin(QDO_APP).on(QDO_APP.appCode.eq(QDO.menusAppCode))
                .where(QDO.menusCode.eq(menuCode))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据菜单编码获取菜单类型
     *
     * @param menuCode 菜单编码
     * @return 菜单类型
     */
    public String getMenuTypeByMenuCode(@NotBlank String menuCode) {
        return super.getValueByValue(QDO.menusType, QDO.menusCode, menuCode);
    }

    /**
     * 根据菜单编码获取菜单名称
     *
     * @param menuCodes
     * @return
     */
    public Map<String, String> getMenuNameByMenuCode(@NotEmpty Collection<String> menuCodes) {
        return jpaQueryFactory.select(QDO.menusCode, QDO.menusName)
                .from(QDO)
                .where(QDO.menusCode.in(menuCodes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.menusCode), t -> t.get(QDO.menusName), (t1, t2) -> t1));
    }

    /**
     * 根据菜单编码获取菜单
     *
     * @param menuCode
     * @return
     */
    public MenuBO getByMenuCode(@NotBlank String menuCode) {
        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(QDO.menusCode.eq(menuCode))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据菜单编码获取菜单
     *
     * @param menuCodes
     * @return
     */
    public List<MenuBO> listByMenuCode(@NotEmpty Collection<String> menuCodes) {
        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(QDO.menusCode.in(menuCodes))
                .fetch();
    }

    /**
     * 根据菜单编码获取菜单
     *
     * @param menuCodes
     * @return
     */
    public List<MenuSimpleBO> listSimpleByMenuCode(@NotEmpty Collection<String> menuCodes) {
        return jpaQueryFactory.select(qBeanSimpleMenu())
                .from(QDO)
                .where(QDO.menusCode.in(menuCodes))
                .fetch();
    }

    /**
     * 根据应用编码获取菜单
     *
     * @param appCodes
     * @return
     */
    public List<MenuBO> queryMenuByAppCode(Set<String> appCodes, boolean includeAction, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andIn(!CollectionUtils.isEmpty(appCodes), QDO.menusAppCode, appCodes)
                .andIn(!includeAction, QDO.nodeType, nodeTypeOfMenu())
                .andEq(QDO.menusState, enabled)
                .build();

        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据应用编码获取菜单
     *
     * @param appCodes
     * @return
     */
    public List<MenuBO> queryMenuByAppCode(Set<String> appCodes, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andIn(!CollectionUtils.isEmpty(appCodes), QDO.menusAppCode, appCodes)
                .andIn(true, QDO.nodeType, nodeTypeOfMenu())
                .andEq(QDO.menusState, enabled)
                .build();

        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 查询管理菜单
     *
     * @param appCodes
     * @param includeAction
     * @return
     */
    public List<MenuBO> queryMenuForMng(Set<String> appCodes, boolean includeAction, Boolean enabled) {
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.menusType, PlatformAppMenusTypeEnum.MENUS_TYPE_SYS.name())
                .andIn(CollUtil.isNotEmpty(appCodes), QDO.menusAppCode, appCodes)
                .andIn(!includeAction, QDO.nodeType, nodeTypeOfMenu())
                .andEq(QDO.menusState, enabled)
                .build();
        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据应用编码获取按钮
     *
     * @param appCodes
     * @return
     */
    public List<MenuBO> queryActionByAppCode(Set<String> appCodes, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andIn(!CollectionUtils.isEmpty(appCodes), QDO.menusAppCode, appCodes)
                .andEq(true, QDO.nodeType, PlatformMenusNodeEnum.BUTTON.name())
                .andEq(QDO.menusState, enabled)
                .build();

        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据菜单编码查询下级菜单
     *
     * @param menuCode
     * @return
     */
    public List<MenuBO> queryActionByMenuCode(String menuCode, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(menuCode), QDO.menusParentCode, menuCode)
                .andEq(true, QDO.nodeType, PlatformMenusNodeEnum.BUTTON.name())
                .andEq(QDO.menusState, enabled)
                .build();

        return jpaQueryFactory.select(qBeanMenu())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    private Set<String> nodeTypeOfMenu() {
        return Set.of(PlatformMenusNodeEnum.MENUS.name(), PlatformMenusNodeEnum.MENUS_GROUP.name());
    }

    private QBean<MenuBO> qBeanMenu() {
        return Projections.bean(MenuBO.class, QDO.id, QDO.menusAppCode, QDO.menusName, QDO.menusType, QDO.nodeType, QDO.menusCode,
                QDO.menusOrder, QDO.menusState, QDO.menusParentCode,
                QDO.menusRoute, QDO.menusIcon, QDO.display, QDO.outerLink, QDO.outerLinkType);
    }

    private QBean<MenuSimpleBO> qBeanSimpleMenu() {
        return Projections.bean(MenuSimpleBO.class, QDO.id, QDO.menusAppCode, QDO.menusName, QDO.menusCode);
    }
}
