package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.jpa.common.BaseTreeRepoProc;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.OrgTreeNodeType;
import com.elitescloud.cloudt.system.convert.OrgConvert;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.req.SysOrgPageQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonOrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.OrgPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOrgDO;
import com.elitescloud.cloudt.system.service.model.entity.SysOrgDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.StringExpression;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/29
 */
@Repository
public class OrgRepoProc extends BaseTreeRepoProc<SysOrgDO> {
    private static final QSysOrgDO QDO = QSysOrgDO.sysOrgDO;
    private static final OrgConvert CONVERT = OrgConvert.INSTANCE;

    public OrgRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 判断编号是否存在
     *
     * @param code 编号
     * @return 是否存在
     */
    public boolean existsCode(@NotBlank String code) {
        return super.exists(QDO.code, code);
    }

    /**
     * 获取启用状态
     *
     * @param id ID
     * @return 启用状态
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 获取是否行政组织
     *
     * @param id ID
     * @return 是否行政组织
     */
    public Boolean getExecutive(long id) {
        return super.getValue(QDO.executive, id);
    }

    /**
     * 根据组织编号获取ID
     *
     * @param code 编号
     * @return ID
     */
    public Long getIdByCode(@NotBlank String code) {
        return super.getIdByValue(QDO.code, code);
    }

    /**
     * 根据编码获取ID
     *
     * @param codes
     * @return
     */
    public Map<String, Long> getIdByCode(Collection<String> codes) {
        return super.jpaQueryFactory.select(QDO.id, QDO.code)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> t.get(QDO.id), (t1, t2) -> t1));
    }

    /**
     * 根据组织名称获取ID
     *
     * @param name 组织名称
     * @return ID
     */
    public Long getIdByName(@NotBlank String name) {
        return super.getIdByValue(QDO.name, name);
    }

    /**
     * 根据组织名称获取ID
     *
     * @param name 组织名称
     * @return ID
     */
    public List<Long> getIdsByName(@NotBlank String name) {
        return super.jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.name.eq(name))
                .fetch();
    }

    /**
     * 查询ID编码和名称
     *
     * @param ids ID
     * @return id编码和名称
     */
    public List<IdCodeNameParam> queryIdCodeName(@NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch();
    }

    /**
     * 查询ID编码和名称
     *
     * @param codes 编码
     * @return id编码和名称
     */
    public List<IdCodeNameParam> queryIdCodeNameByCodes(@NotEmpty Collection<String> codes) {
        return jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch();
    }

    /**
     * 根据ID获取编码
     *
     * @param id
     * @return
     */
    public String getCodeById(long id) {
        return super.getValue(QDO.code, id);
    }

    /**
     * 根据ID获取名称
     *
     * @param id
     * @return
     */
    public String getNameById(long id) {
        return super.getValue(QDO.name, id);
    }

    /**
     * 获取序号
     *
     * @param id
     * @return
     */
    public Integer getSortNo(long id) {
        return super.getValue(QDO.sortNo, id);
    }

    /**
     * 获取根组织ID
     *
     * @param id
     * @return
     */
    public Long getRootId(long id) {
        return super.getValue(QDO.rootId, id);
    }

    /**
     * 获取上级ID
     *
     * @param code
     * @return
     */
    public Long getParentIdByCode(String code) {
        return super.getValueByValue(QDO.pId, QDO.code, code);
    }

    /**
     * 获取上级编码
     *
     * @param codes
     * @return
     */
    public Map<String, String> getParentCodeByCode(Collection<String> codes) {
        return super.jpaQueryFactory.select(QDO.code, QDO.parentCode)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> t.get(QDO.parentCode), (t1, t2) -> t1));
    }

    /**
     * 获取最大序号
     *
     * @param parentCode 父节点
     * @return 最大序号
     */
    public Integer getMaxSortNo(@NotBlank String parentCode) {
        return jpaQueryFactory.select(QDO.sortNo.max())
                .from(QDO)
                .where(QDO.parentCode.eq(parentCode))
                .fetchOne();
    }

    /**
     * 获取左侧最大节点组织ID
     *
     * @param parentId
     * @param sortNo
     * @return
     */
    public Long getLeftOrgId(long parentId, int sortNo) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.pId.eq(parentId).and(QDO.sortNo.lt(sortNo)))
                .orderBy(QDO.sortNo.asc())
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据编码获取组织名称
     *
     * @param codes
     * @return
     */
    public Map<String, String> getNameByCode(@NotEmpty Collection<String> codes) {
        return jpaQueryFactory.select(QDO.name, QDO.code)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> t.get(QDO.name), (t1, t2) -> t1));
    }

    /**
     * 获取所有组织ID
     *
     * @param tenantId
     * @return
     */
    public Set<Long> queryAllOrgIds(Long tenantId, boolean includeDisabled) {
        if (tenantId == null) {
            tenantId = TenantConstant.DEFAULT_TENANT_ID;
        }
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.tenantId, tenantId)
                .andEq(!includeDisabled, QDO.enabled, true)
                .build();
        return new HashSet<>(jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(predicate)
                .fetch());
    }

    /**
     * 获取下属组织ID
     *
     * @param orgIdBelong
     * @param includeDisabled
     * @return
     */
    public Set<Long> queryOrgIdsByBelong(long orgIdBelong, boolean includeDisabled) {
        var predicate = super.predicateForChildren(orgIdBelong);
        if (!includeDisabled) {
            predicate = predicate.and(QDO.enabled.eq(true));
        }
        return new HashSet<>(jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(predicate)
                .fetch());
    }

    /**
     * 获取组织树
     *
     * @param orgIdBelong
     * @param includeDisabled
     * @return
     */
    public List<OrgTreeNodeRespVO> getOrgTree(Long orgIdBelong, boolean includeDisabled) {
        return super.getTree(orgIdBelong, QDO.executive.eq(true), t -> includeDisabled || t.getEnabled(), t -> {
            var respVO = CONVERT.do2TreeNodeRespVO(t);
            respVO.setName(t.getPrettyName());
            respVO.setParentId(t.getPId());
            respVO.setParentCode(t.getParentCode());
            respVO.setNodeType(OrgTreeNodeType.ORG.getValue());
            respVO.setNodeTypeName(OrgTreeNodeType.ORG.getDescription());
            respVO.setDataType(t.getType());

            return respVO;
        });
    }

    /**
     * 获取组织树
     *
     * @param pId             父节点
     * @param includeDisabled
     * @return
     */
    public List<OrgTreeNodeRespVO> getOrgTreeByParentId(Long pId, boolean includeDisabled) {
        if (pId == null) {
            pId = BaseTreeModel.DEFAULT_PARENT;
        }

        Predicate predicate = QDO.pId.eq(pId);
        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.enabled.eq(true));
        }

        return jpaQueryFactory.select(QDO.id, QDO.code, QDO.name, QDO.shortName, QDO.type, QDO.sortNo, QDO.pId, QDO.lft, QDO.rgt)
                .from(QDO)
                .where(predicate)
                .fetch()
                .stream()
                .map(t -> {
                    OrgTreeNodeRespVO respVO = new OrgTreeNodeRespVO();
                    respVO.setId(t.get(QDO.id));
                    respVO.setCode(t.get(QDO.code));
                    respVO.setName(prettyName(t.get(QDO.name), t.get(QDO.shortName)));
                    respVO.setNodeType(OrgTreeNodeType.ORG.getValue());
                    respVO.setNodeTypeName(OrgTreeNodeType.ORG.getDescription());
                    respVO.setDataType(t.get(QDO.type));
                    respVO.setSortNo(t.get(QDO.sortNo));
                    respVO.setParentId(t.get(QDO.pId));
                    respVO.setHasChildren(super.hasChildren(t.get(QDO.lft), t.get(QDO.rgt)));
                    respVO.setChildren(Collections.emptyList());

                    return respVO;
                })
                .collect(Collectors.toList());
    }

    /**
     * 获取组织信息
     *
     * @param id
     * @return
     */
    public SysOrgBasicDTO getBasicDto(long id) {
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(QDO.id.eq(id))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据编码获取组织信息
     *
     * @param code 组织编码
     * @return 组织信息
     */
    public SysOrgDO getByCode(@NotBlank String code) {
        return super.getOneByValue(QDO.code, code);
    }

    /**
     * 获取组织信息
     *
     * @param code 组织编码
     * @return 组织信息
     */
    public SysOrgBasicDTO getBasicDtoByCode(@NotBlank String code) {
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(QDO.code.eq(code))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取上级组织信息
     *
     * @param code
     * @return
     */
    public SysOrgBasicDTO getParentBasicDtoByCode(@NotBlank String code) {
        var child = new QSysOrgDO("_parent");
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(QDO.id.eq(
                        JPAExpressions.select(child.pId).from(child).where(child.code.eq(code)).limit(1)
                ))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取上级组织信息
     *
     * @param codes
     * @return
     */
    public List<SysOrgBasicDTO> getParentBasicDtoByCode(Collection<String> codes) {
        var child = new QSysOrgDO("_parent");
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(QDO.id.in(
                        JPAExpressions.select(child.pId).from(child).where(child.code.in(codes))
                ))
                .fetch();
    }

    /**
     * 获取组织列表信息
     *
     * @param ids
     * @return
     */
    public List<SysOrgBasicDTO> getBasicDtoList(Collection<Long> ids) {
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch();
    }

    /**
     * 获取上级所有组织
     *
     * @param id
     * @return
     */
    public List<SysOrgBasicDTO> getBasicDtoOfParents(Long id) {
        var predicate = predicateForParents(id);
        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.lft.asc())
                .fetch();
    }

    /**
     * 查询组织列表
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    public List<SysOrgBasicDTO> queryList(SysOrgQueryDTO queryDTO) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.id, queryDTO.getIds())
                .andIn(QDO.code, queryDTO.getCodes())
                .andEq(QDO.parentCode, queryDTO.getParentCode())
                .andEq(QDO.executive, queryDTO.getExecutive())
                .andEq(Boolean.TRUE.equals(queryDTO.getRoot()), QDO.pId, BaseTreeModel.DEFAULT_PARENT)
                .andNe(Boolean.FALSE.equals(queryDTO.getRoot()), QDO.pId, BaseTreeModel.DEFAULT_PARENT)
                .andEq(QDO.type, queryDTO.getType())
                .andEq(QDO.entity, queryDTO.getEntity())
                .andEq(QDO.enabled, queryDTO.getEnabled())
                .andEq(QDO.ouId, queryDTO.getOuId())
                .and(queryDTO.getOrgIdBelong() != null, () -> super.predicateForChildren(queryDTO.getOrgIdBelong()).and(QDO.executive.eq(true)))
                .andLike(QDO.name, queryDTO.getName())
                .build();
        if (predicate == null) {
            return Collections.emptyList();
        }

        return jpaQueryFactory.select(qBeanSysOrgBasicDto())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 分页查询管理组织树
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysOrgDO> pageMng(OrgPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.parentCode, queryVO.getParentCode())
                .andEq(queryVO.getOuId() != null, QDO.ouId, queryVO.getOuId())
                .andLike(QDO.code, queryVO.getCode())
                .andLike(QDO.name, queryVO.getName())
                .andEq(queryVO.getExecutive() != null, QDO.executive, queryVO.getExecutive())
                .andEq(QDO.type, queryVO.getType())
                .andEq(queryVO.getEntity() != null, QDO.entity, queryVO.getEntity())
                .andEq(queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest());
    }

    /**
     * 分页查询组织
     *
     * @param queryVO 查询参数
     * @return 组织列表
     */
    public PagingVO<SysOrgDO> pageQuery(CommonOrgPageQueryVO queryVO) {
        Supplier<Predicate> predicateChildren = null;
        if (queryVO.getExcludeBelongId() != null) {
            // 如果节点是根节点，则忽略
            String parentCode = super.getValueByValue(QDO.parentCode, QDO.id, queryVO.getExcludeBelongId());
            predicateChildren = StringUtils.hasText(parentCode) ? () -> super.predicateForChildren(queryVO.getExcludeBelongId()).not()
                    : () -> QDO.id.ne(queryVO.getExcludeBelongId());
        }
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.ouId, queryVO.getOuId())
                .andLike(QDO.code, queryVO.getCode())
                .andLike(QDO.name, queryVO.getName())
                .andEq(QDO.executive, queryVO.getExecutive())
                .andEq(QDO.type, queryVO.getType())
                .and(predicateChildren != null, predicateChildren)
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest());
    }

    /**
     * 分页查询组织
     *
     * @param queryDTO 查询参数
     * @return 组织列表
     */
    public PagingVO<SysOrgDO> pageQuery(SysOrgPageQueryDTO queryDTO) {
        Long orgIdBelong = queryDTO.getOrgIdBelong();
        if (orgIdBelong == null && StringUtils.hasText(queryDTO.getOrgCodeBelong())) {
            orgIdBelong = this.getIdByCode(queryDTO.getOrgCodeBelong());
        }
        Long finalOrgIdBelong = orgIdBelong;
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.id, queryDTO.getIds())
                .andIn(QDO.code, queryDTO.getCodes())
                .andEq(QDO.pId, queryDTO.getParentId())
                .andEq(QDO.parentCode, queryDTO.getParentCode())
                .andEq(QDO.executive, queryDTO.getExecutive())
                .andEq(QDO.type, queryDTO.getType())
                .andEq(QDO.entity, queryDTO.getEntity())
                .andEq(QDO.enabled, queryDTO.getEnabled())
                .andEq(QDO.ouId, queryDTO.getOuId())
                .and(orgIdBelong != null, () -> super.predicateForChildren(finalOrgIdBelong).and(QDO.executive.eq(true)))
                .andLike(QDO.name, queryDTO.getName())
                .andRightLike(QDO.code, queryDTO.getCode())
                .andLike(QDO.shortName, queryDTO.getShortName())
                .andLike(new StringExpression[]{QDO.code, QDO.name, QDO.shortName}, queryDTO.getKeyword())
                .build();
        return super.queryByPage(predicate, queryDTO.getPageRequest());
    }

    /**
     * 获取根组织ID
     * <p>
     * 原则上行政根组织是只有一个
     *
     * @return 根组织ID
     */
    public Long getRootOrgId() {
        return super.jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.parentCode.isNull().or(QDO.parentCode.eq("")).and(QDO.executive.eq(true)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 查询编码和名称
     *
     * @param rootId    根组织ID
     * @param executive 是否行政组织
     * @return 编码和名称列表
     */
    public List<IdCodeNameParam> queryCodeName(Long rootId, Boolean executive) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.executive, executive)
                .andEq(QDO.rootId, rootId)
                .build();
        return jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 查询组织的指定类型的上级
     *
     * @param ids
     * @param type
     * @param onlyEnabled
     * @return
     */
    public Map<Long, IdCodeNameParam> queryParentNameForType(@NotEmpty Collection<Long> ids, @NotBlank String type, boolean onlyEnabled) {
        // 先查询ID及其上级ID
        var qdoParent = new QSysOrgDO("child");
        Map<Long, Long> withParentIdMap = super.filterParentId(qdoParent,
                qdoParent.type.eq(type).and(onlyEnabled ? qdoParent.enabled.eq(true) : null),
                QDO.id.in(ids).and(QDO.executive.eq(true)));

        // 查询上级的ID、编码和名称
        var parentIds = withParentIdMap.values().stream().filter(t -> t != -1).collect(Collectors.toSet());
        if (parentIds.isEmpty()) {
            return Collections.emptyMap();
        }
        var withParentMap = this.queryIdCodeName(parentIds).stream().collect(Collectors.toMap(IdCodeNameParam::getId, t -> t, (t1, t2) -> t1));

        // 返回结果
        Map<Long, IdCodeNameParam> idWithParentMap = new HashMap<>(withParentIdMap.size());
        for (Map.Entry<Long, Long> entry : withParentIdMap.entrySet()) {
            var withParent = withParentMap.get(entry.getValue());
            if (withParent == null) {
                continue;
            }
            idWithParentMap.put(entry.getKey(), withParent);
        }
        return idWithParentMap;
    }

    /**
     * 包含下属所有组织
     *
     * @param id 组织ID
     * @return 包含下属所有组织的条件
     */
    BooleanExpression predicateOfOrgChildren(Long id) {
        return super.predicateForChildren(id);
    }

    private String prettyName(String name, String shortName) {
        return CharSequenceUtil.blankToDefault(shortName, name);
    }

    private QBean<IdCodeNameParam> qBeanIdCodeName() {
        return Projections.bean(IdCodeNameParam.class, QDO.id, QDO.code, QDO.name);
    }

    private QBean<SysOrgBasicDTO> qBeanSysOrgBasicDto() {
        return Projections.bean(SysOrgBasicDTO.class, QDO.id, QDO.code, QDO.name, QDO.pId.as("parentId"), QDO.parentCode, QDO.shortName,
                QDO.type, QDO.enabled, QDO.ouId, QDO.remark);
    }
}
