package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysRoleBusiness;
import com.elitescloud.cloudt.system.constant.RoleType;
import com.elitescloud.cloudt.system.model.vo.query.role.RolePageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Repository
public class RoleRepoProc extends BaseRepoProc<SysRoleDO> {
    private static final QSysRoleDO QDO = QSysRoleDO.sysRoleDO;

    public RoleRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 清楚角色组的关联关系
     *
     * @param groupId
     */
    public void clearGroupId(long groupId) {
        super.updateValueByValue(QDO.groupId, null, QDO.groupId, groupId);
    }

    /**
     * 判断编号是否存在
     *
     * @param code 编号
     * @return 是否存在
     */
    public boolean existsCode(@NotBlank String code) {
        return super.exists(QDO.code, code);
    }

    /**
     * 获取启用状态
     *
     * @param id
     * @return
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 根据角色编码获取角色ID
     *
     * @param codes 角色编码
     * @return 角色ID
     */
    public Set<Long> getIds(@NotEmpty Collection<String> codes) {
        return super.getIdsByValue(QDO.code, codes).stream().collect(Collectors.toSet());
    }

    /**
     * 根据角色编码获取角色ID
     *
     * @param code 角色编码
     * @return 角色ID
     */
    public Long getIdByCode(@NotBlank String code) {
        return super.getIdByValue(QDO.code, code);
    }

    /**
     * 根据角色编码获取角色ID
     *
     * @param codes 角色编码
     * @return 角色ID
     */
    public Map<String, Long> getIdAndCodeByCode(@NotEmpty Collection<String> codes) {
        return super.jpaQueryFactory.select(QDO.id, QDO.code)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.code), t -> t.get(QDO.id), (t1, t2) -> t1));
    }

    /**
     * 过滤出启用状态的编码
     *
     * @param codes
     * @return
     */
    public Set<String> filterEnabled(Collection<String> codes) {
        return new HashSet<>(jpaQueryFactory.select(QDO.code)
                .from(QDO)
                .where(QDO.code.in(codes).and(QDO.enabled.eq(true)))
                .fetch());
    }

    /**
     * 过滤出启用状态的编码
     *
     * @param codes
     * @return
     */
    public Set<Long> filterEnabledId(Collection<String> codes) {
        return new HashSet<>(jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.code.in(codes).and(QDO.enabled.eq(true)))
                .fetch());
    }

    /**
     * 根据分组统计各分组下的数量
     *
     * @param groupIds
     * @return
     */
    public Map<Long, Long> countByGroupId(Collection<Long> groupIds) {
        return jpaQueryFactory.select(QDO.groupId, QDO.id)
                .from(QDO)
                .where(QDO.groupId.in(groupIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.groupId), Collectors.counting()))
                ;
    }

    /**
     * 分页查询角色
     * <p>
     * 非管理员时不能查询系统角色
     *
     * @return 角色列表
     */
    public PagingVO<SysRoleDO> pageMng(RoleType type, String typeId, RolePageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.code, queryVO.getCode())
                .andLike(QDO.name, queryVO.getName())
                .andEq(type != null, QDO.type, type.getValue())
                .andEq(QDO.typeId, typeId)
                .andEq(QDO.businessKey, queryVO.getBusinessKey())
                .andNe(type == null || type == RoleType.CUSTOM, QDO.businessKey, SysRoleBusiness.SYS.getValue())
                .andEq(QDO.enabled, queryVO.getEnabled())
                .and(queryVO.getGroupId() != null, () -> {
                    if (SysRoleDO.DEFAULT_GROUP_ID == queryVO.getGroupId()) {
                        return QDO.groupId.eq(SysRoleDO.DEFAULT_GROUP_ID).or(QDO.groupId.isNull());
                    }
                    return QDO.groupId.eq(queryVO.getGroupId());
                })
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 查询角色列表
     *
     * @param type        类型
     * @param typeId      类型标识
     * @param businessKey 业务标识
     * @param name        菜单名称
     * @return 菜单列表
     */
    public List<IdCodeNameParam> listQuery(RoleType type, String typeId, String businessKey, String name) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.name, name)
                .andEq(type != null, QDO.type, type.getValue())
                .andEq(QDO.typeId, typeId)
                .andEq(QDO.businessKey, businessKey)
                .andNe(type == null || type == RoleType.CUSTOM, QDO.businessKey, SysRoleBusiness.SYS.getValue())
                .build();
        return super.jpaQueryFactory.select(Projections.bean(IdCodeNameParam.class, QDO.id, QDO.code, QDO.name))
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    public List<IdCodeNameParam> queryIdCodeNames(Collection<Long> ids) {
        return super.jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch();
    }

    public List<IdCodeNameParam> queryIdCodeNamesByCode(Collection<String> codes) {
        return super.jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch();
    }

    private QBean<IdCodeNameParam> qBeanIdCodeName() {
        return Projections.bean(IdCodeNameParam.class, QDO.id, QDO.code, QDO.name);
    }
}
