package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplQueryParam;
import com.elitescloud.cloudt.system.service.model.entity.QSysTmplDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTmplDO;
import com.querydsl.core.types.Predicate;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/31
 */
@Repository
public class SysTmplRepoProc extends BaseRepoProc<SysTmplDO> {

    private static final QSysTmplDO QDO = QSysTmplDO.sysTmplDO;

    public SysTmplRepoProc() {
        super(QDO);
    }

    /**
     * 更新删除标识
     *
     * @param id      模板ID
     * @param deleted 删除标识
     */
    public void updateDeleted(long id, int deleted) {
        super.updateDeleteFlag(id, deleted);
    }

    /**
     * 更新模板文件标识
     *
     * @param id       模板ID
     * @param fileCode 文件标识
     */
    public void updateFileCode(long id, String fileCode) {
        super.updateValue(QDO.fileCode, fileCode, id);
    }

    /**
     * 启用或停用
     *
     * @param id 版本记录ID
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 判断编号是否存在
     *
     * @param code 模板编号
     * @return 是否存在
     */
    public boolean existsByCode(String code) {
        return super.exists(QDO.code, code);
    }

    /**
     * 判断编号是否存在
     *
     * @param code 模板名称
     * @param id   模板ID
     * @return 是否存在
     */
    public boolean existsByCode(String code, Long id) {
        return super.exists(QDO.code, code, id);
    }

    /**
     * 根据模板编号获取模板ID
     *
     * @param code 模板编码
     * @return 模板ID
     */
    public Long getIdByCode(String code) {
        return super.getIdByValue(QDO.code, code);
    }

    /**
     * 获取编码
     *
     * @param id 模板ID
     * @return 模板编码
     */
    public String getCode(Long id) {
        return super.getValue(QDO.code, id);
    }

    /**
     * 获取是否启用
     *
     * @param id 模板ID
     * @return 是否启用
     */
    public Boolean getEnabled(Long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 获取是否是导出
     *
     * @param id 模板ID
     * @return 是否启用
     */
    public Boolean getExport(Long id) {
        return super.getValue(QDO.export, id);
    }

    /**
     * 获取文件编码
     *
     * @param id 模板ID
     * @return 模板文件编码
     */
    public String getFileCode(Long id) {
        return super.getValue(QDO.fileCode, id);
    }

    /**
     * 根据模板编号获取文件编码
     *
     * @param code 模板编号
     * @return 模板文件编码
     */
    public String getFileCodeByCode(String code) {
        return super.getValueByValue(QDO.fileCode, QDO.code, code);
    }

    /**
     * 根据编号获取模板信息
     *
     * @param code 模板编号
     * @return 模板信息
     */
    public SysTmplDO getByCode(String code) {
        return super.getOneByValue(QDO.code, code);
    }

    /**
     * 分页查询管理
     *
     * @param queryParam 查询参数
     * @return 数据列表
     */
    public PagingVO<SysTmplDO> pageMng(TmplQueryParam queryParam) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(queryParam.getAppCode()), QDO.appCode, queryParam.getAppCode())
                .andEq(QDO.code, queryParam.getTmplCode())
                .andLike(QDO.name, queryParam.getKeyword())
                .andEq(queryParam.getExport() != null, QDO.export, queryParam.getExport())
                .build();

        return super.queryByPage(predicate, queryParam.getPageRequest());
    }

    /**
     * 分页查询
     *
     * @param queryParam 查询参数
     * @return 数据列表
     */
    public PagingVO<SysTmplDO> pageQuery(TmplQueryParam queryParam) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(queryParam.getAppCode()), QDO.appCode, queryParam.getAppCode())
                .andEq(QDO.code, queryParam.getTmplCode())
                .andLike(QDO.name, queryParam.getKeyword())
                .andEq(queryParam.getExport() != null, QDO.export, queryParam.getExport())
                .andEq(QDO.enabled, true)
                .build();

        return super.queryByPage(predicate, queryParam.getPageRequest());
    }
}
