package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.req.ThirdApiLogQueryDTO;
import com.elitescloud.cloudt.system.model.entity.QSysThirdApiBusinessDO;
import com.elitescloud.cloudt.system.model.entity.QSysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.vo.query.extend.ThirdApiLogQueryVO;
import org.springframework.stereotype.Repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/4
 */
@Repository
public class ThirdApiLogRepoProc extends BaseRepoProc<SysThirdApiLogDO> {
    private static final QSysThirdApiLogDO QDO = QSysThirdApiLogDO.sysThirdApiLogDO;
    private static final QSysThirdApiBusinessDO QDO_BUSINESS = QSysThirdApiBusinessDO.sysThirdApiBusinessDO;

    public ThirdApiLogRepoProc() {
        super(QDO);
    }

    /**
     * 更新重试记录ID
     *
     * @param id
     * @param lastRetryLogId
     */
    public void updateLastRetryLogId(long id, long lastRetryLogId) {
        super.updateValue(QDO.lastRetryId, lastRetryLogId, id);
    }

    /**
     * 更新发送结果
     *
     * @param id
     * @param success
     * @param failReason
     */
    public void updateReqResult(long id, boolean success, String failReason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.reqSuccess, success)
                .set(QDO.retryFailReason, failReason)
                .set(QDO.respTime, LocalDateTime.now())
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();

    }

    /**
     * 更新重试次数
     *
     * @param id
     */
    public void updateRetryTimes(long id) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新响应结果
     *
     * @param id
     * @param success
     * @param failReason
     * @param respBody
     */
    public void updateRespResult(long id, boolean success, String failReason, String respBody) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();
        jpaQueryFactory.update(QDO)
                .set(QDO.reqSuccess, true)
                .set(QDO.respSuccess, success)
                .set(QDO.respTime, LocalDateTime.now())
                .set(QDO.retryFailReason, failReason)
                .set(QDO.respBody, respBody)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新请求时间
     *
     * @param id
     * @param reqTime
     */
    public void updateReqTime(long id, LocalDateTime reqTime) {
        super.updateValue(QDO.reqTime, reqTime, id);
    }

    /**
     * 更新重试失败的结果
     *
     * @param id
     * @param reason
     */
    public void updateRetryFailResult(long id, String reason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.remark, reason)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 获取原始记录ID
     *
     * @param id
     * @return
     */
    public Long getOriginalLogId(long id) {
        return super.getValue(QDO.originalId, id);
    }

    /**
     * 查询最后一次重试记录ID
     *
     * @param ids
     * @return
     */
    public Map<Long, Long> queryLastRetry(Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.lastRetryId)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.lastRetryId), (t1, t2) -> t1));
    }

    /**
     * 分页查询
     *
     * @param requestDTO
     * @return
     */
    public PagingVO<SysThirdApiLogDO> queryByPage(Long sysTenantId, ThirdApiLogQueryDTO requestDTO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.sysTenantId, sysTenantId)
                .andEq(QDO.appCode, requestDTO.getAppCode())
                .andEq(QDO.restful, requestDTO.getRestful())
                .andEq(QDO.server, requestDTO.getServer())
                .andEq(QDO.clientId, requestDTO.getClientId())
                .andEq(QDO.userId, requestDTO.getUserId())
                .andEq(QDO.username, requestDTO.getUsername())
                .andEq(QDO.uri, requestDTO.getUri())
                .andEq(QDO.reqSuccess, requestDTO.getReqSuccess())
                .andEq(QDO.respSuccess, requestDTO.getRespSuccess())
                .andBetween(QDO.reqTime, requestDTO.getReqTimeStart(), requestDTO.getReqTimeEnd())
                .andEq(QDO.retried, false)
                .build();

        return super.queryByPage(predicate, requestDTO.getPageRequest());
    }

    /**
     * 分页查询
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysThirdApiLogDO> queryByPage(ThirdApiLogQueryVO queryVO, Long sysTenantId, Boolean server, String thirdApp) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.server, server)
                .andEq(QDO.thirdApp, thirdApp)
                .andEq(QDO.businessType, queryVO.getBusinessType())
                .andEq(QDO.businessKey, queryVO.getBusinessKey())
                .andRightLike(QDO.uri, queryVO.getUri())
                .andEq(QDO.respSuccess, queryVO.getRespSuccess())
                .andBetween(QDO.reqTime, queryVO.getReqTimeStart(), queryVO.getReqTimeEnd())
                .andEq(QDO.retried, false)
                .andEq(QDO.sysTenantId, sysTenantId)
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.reqTime.desc());
    }

    /**
     * 获取重试列表
     *
     * @param originalId
     * @return
     */
    public List<SysThirdApiLogDO> listRetryRecord(long originalId) {
        return super.getListByValue(QDO.originalId, originalId);
    }
}
