package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.platform.model.entity.QSysPlatformUdcValueDO;
import com.elitescloud.cloudt.platform.model.entity.SysPlatformUdcValueDO;
import com.elitescloud.cloudt.system.model.bo.SysUdcValueBO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Repository
public class UdcValueRepoProc extends BaseRepoProc<SysPlatformUdcValueDO> {
    private static final QSysPlatformUdcValueDO QDO = QSysPlatformUdcValueDO.sysPlatformUdcValueDO;

    public UdcValueRepoProc() {
        super(QDO);
    }

    /**
     * 删除UDC
     *
     * @param appCode
     * @param udcCode
     */
    public void delete(@NotBlank String appCode, @NotBlank String udcCode) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode)))
                .execute();
    }

    /**
     * 修改启用状态
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(Long id, Boolean enabled) {
        super.updateValue(QDO.allowStart, enabled, id);
    }

    /**
     * 获取值列表
     *
     * @param appCode
     * @param udcCode
     * @param includeDisabled
     * @return
     */
    public List<SysPlatformUdcValueDO> listByUdc(String appCode, String udcCode, boolean includeDisabled) {
        Predicate predicate = QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode));
        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.allowStart.eq(true));
        }

        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.udcOrder.desc())
                .fetch();
    }

    /**
     * 获取值列表
     *
     * @param appCode
     * @param udcCode
     * @param includeDisabled
     * @return
     */
    public List<SysUdcValueBO> listBoByUdc(String appCode, String udcCode, boolean includeDisabled) {
        Predicate predicate = QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode));
        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.allowStart.eq(true));
        }

        return jpaQueryFactory.select(qBeanValueBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.udcOrder.asc())
                .fetch();
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCodes
     * @param includeDisabled
     * @return
     */
    public List<SysPlatformUdcValueDO> queryByUdc(String appCode, Set<String> udcCodes, boolean includeDisabled) {
        Predicate predicate = QDO.udcCode.in(udcCodes).and(QDO.appCode.eq(appCode));
        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.allowStart.eq(true));
        }

        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.udcOrder.asc())
                .fetch();
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCodes
     * @param includeDisabled
     * @return
     */
    public List<SysUdcValueBO> queryBoByUdc(String appCode, Collection<String> udcCodes, boolean includeDisabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(StringUtils.hasText(appCode), QDO.appCode, appCode)
                .andIn(CollUtil.isNotEmpty(udcCodes), QDO.udcCode, udcCodes)
                .andEq(!includeDisabled, QDO.allowStart, true)
                .build();

        return jpaQueryFactory.select(qBeanValueBO())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.udcOrder.asc())
                .fetch();
    }

    private QBean<SysUdcValueBO> qBeanValueBO() {
        return Projections.bean(SysUdcValueBO.class, QDO.id, QDO.appCode, QDO.udcCode, QDO.udcValueCode, QDO.udcValueName, QDO.udcOrder, QDO.allowStart,
                QDO.udcValueDescribe, QDO.allowDefault, QDO.parentUdcValueCode, QDO.valueAliasName);
    }
}
