package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserListQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserListRespVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserTypeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.StringExpression;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAQuery;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/22
 */
@Repository
public class UserRepoProc extends BaseRepoProc<SysUserDO> {
    private static final QSysUserDO QDO = QSysUserDO.sysUserDO;
    private static final QSysTenantUserDO QDO_TENANT_USER = QSysTenantUserDO.sysTenantUserDO;
    private static final QSysUserTypeDO QDO_USER_TYPE = QSysUserTypeDO.sysUserTypeDO;

    private final TenantClientProvider tenantClientProvider;

    public UserRepoProc(TenantClientProvider tenantClientProvider) {
        super(QDO);
        this.tenantClientProvider = tenantClientProvider;
    }

    /**
     * 更新启用禁用
     *
     * @param id      用户ID
     * @param enabled 是否启用
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 更新手机号
     *
     * @param id     用户ID
     * @param mobile 手机号
     */
    public void updateMobile(long id, String mobile) {
        super.updateValue(QDO.mobile, mobile, id);
    }

    /**
     * 更新邮箱
     *
     * @param id    用户ID
     * @param email 邮箱
     */
    public void updateEmail(long id, String email) {
        super.updateValue(QDO.email, email, id);
    }

    /**
     * 修改密码
     *
     * @param id        用户ID
     * @param needReset 是否需要重置密码
     * @param password  新密码，密文
     */
    public void updatePassword(long id, boolean needReset, @NotBlank String password) {
        jpaQueryFactory.update(QDO)
                .set(QDO.password, password)
                .set(QDO.needReset, needReset)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新CAS同步结果
     *
     * @param userName
     * @param casUserId
     */
    public void updateCasSyncResult(@NotBlank String userName, Long casUserId) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.casUserId, casUserId)
                .set(QDO.syncCas, false)
                .where(QDO.username.eq(userName))
                .execute();
    }

    /**
     * 更新用户头像
     *
     * @param id         用户ID
     * @param avatarUrl  头像url
     * @param avatarCode 头像文件code
     */
    public void updateAvatar(Long id, String avatarUrl, String avatarCode) {
        jpaQueryFactory.update(QDO)
                .set(QDO.avatarUrl, avatarUrl)
                .set(QDO.avatarCode, avatarCode)
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 清除CAS userId
     *
     * @param ids
     */
    public void clearCasUserId(Collection<Long> ids) {
        super.jpaQueryFactory.update(QDO)
                .setNull(QDO.casUserId)
                .set(QDO.syncCas, true)
                .where(QDO.id.in(ids))
                .execute();
    }

    /**
     * 获取是否启用
     *
     * @param id 用户ID
     * @return 是否启用
     */
    public Boolean getEnabled(long id) {
        var userEnabled = super.getValue(QDO.enabled, id);
        if (Boolean.FALSE.equals(userEnabled)) {
            return false;
        }
        if (this.supportTenant()) {
            // 获取租户下的状态
            var tenantUserEnabled = super.jpaQueryFactory.select(QDO_TENANT_USER.enabled)
                    .from(QDO_TENANT_USER)
                    .where(QDO_TENANT_USER.sysUserId.eq(id))
                    .limit(1)
                    .fetchOne();
            return Boolean.TRUE.equals(tenantUserEnabled);
        }
        return userEnabled;
    }

    /**
     * 获取是否启用
     *
     * @param tenantId 租户ID
     * @param ids      用户ID
     * @return 是否启用
     */
    public Map<Long, Boolean> getEnabled(Long tenantId, @NotEmpty Collection<Long> ids) {
        if (tenantId == null || tenantId.longValue() == TenantConstant.DEFAULT_TENANT_ID) {
            // 支持租户
            return super.jpaQueryFactory.select(QDO.id, QDO.enabled, QDO_TENANT_USER.enabled)
                    .from(QDO)
                    .leftJoin(QDO_TENANT_USER).on(QDO_TENANT_USER.sysUserId.eq(QDO.id))
                    .fetch()
                    .stream()
                    .collect(Collectors.toMap(t -> t.get(QDO.id),
                            t -> Boolean.TRUE.equals(t.get(QDO.enabled)) && Boolean.TRUE.equals(t.get(QDO_TENANT_USER.enabled)),
                            (t1, t2) -> t1));
        }

        // 不支持租户
        return super.jpaQueryFactory.select(QDO.id, QDO.enabled)
                .from(QDO)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id),
                        t -> Boolean.TRUE.equals(t.get(QDO.enabled)),
                        (t1, t2) -> t1));
    }

    /**
     * 获取用户账号
     *
     * @param id
     * @return
     */
    public String getUsername(long id) {
        return super.getValue(QDO.username, id);
    }

    /**
     * 获取密码
     *
     * @param id
     * @return
     */
    public String getPassword(long id) {
        return super.getValue(QDO.password, id);
    }

    /**
     * 获取统一认证中心的用户ID
     *
     * @param id
     * @return
     */
    public Long getCasUserId(long id) {
        return super.getValue(QDO.casUserId, id);
    }

    /**
     * 判断用户名是否存在
     *
     * @param username 用户名
     * @return 是否存在
     */
    public boolean existsUsername(@NotBlank String username) {
        return exists(QDO.username, username);
    }

    /**
     * 判断用户名是否存在
     *
     * @param username 用户名
     * @return 是否存在
     */
    public boolean existsUsername(@NotBlank String username, Long id) {
        return exists(QDO.username, username, id);
    }

    /**
     * 判断手机号是否存在
     *
     * @param mobile
     * @return
     */
    public boolean existsMobile(@NotBlank String mobile) {
        return exists(QDO.mobile, mobile);
    }

    /**
     * 判断手机号是否存在
     *
     * @param mobile
     * @return
     */
    public boolean existsMobile(@NotBlank String mobile, Long id) {
        return exists(QDO.mobile, mobile, id);
    }

    /**
     * 判断邮箱是否存在
     *
     * @param email 邮箱
     * @return 是否存在
     */
    public boolean existsEmail(@NotBlank String email) {
        return exists(QDO.email, email);
    }

    /**
     * 判断邮箱是否存在
     *
     * @param email 邮箱
     * @return 是否存在
     */
    public boolean existsEmail(@NotBlank String email, Long id) {
        return exists(QDO.email, email, id);
    }

    /**
     * 判断身份证号是否存在
     *
     * @param idCard 身份证号
     * @return 是否存在
     */
    public boolean existsIdCard(@NotBlank String idCard) {
        return exists(QDO.idCard, idCard);
    }

    /**
     * 判断身份证号是否存在
     *
     * @param idCard 身份证号
     * @return 是否存在
     */
    public boolean existsIdCard(@NotBlank String idCard, Long id) {
        return exists(QDO.idCard, idCard, id);
    }

    /**
     * 根据登录号获取用户ID
     *
     * @param username
     * @return
     */
    public Long getIdByUsername(@NotBlank String username) {
        return super.getIdByValue(QDO.username, username);
    }

    /**
     * 根据登录号获取用户ID
     *
     * @param usernames
     * @return
     */
    public List<Long> getIdByUsername(@NotEmpty Collection<String> usernames) {
        return super.getIdsByValue(QDO.username, usernames);
    }

    /**
     * 获取用户ID
     *
     * @param usernames 用户账号
     * @return 账号与ID
     */
    public Map<String, Long> getIdAndUsernameByUsername(@NotEmpty Collection<String> usernames) {
        return super.jpaQueryFactory.select(QDO.id, QDO.username)
                .from(QDO)
                .where(QDO.username.in(usernames))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.username), t -> t.get(QDO.id), (t1, t2) -> t1));
    }

    /**
     * 获取用户username
     *
     * @param ids 用户id
     * @return ID与账号
     */
    public Map<Long, String> getUsernameAndId(@NotEmpty Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.username)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.username), (t1, t2) -> t1));
    }

    /**
     * 根据手机号获取用户ID
     *
     * @param mobile
     * @return
     */
    public List<Long> getIdByMobile(@NotBlank String mobile) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.mobile.eq(mobile))
                .fetch();
    }

    /**
     * 根据邮箱获取用户ID
     *
     * @param email
     * @return
     */
    public List<Long> getIdByEmail(@NotBlank String email) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.email.eq(email))
                .fetch();
    }

    /**
     * 根据身份证号获取用户ID
     *
     * @param idCard
     * @return
     */
    public List<Long> getIdByIdCard(@NotBlank String idCard) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.idCard.eq(idCard))
                .fetch();
    }

    /**
     * 获取用户基本信息
     *
     * @param id
     * @return
     */
    public SysUserBasicDTO getBasicDto(long id) {
        var user = this.jpaQueryUserBasic()
                .where(QDO.id.eq(id))
                .limit(1)
                .fetchOne();
        return this.replaceTenantInfoOfBasic(user);
    }

    /**
     * 获取用户基本信息
     *
     * @param ids
     * @return
     */
    public List<SysUserBasicDTO> getBasicDto(Collection<Long> ids) {
        var userList = this.jpaQueryUserBasic()
                .where(QDO.id.in(ids))
                .fetch();
        return this.replaceTenantInfoOfBasic(userList);
    }

    /**
     * 查询用户列表
     *
     * @param queryDTO 查询参数
     * @return 用户列表
     */
    public List<SysUserBasicDTO> queryBasicDto(UserQueryDTO queryDTO) {
        Predicate predicate = buildPredicate(queryDTO);
        if (predicate == null) {
            return Collections.emptyList();
        }

        var userList = this.jpaQueryUserBasic()
                .where(predicate)
                .fetch();
        return this.replaceTenantInfoOfBasic(userList);
    }

    /**
     * 查询用户列表
     *
     * @param queryVO
     * @return
     */
    public List<UserListRespVO> queryUser(UserListQueryVO queryVO) {
        var predicateUser = PredicateBuilder.builder()
                .andLike(QDO.username, queryVO.getUsernameLike())
                .andLike(new StringExpression[]{QDO.lastName, QDO.firstName}, queryVO.getFullNameLike())
                .andLike(new StringExpression[]{QDO.username, QDO.lastName, QDO.firstName}, queryVO.getFullNameOrUsernameLike())
                .build();
//        if (predicateUser == null) {
//            return Collections.emptyList();
//        }

        var tenantId = this.filterTenant();
        var predicateTenantUser = PredicateBuilder.builder()
                .andEq(QDO.enabled, true)
                .and(supportTenant(), () -> JPAExpressions.select(QDO_TENANT_USER.id)
                        .from(QDO_TENANT_USER)
                        .where(QDO_TENANT_USER.sysUserId.eq(QDO.id).and(QDO_TENANT_USER.sysTenantId.eq(tenantId))
                                .and(QDO_TENANT_USER.enabled.eq(true))).exists())
                .build();
        var predicate = ExpressionUtils.and(predicateUser, predicateTenantUser);

        return jpaQueryFactory.select(qBeanUserListVO())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 查询需要向CAS同步的用户
     *
     * @param size
     * @return
     */
    public List<SysUserDO> queryToSyncCas(int size) {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(QDO.syncCas.eq(true).or(QDO.syncCas.isNull()))
                .limit(size)
                .fetch();
    }

    /**
     * 查询禁用的用户ID
     *
     * @return
     */
    public List<Long> queryIdsOfDisabled() {
        var tenantId = this.filterTenant();
        if (tenantId == TenantConstant.DEFAULT_TENANT_ID) {
            // 默认租户
            return jpaQueryFactory.select(QDO.id)
                    .from(QDO)
                    .where(QDO.enabled.eq(false))
                    .fetch();
        }

        var tenantUserPredicate = JPAExpressions.select(QDO_TENANT_USER.id)
                .from(QDO_TENANT_USER)
                .where(QDO_TENANT_USER.sysUserId.eq(QDO.id)
                        .and(QDO_TENANT_USER.sysTenantId.eq(tenantId))
                        .and(QDO_TENANT_USER.enabled.eq(false))
                ).exists();
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.enabled.eq(false).or(tenantUserPredicate))
                .fetch();
    }

    /**
     * 分页查询管理用户
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysUserDO> pageMng(@NotNull GeneralUserDetails currentUser, @NotNull UserPageQueryVO queryVO) {
        var tenantId = filterTenant();
        // 当前是租户查询
        var isTenant = tenantId != TenantConstant.DEFAULT_TENANT_ID;

        var jpaQuery = jpaQueryFactory.select(QDO)
                .from(QDO);
        if (isTenant) {
            // 使用租户过滤
            jpaQuery.leftJoin(QDO_TENANT_USER)
                    .on(QDO_TENANT_USER.sysUserId.eq(QDO.id).and(QDO_TENANT_USER.sysTenantId.eq(tenantId)));
        }

        // 查询条件
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.username, queryVO.getUsername())
                .andLike(QDO.mobile, queryVO.getMobile())
                .andLike(QDO.email, queryVO.getEmail())
                .andLike(QDO.lastName, queryVO.getFullName())
                // 当前是租户，则根据租户的用户状态判断
                .andEq(isTenant && queryVO.getEnabled() != null, QDO_TENANT_USER.enabled, queryVO.getEnabled())
                // 当前不是租户，则根据用户的状态判断
                .andEq(!isTenant && queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .andEq(QDO.sourceType, queryVO.getSourceType())
                .andEq(isTenant, QDO_TENANT_USER.sysTenantId, tenantId)
                .andNe(true, QDO.id, currentUser.getUserId())
                .andEq(QDO.gender, queryVO.getGender())
                .andBetween(QDO.createTime, queryVO.getCreateTimeStart(), queryVO.getCreateTimeEnd())
                .build();

        jpaQuery.where(predicate);
        var pageData = queryByPage(jpaQuery, queryVO.getPageRequest());
        this.replaceTenantInfoOfDo(pageData.getRecords());
        return pageData;
    }

    /**
     * 分页查询用户
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysUserDO> pageQuery(@NotNull GeneralUserDetails currentUser, @NotNull UserPageQueryVO queryVO) {
        Long tenantId;
        if (!ObjectUtil.defaultIfNull(queryVO.getAllTenant(), false)) {
            // 不是查询所有，则默认只查所在租户的
            tenantId = tenantClientProvider.enabledTenant() ? ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID) : null;
        } else {
            tenantId = null;
        }

        var jpaQuery = jpaQueryFactory.select(QDO)
                .from(QDO);
        if (tenantId != null) {
            // 如果限制了租户，则使用租户过滤
            jpaQuery.leftJoin(QDO_TENANT_USER)
                    .on(QDO_TENANT_USER.sysUserId.eq(QDO.id).and(QDO_TENANT_USER.sysTenantId.eq(tenantId)));
        }

        // 查询条件
        var predicate = PredicateBuilder.builder()
                .andRightLike(StringUtils.hasText(queryVO.getUsername()), QDO.username, queryVO.getUsername())
                .andRightLike(StringUtils.hasText(queryVO.getMobile()), QDO.mobile, queryVO.getMobile())
                .andRightLike(tenantId == null && StringUtils.hasText(queryVO.getEmail()), QDO.email, queryVO.getEmail())
                .andLike(StringUtils.hasText(queryVO.getFullName()), QDO.lastName, queryVO.getFullName())
                .andEq(tenantId == null && queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .and(tenantId != null && queryVO.getEnabled() != null, () -> QDO.enabled.eq(queryVO.getEnabled()).or(QDO_TENANT_USER.enabled.eq(queryVO.getEnabled())))
                .andEq(StringUtils.hasText(queryVO.getSourceType()), QDO.sourceType, queryVO.getSourceType())
                .andEq(tenantId != null, QDO_TENANT_USER.sysTenantId, tenantId)
                .and(StringUtils.hasText(queryVO.getUserType()), () -> JPAExpressions.select(QDO_USER_TYPE.id).from(QDO_USER_TYPE).where(
                        QDO_USER_TYPE.userId.eq(QDO.id).and(QDO_USER_TYPE.type.eq(queryVO.getUserType())).and(tenantId == null ? null : QDO_USER_TYPE.sysTenantId.eq(tenantId))
                ).exists())
                .and(CollUtil.isNotEmpty(queryVO.getUserTypeList()), () -> JPAExpressions.select(QDO_USER_TYPE.id).from(QDO_USER_TYPE).where(
                        QDO_USER_TYPE.userId.eq(QDO.id).and(QDO_USER_TYPE.type.in(queryVO.getUserTypeList())).and(tenantId == null ? null : QDO_USER_TYPE.sysTenantId.eq(tenantId))
                ).exists())
                .andNe(!ObjectUtil.defaultIfNull(queryVO.getIncludeSelf(), false), QDO.id, currentUser.getUserId())
                .andLike(new StringExpression[]{QDO.username, QDO.lastName, QDO.firstName}, queryVO.getKeyword())
                .build();

        jpaQuery.where(predicate);
        var pageData = queryByPage(jpaQuery, queryVO.getPageRequest());
        this.replaceTenantInfoOfDo(pageData.getRecords());
        return pageData;
    }

    /**
     * 查询用户姓名
     *
     * @param ids
     * @return
     */
    public List<IdNameParam> queryUserName(Collection<Long> ids) {
        return jpaQueryFactory.select(QDO.id, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .map(t -> new IdNameParam(t.get(QDO.id), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username))))
                .collect(Collectors.toList());
    }

    private String buildUserName(String firstName, String lastName, String username) {
        if (CharSequenceUtil.isBlank(lastName)) {
            return CharSequenceUtil.blankToDefault(firstName, username);
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }

    private Predicate buildPredicate(UserQueryDTO queryDTO) {
        List<Predicate> predicates = new ArrayList<>();
        if (!CollectionUtils.isEmpty(queryDTO.getUserIds())) {
            predicates.add(QDO.id.in(queryDTO.getUserIds()));
        }
        if (!CollectionUtils.isEmpty(queryDTO.getUsernames())) {
            predicates.add(QDO.username.in(queryDTO.getUsernames()));
        }
        if (!CollectionUtils.isEmpty(queryDTO.getUsernameLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getUsernameLike()) {
                if (temp == null) {
                    temp = QDO.username.like("%" + s + "%");
                    continue;
                }
                temp = QDO.username.like("%" + s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getUsernameRightLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getUsernameRightLike()) {
                if (temp == null) {
                    temp = QDO.username.like(s + "%");
                    continue;
                }
                temp = QDO.username.like(s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getMobiles())) {
            predicates.add(QDO.mobile.in(queryDTO.getMobiles()));
        }
        if (!CollectionUtils.isEmpty(queryDTO.getMobileLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getMobileLike()) {
                if (temp == null) {
                    temp = QDO.mobile.like("%" + s + "%");
                    continue;
                }
                temp = QDO.mobile.like("%" + s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getMobileRightLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getMobileRightLike()) {
                if (temp == null) {
                    temp = QDO.mobile.like(s + "%");
                    continue;
                }
                temp = QDO.mobile.like(s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getEmails())) {
            predicates.add(QDO.email.in(queryDTO.getEmails()));
        }
        if (!CollectionUtils.isEmpty(queryDTO.getEmailLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getEmailLike()) {
                if (temp == null) {
                    temp = QDO.email.like("%" + s + "%");
                    continue;
                }
                temp = QDO.email.like("%" + s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getEmailRightLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getEmailRightLike()) {
                if (temp == null) {
                    temp = QDO.email.like(s + "%");
                    continue;
                }
                temp = QDO.email.like(s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getFullNames())) {
            predicates.add(QDO.lastName.in(queryDTO.getFullNames()));
        }
        if (!CollectionUtils.isEmpty(queryDTO.getFullNameLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getFullNameLike()) {
                if (temp == null) {
                    temp = QDO.lastName.like("%" + s + "%");
                    continue;
                }
                temp = QDO.lastName.like("%" + s + "%").or(temp);
            }
            predicates.add(temp);
        }
        if (!CollectionUtils.isEmpty(queryDTO.getFullNameRightLike())) {
            Predicate temp = null;
            for (String s : queryDTO.getFullNameRightLike()) {
                if (temp == null) {
                    temp = QDO.lastName.like(s + "%");
                    continue;
                }
                temp = QDO.lastName.like(s + "%").or(temp);
            }
            predicates.add(temp);
        }

        if (queryDTO.getEnabled() != null) {
            if (this.supportTenant()) {
                if (queryDTO.getEnabled()) {
                    predicates.add(QDO.enabled.eq(true).and(QDO_TENANT_USER.enabled.eq(true)));
                } else {
                    predicates.add(QDO.enabled.eq(false).or(QDO_TENANT_USER.enabled.eq(false)));
                }
            } else {
                predicates.add(QDO.enabled.eq(queryDTO.getEnabled()));
            }
        }
        if (CollUtil.isNotEmpty(queryDTO.getUserTypes())) {
            var subQuery = JPAExpressions.select(QDO_USER_TYPE.userId).from(QDO_USER_TYPE).where(QDO_USER_TYPE.type.in(queryDTO.getUserTypes()));
            predicates.add(QDO.id.in(subQuery));
        }

        return ExpressionUtils.allOf(predicates);
    }

    private List<SysUserDO> replaceTenantInfoOfDo(List<SysUserDO> userList) {
        long tenantId = this.filterTenant();
        if (!userList.isEmpty() && TenantConstant.DEFAULT_TENANT_ID != tenantId) {
            var userIds = userList.stream().map(SysUserDO::getId).collect(Collectors.toSet());
            var tenantUserMap = jpaQueryFactory.select(QDO_TENANT_USER.sysUserId, QDO_TENANT_USER.enabled).from(QDO_TENANT_USER)
                    .where(QDO_TENANT_USER.sysUserId.in(userIds).and(QDO_TENANT_USER.sysTenantId.eq(tenantId)))
                    .fetch()
                    .stream()
                    .collect(Collectors.toMap(t -> t.get(QDO_TENANT_USER.sysUserId), t -> ObjectUtil.defaultIfNull(t.get(QDO_TENANT_USER.enabled), false), (t1, t2) -> t1));
            for (SysUserDO sysUserDO : userList) {
                if (Boolean.TRUE.equals(sysUserDO.getEnabled())) {
                    sysUserDO.setEnabled(tenantUserMap.getOrDefault(sysUserDO.getId(), true));
                }
            }
        }
        return userList;
    }

    private List<SysUserBasicDTO> replaceTenantInfoOfBasic(List<SysUserBasicDTO> userList) {
        long tenantId = this.filterTenant();
        if (!userList.isEmpty() && TenantConstant.DEFAULT_TENANT_ID != tenantId) {
            var userIds = userList.stream().map(SysUserBasicDTO::getId).collect(Collectors.toSet());
            var tenantUserMap = jpaQueryFactory.select(QDO_TENANT_USER.sysUserId, QDO_TENANT_USER.enabled).from(QDO_TENANT_USER)
                    .where(QDO_TENANT_USER.sysUserId.in(userIds).and(QDO_TENANT_USER.sysTenantId.eq(tenantId)))
                    .fetch()
                    .stream()
                    .collect(Collectors.toMap(t -> t.get(QDO_TENANT_USER.sysUserId), t -> ObjectUtil.defaultIfNull(t.get(QDO_TENANT_USER.enabled), false), (t1, t2) -> t1));
            for (SysUserBasicDTO userDto : userList) {
                userDto.setEnabled(tenantUserMap.getOrDefault(userDto.getId(), false));
            }
        }
        return userList;
    }

    private List<Long> filterUserIdByTenant(List<Long> ids, Boolean enabled) {
        long tenantId = this.filterTenant();
        if (CollUtil.isEmpty(ids) || TenantConstant.DEFAULT_TENANT_ID == tenantId) {
            return ids;
        }

        var predicate = PredicateBuilder.builder()
                .andIn(QDO_TENANT_USER.sysUserId, ids)
                .andEq(QDO_TENANT_USER.enabled, enabled)
                .build();
        return super.jpaQueryFactory.select(QDO_TENANT_USER.sysUserId)
                .from(QDO_TENANT_USER)
                .where(predicate)
                .fetch();
    }

    private SysUserBasicDTO replaceTenantInfoOfBasic(SysUserBasicDTO user) {
        long tenantId = this.filterTenant();
        if (user != null && TenantConstant.DEFAULT_TENANT_ID != tenantId) {
            var enabled = jpaQueryFactory.select(QDO_TENANT_USER.enabled).from(QDO_TENANT_USER)
                    .where(QDO_TENANT_USER.sysUserId.eq(user.getId()).and(QDO_TENANT_USER.sysTenantId.eq(tenantId)))
                    .limit(1)
                    .fetchOne();
            user.setEnabled(ObjectUtil.defaultIfNull(enabled, false));
        }
        return user;
    }

    /**
     * 过滤的租户
     * <p>
     * 如果没有启用租户功能，或当前用户是运营机构用户，则无需租户过滤，即可查询所有用户，否则只能查询自己租户下的用户
     *
     * @return 是否
     */
    private long filterTenant() {
        var tenant = tenantClientProvider.getCurrentTenant();
        return tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();
    }

    private boolean supportTenant() {
        var tenant = tenantClientProvider.getCurrentTenant();
        return tenant != null;
    }

    private QBean<SysUserBasicDTO> qBeanUserBasic() {
        return Projections.bean(SysUserBasicDTO.class, QDO.id, QDO.username, QDO.lastName, QDO.firstName, QDO.gender, QDO.nickName,
                QDO.mobile, QDO.email, QDO.enabled, QDO.idCard, QDO.avatarUrl, QDO.avatarCode);
    }

    private QBean<UserListRespVO> qBeanUserListVO() {
        return Projections.bean(UserListRespVO.class, QDO.id, QDO.username, QDO.lastName, QDO.firstName, QDO.nickName, QDO.mobile, QDO.email);
    }

    private JPAQuery<SysUserBasicDTO> jpaQueryUserBasic() {
        boolean support = this.supportTenant();
        var jpaQuery = jpaQueryFactory.select(this.qBeanUserBasic())
                .from(QDO);
        if (support) {
            return jpaQuery.leftJoin(QDO_TENANT_USER).on(QDO_TENANT_USER.sysUserId.eq(QDO.id));
        }
        return jpaQuery;
    }

}
