package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.bo.SysUserTypeBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserTypeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserTypeDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.Expressions;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/26
 */
@Repository
public class UserTypeRepoProc extends BaseRepoProc<SysUserTypeDO> {
    private static final QSysUserTypeDO QDO = QSysUserTypeDO.sysUserTypeDO;

    public UserTypeRepoProc() {
        super(QDO);
    }

    /**
     * 更新身份ID
     *
     * @param userId
     * @param tenantId
     * @param userType
     * @param identityId
     */
    public void updateIdentityId(long userId, long tenantId, @NotBlank String userType, String identityId) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.identityId, identityId)
                .where(QDO.userId.eq(userId).and(QDO.type.eq(userType)).and(QDO.sysTenantId.eq(tenantId))
                        .and(QDO.identityId.isNull().or(QDO.identityId.isEmpty())))
                .execute();
    }

    /**
     * 删除身份标识
     *
     * @param userId
     * @param tenantId
     * @param userType
     * @param identityId
     * @return
     */
    public void deleteIdentityId(long userId, long tenantId, @NotBlank String userType, String identityId) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.userId, userId)
                .andEq(QDO.sysTenantId, tenantId)
                .andEq(QDO.type, userType)
                .and(true, () -> {
                    if (CharSequenceUtil.isBlank(identityId)) {
                        return QDO.identityId.isNull().or(QDO.identityId.isEmpty());
                    }
                    return QDO.identityId.eq(identityId);
                })
                .build();

        super.delete(predicate);
    }

    /**
     * 根据用户ID删除
     *
     * @param userId
     */
    public void deleteByUserId(Long userId, long tenantId) {
        super.delete(QDO.userId.eq(userId).and(QDO.sysTenantId.eq(tenantId)));
    }

    /**
     * 删除
     *
     * @param userId
     * @param types
     */
    public void delete(Long userId, Collection<String> types, long tenantId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.userId.eq(userId).and(QDO.type.in(types)).and(QDO.sysTenantId.eq(tenantId)))
                .execute();
    }

    /**
     * 获取用户的类型
     *
     * @param userId
     * @return
     */
    public Set<String> getUserType(Long userId, long tenantId) {
        return new HashSet<>(jpaQueryFactory.select(QDO.type)
                .from(QDO)
                .where(QDO.userId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .fetch());
    }

    /**
     * 获取用户的账号类型
     *
     * @param userId
     * @param tenantId
     * @return
     */
    public List<SysUserTypeDO> getUserTypeList(long userId, long tenantId) {
        return super.getList(QDO.userId.eq(userId).and(QDO.sysTenantId.eq(tenantId)));
    }

    /**
     * 获取账号类型
     *
     * @param userId
     * @param tenantId
     * @return
     */
    public List<SysUserTypeBO> getUserTypeBO(Long userId, long tenantId) {
        return jpaQueryFactory.select(qBeanUserType())
               .from(QDO)
               .where(QDO.userId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
               .fetch();
    }

    /**
     * 获取用户的类型
     *
     * @param userId
     * @return
     */
    public Map<Long, Set<String>> getUserTypes(Collection<Long> userId, long tenantId) {
        return jpaQueryFactory.select(QDO.userId, QDO.type)
                .from(QDO)
                .where(QDO.userId.in(userId).and(QDO.sysTenantId.eq(tenantId)))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.userId),
                        Collectors.collectingAndThen(Collectors.toList(),
                                t -> t.stream().map(tt -> tt.get(QDO.type)).collect(Collectors.toSet()))));
    }

    private QBean<SysUserTypeBO> qBeanUserType() {
        return Projections.bean(SysUserTypeBO.class, QDO.type, QDO.identityId);
    }
}
